/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.auth.cstore.xml;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.ext.feature.auth.cstore.xml.util.Constants;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.HostInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.Credential;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStore;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStoreFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStoreFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStoreFeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.Password;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class CredentialStoreFeatureImpl implements CredentialStoreFeature.Provider {

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(CredentialStoreFeature.class);

    public static final String IMPL = CredentialStoreFeatureImpl.class.getName();

    private DeploymentContext deploymentContext;
    private CredentialStoreImpl credentialStore;
    private CredentialStoreFeatureFactory factory;
    private Element configuration;

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getExtensionImpl()
     */
    public String spi_getExtensionImpl() {
        return IMPL;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws FeatureException {

        log.debug("init begin");

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.auth.cstore.xml.config.1.0.xml";
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
            } catch (Exception ex) {
                throw new FeatureException("Error retrieving default configuration. " + ex);
            }
        }

        validateConfiguration(new DOMSource(configuration));

        Map<HostInfo, Map<String, Map<String, Credential>>> hosts;
        hosts = new HashMap<HostInfo, Map<String, Map<String, Credential>>>();

        QName childName = new QName(Constants.FEATURE_SCHEMA_NS, "cstore");
        Element cstoreElement = DOMUtils.getChildElement(configuration, childName);
        String id = cstoreElement.getAttribute("id");

        if (id.equals("")) {
            id = CredentialStoreFeature.DEFAULT_CSTORE_ID;
        }

        childName = new QName(Constants.FEATURE_SCHEMA_NS, "hosts");
        Element hostsElement = DOMUtils.getChildElement(cstoreElement, childName);
        childName = new QName(Constants.FEATURE_SCHEMA_NS, "host");
        List<Element> hostElements = DOMUtils.getChildElements(hostsElement, childName);
        for (Element hostElement : hostElements) {
            String host = hostElement.getAttribute("name");
            int port = -1;
            if (!hostElement.getAttribute("port").equals("")) {
                port = Integer.parseInt(hostElement.getAttribute("port"));
            }
            if (host.equals(HostInfo.MY_HOST)) {
                if (port != -1) {
                    throw new CredentialStoreFeatureException("Port specified for host '"
                            + host
                            + "' must be '-1' which implies all ports.");
                }
            }
            HostInfo hostInfo = new HostInfo(host, port);
            if (hosts.containsKey(hostInfo)) {
                throw new CredentialStoreFeatureException("Host '"
                        + host
                        + "' and port '"
                        + port
                        + "' not unique within context of cstore.");
            }
            Map<String, Map<String, Credential>> realms = new HashMap<String, Map<String, Credential>>();
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "realms");
            Element realmsElement = DOMUtils.getChildElement(hostElement, childName);
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "realm");
            List<Element> realmElements = DOMUtils.getChildElements(realmsElement, childName);
            for (Element realmElement : realmElements) {
                String realm = realmElement.getAttribute("name");
                if (realms.containsKey(realm)) {
                    throw new CredentialStoreFeatureException("realm '"
                            + realm
                            + "' not unique within context of host '"
                            + host
                            + "' and port '"
                            + port
                            + "'.");
                }
                Map<String, Credential> users = new HashMap<String, Credential>();
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "users");
                Element usersElement = DOMUtils.getChildElement(realmElement, childName);
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "user");
                List<Element> userElements = DOMUtils.getChildElements(usersElement, childName);
                for (Element userElement : userElements) {
                    childName = new QName(Constants.FEATURE_SCHEMA_NS, "name");
                    Element nameElement = DOMUtils.getChildElement(userElement, childName);
                    String userName = DOMUtils.getText(nameElement);
                    if (users.containsKey(userName)) {
                        throw new CredentialStoreFeatureException("user '"
                                + userName
                                + "' not unique within context of realm '"
                                + realm
                                + "' defined within context of host '"
                                + host
                                + ".");
                    }
                    Password password = null;
                    childName = new QName(Constants.FEATURE_SCHEMA_NS, "password");
                    Element passwordElement = DOMUtils.getChildElement(userElement, childName);
                    if (passwordElement != null) {
                        password = getPassword(DOMUtils.getText(passwordElement));
                    }
                    Credential credential = new Credential(userName, password);
                    users.put(userName, credential);
                }
                realms.put(realm, users);
            }
            hosts.put(hostInfo, realms);
        }

        credentialStore = new CredentialStoreImpl(id, hosts);

        log.debug("init end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        credentialStore = null;
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        factory = (CredentialStoreFeatureFactory)((CredentialStoreFeature)consumer).getExtensionFactory();
        deploymentContext = factory.getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStoreFeature$Provider#spi_getCredentialStore(java.lang.String)
     */
    public CredentialStore spi_getCredentialStore(String id) {
        if (credentialStore == null || id == null) {
            return null;
        }
        if (credentialStore.getID().equals(id)) {
            return credentialStore;
        } else {
            return null;
        }
    }

    private void validateConfiguration(DOMSource domSource) throws FeatureException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource(Constants.FEATURE_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new FeatureException("Error validating configuration. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new FeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new FeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new FeatureException("Configuration invalid. " + validationError);
        }

    }

    public Password getPassword(String credential) {
        org.mortbay.jetty.security.Credential temp = org.mortbay.jetty.security.Credential
                .getCredential(credential);
        if (temp instanceof org.mortbay.jetty.security.Password) {
            return new ClearTextPasswordImpl(factory, temp.toString());
        } else if (temp instanceof org.mortbay.jetty.security.Credential.MD5) {
            return new MessageDigestPasswordImpl(factory, (org.mortbay.jetty.security.Credential.MD5)temp);
        } else {
            throw new IllegalArgumentException("Unsupported credential type " + temp.getClass().getName());
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_setConfiguration(org.w3c.dom.Element)
     */
    public void spi_setConfiguration(Element configuration) {
        this.configuration = configuration;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

}
