/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.connector.db.derby.client;

import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector;
import org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector.Dialect;
import org.bluestemsoftware.specification.eoa.ext.Extension;

public class DataSourceConnectorImpl implements DataSourceConnector.Provider {

    private DataSource dataSource;
    private Method unwrapMethod;
    private Method isWrapperForMethod;

    public static String IMPL = DataSourceConnectorImpl.class.getName();

    public DataSourceConnectorImpl(DataSource dataSource) {
        super();
        this.dataSource = dataSource;
        try {
            Class<?> clazz = DataSource.class;
            Class<?>[] signature = { Class.class };
            unwrapMethod = clazz.getDeclaredMethod("unwrap", signature);
        } catch (NoSuchMethodException ignore) {
        }
        try {
            Class<?> clazz = DataSource.class;
            Class<?>[] signature = { Class.class };
            isWrapperForMethod = clazz.getDeclaredMethod("isWrapperFor", signature);
        } catch (NoSuchMethodException ignore) {
        }
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getConnection()
     */
    public Connection getConnection() throws SQLException {
        return dataSource.getConnection();
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getConnection(java.lang.String, java.lang.String)
     */
    public Connection getConnection(String username, String password) throws SQLException {
        return dataSource.getConnection(username, password);
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getLogWriter()
     */
    public PrintWriter getLogWriter() throws SQLException {
        return dataSource.getLogWriter();
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getLoginTimeout()
     */
    public int getLoginTimeout() throws SQLException {
        return dataSource.getLoginTimeout();
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#setLogWriter(java.io.PrintWriter)
     */
    public void setLogWriter(PrintWriter out) throws SQLException {
        dataSource.setLogWriter(out);
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#setLoginTimeout(int)
     */
    public void setLoginTimeout(int seconds) throws SQLException {
        dataSource.getLoginTimeout();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    /*
     * (non-Javadoc)
     * 
     * JDK 1.6 DataSource method. Implemented using reflection API which allows
     * specification-eoa-ext to be compiled using JDK 1.5 in cases where underlying
     * dataSource(s) unavailable for JDK 1.6.
     * 
     * @see java.sql.Wrapper#isWrapperFor(java.lang.Class)
     */
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        if (isWrapperForMethod != null) {
            Boolean answer;
            try {
                answer = (Boolean)isWrapperForMethod.invoke(dataSource, new Object[] { iface });
            } catch (Exception ex) {
                throw new SQLException(ex.toString());
            }
            return answer.booleanValue();
        } else {
            return false;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * JDK 1.6 DataSource method. Implemented using reflection API which allows
     * specification-eoa-ext to be compiled using JDK 1.5 in cases where underlying
     * dataSource(s) unavailable for JDK 1.6.
     * 
     * @see java.sql.Wrapper#unwrap(java.lang.Class)
     */
    public <T> T unwrap(Class<T> iface) throws SQLException {
        if (unwrapMethod != null) {
            try {
                return iface.cast(unwrapMethod.invoke(dataSource, new Object[] { iface }));
            } catch (Exception ex) {
                throw new SQLException(ex.toString());
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector$Provider#spi_getDialect()
     */
    public Dialect spi_getDialect() {
        return Dialect.DERBY;
    }

}
