/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.db.server.derby;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.io.StringWriter;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;

import javax.sql.DataSource;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.derby.drda.NetworkServerControl;
import org.apache.derby.jdbc.EmbeddedDataSource;
import org.apache.derby.jdbc.EmbeddedXADataSource;
import org.apache.xbean.spring.context.impl.XBeanXmlBeanFactory;
import org.bluestemsoftware.open.eoa.ext.feature.db.server.derby.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.feature.db.server.derby.xbean.DerbyConfig;
import org.bluestemsoftware.open.eoa.ext.feature.db.server.derby.xbean.Property;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.db.server.DBServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.db.server.DBServerFeatureException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.core.io.InputStreamResource;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class DBServerFeatureImpl implements DBServerFeature.Provider {

    private static final Log log = SystemContext.getContext().getSystem().getLog(DBServerFeature.class);
    
    private DeploymentContext deploymentContext;
    private Extension consumer;
    private Element configuration;

    public static final String IMPL = DBServerFeatureImpl.class.getName();

    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = consumer;
        this.deploymentContext = consumer.getExtensionFactory().getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_setConfiguration(org.w3c.dom.Element)
     */
    public void spi_setConfiguration(Element configuration) {
        this.configuration = configuration;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws FeatureException {

        log.debug("init begin");
        
        // if configuration is null, retrieve default configuration contained within
        // deployment

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.db.server.derby.config.1.0.xml";
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
            } catch (Exception ex) {
                throw new FeatureException("Error retrieving default configuration. " + ex);
            }
        }

        ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();

        String databaseName = null;
        
        try {

            // unfortunately spring has no dom input resource, so we must convert
            // to string and then into a byte inputstream

            ByteArrayInputStream bais = null;
            try {
                StringWriter stringWriter = new StringWriter();
                DOMSerializer.serializeNode(configuration, stringWriter, "UTF-8", true);
                bais = new ByteArrayInputStream(stringWriter.toString().getBytes("UTF-8"));
            } catch (Exception ex) {
                throw new DBServerFeatureException("Error parsing bean definitions. ", ex);
            }

            XBeanXmlBeanFactory beanFactory = null;
            Thread.currentThread().setContextClassLoader(deploymentContext.getClassLoader());
            try {
                beanFactory = new XBeanXmlBeanFactory(new InputStreamResource(bais));
            } catch (Exception ex) {
                throw new DBServerFeatureException(ex.toString(), ex);
            }

            PropertyPlaceholderConfigurer pphc = new PropertyPlaceholderConfigurer();
            pphc.postProcessBeanFactory(beanFactory);

            try {
                DerbyConfig config = null;
                String[] names = beanFactory.getBeanNamesForType(DerbyConfig.class);
                for (int i = 0; i < names.length; i++) {
                    String name = names[i];
                    config = (DerbyConfig)beanFactory.getBean(name);
                    if (config != null) {
                        break;
                    }
                }
                if (config == null) {
                    throw new DBServerFeatureException("No 'derby' xbean found within supplied configuration.");
                } else {
                    databaseName = config.getDatabaseName();
                    if (config.getProperties() != null) {
                        Properties sysProps = System.getProperties();
                        for (Property property : config.getProperties()) {
                            if (property.getName().equals("derby.system.home")) {
                                log.warn("'derby.system.home' cannot be overridden");
                                continue;
                            }
                            String name = property.getName();
                            String value = property.getValue();
                            log.debug("setting property '" + name + "' with value '" + value + "'");
                            sysProps.put(name, value);
                        }
                    }
                }
            } catch (BeansException be) {
                throw new DBServerFeatureException("Error parsing bean definitions. " + be);
            }

        } finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }

        // set up system home property, which derby will use to locate error log and
        // which will be used to locate database file

        try {
            Properties sysProps = System.getProperties();
            sysProps.put("derby.system.home", ((Feature)consumer).getFeatureVarDir().getAbsolutePath());
            log.debug("derby.system.home " + sysProps.getProperty("derby.system.home"));
        } catch (Exception ex) {
            throw new DBServerFeatureException("Error setting derby system home. " + ex);
        }

        // derby manual indicates that the embedded driver class must be unloaded
        // if already loaded to restart derby in same jvm. so ... just in case we're
        // restarting

        System.gc();

        try {
            String dbURL = "jdbc:derby:" + databaseName + ";create=true";
            ClassLoader cl = deploymentContext.getClassLoader();
            Class.forName("org.apache.derby.jdbc.EmbeddedDriver", true, cl);
            DriverManager.getConnection(dbURL);
        } catch (Throwable th) {
            StringBuffer sb = new StringBuffer();
            try {
                sb.append("Error starting dbms. ");
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                th.printStackTrace(new PrintStream(baos, true, "UTF-8"));
                sb.append(System.getProperty("line.separator"));
                sb.append(new String(baos.toByteArray(), "UTF-8"));
            } catch (Exception exception) {
                sb.append(th.toString());
            }
            throw new DBServerFeatureException(sb.toString());
        }

        log.debug("init end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");

        // if property 'derby.drda.startNetworkServer=true' was
        // set by user, then we need to shut it down
        try {
            NetworkServerControl server = new NetworkServerControl();
            server.shutdown();
        } catch (Exception ex) {
            // property was not set, i.e. server wasn't started
        }
        try {
            DriverManager.getConnection("jdbc:derby:;shutdown=true");
        } catch (SQLException se) {
            // a sql exception is thrown by default on shutdown
        }
        
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.db.server.DBServerFeature$Provider#spi_getEmbeddedDataSource(boolean)
     */
    public DataSource spi_getEmbeddedDataSource(boolean isXA) {
        DataSource embeddedDataSource;
        if (isXA) {
            embeddedDataSource = new EmbeddedXADataSource();
        } else {
            embeddedDataSource = new EmbeddedDataSource();
        }
        return embeddedDataSource;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

}
