/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.connector.db.derby.embedded;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.commons.beanutils.PropertyUtils;
import org.bluestemsoftware.specification.eoa.ext.feature.db.server.DBServerFeature;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.tranql.connector.NoExceptionsAreFatalSorter;
import org.tranql.connector.jdbc.AbstractLocalDataSourceMCF;

/**
 * Given that an embedded datasource requires a reference to in-memory representation of
 * database, i.e. org.apache.derby.iapi.db.Database, which is scoped to derby-db-server feature
 * deployment, we must defer creation of EmbeddedDataSource to a point at which EOA Server is
 * started and derby-db-server feature is available, i.e. when a physical connection is
 * requested.
 */
public final class ManagedConnectionFactoryImpl extends AbstractLocalDataSourceMCF {

    private static final long serialVersionUID = 1L;

    public ManagedConnectionFactoryImpl(boolean isXA) {
        super(new LazyEmbeddedDataSource(isXA), new NoExceptionsAreFatalSorter(), false);
    }

    /** *********************** Derby specific DataSource properties *********************** * */

    public String getUserName() {
        return ((LazyEmbeddedDataSource)dataSource).getUser();
    }

    public void setUserName(String user) {
        ((LazyEmbeddedDataSource)dataSource).setUser(user);
    }

    public String getPassword() {
        return ((LazyEmbeddedDataSource)dataSource).getPassword();
    }

    public void setPassword(String password) {
        ((LazyEmbeddedDataSource)dataSource).setPassword(password);
    }

    public String getDatabaseName() {
        return ((LazyEmbeddedDataSource)dataSource).getDatabaseName();
    }

    public void setDatabaseName(String name) {
        ((LazyEmbeddedDataSource)dataSource).setDatabaseName(name);
    }

    public Boolean getCreateDatabase() {
        return Boolean.valueOf("create".equals(((LazyEmbeddedDataSource)dataSource).getCreateDatabase()));
    }

    public void setCreateDatabase(Boolean create) {
        ((LazyEmbeddedDataSource)dataSource).setCreateDatabase(create.booleanValue() ? "create" : null);
    }

    public void setShutdownDatabase(String shutDown) {
        ((LazyEmbeddedDataSource)dataSource).setShutdownDatabase(shutDown);
    }

    public String getShutdownDatabase() {
        return ((LazyEmbeddedDataSource)dataSource).getShutdownDatabase();
    }

    static class LazyEmbeddedDataSource implements DataSource {

        private boolean isXA;
        private String user;
        private String password;
        private String databaseName;
        private String createDatabase;
        private String shutdownDatabase;
        private PrintWriter logWriter;
        private int loginTimeout;
        private DataSource underlying;

        public LazyEmbeddedDataSource(boolean isXA) {
            this.isXA = isXA;
        }

        public Connection getConnection() throws SQLException {
            if (underlying == null) {
                createUnderlying();
            }
            return underlying.getConnection();
        }

        public Connection getConnection(String username, String password) throws SQLException {
            if (underlying == null) {
                createUnderlying();
            }
            return underlying.getConnection(username, password);
        }

        public PrintWriter getLogWriter() throws SQLException {
            return logWriter;
        }

        public int getLoginTimeout() throws SQLException {
            return loginTimeout;
        }

        public void setLogWriter(PrintWriter out) throws SQLException {
            this.logWriter = out;
        }

        public void setLoginTimeout(int seconds) throws SQLException {
            this.loginTimeout = seconds;
        }

        public boolean isWrapperFor(Class<?> iface) throws SQLException {
            return false;
        }

        public <T> T unwrap(Class<T> iface) throws SQLException {
            return null;
        }

        public String getCreateDatabase() {
            return createDatabase;
        }

        public void setCreateDatabase(String createDatabase) {
            this.createDatabase = createDatabase;
        }

        public String getDatabaseName() {
            return databaseName;
        }

        public void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public String getShutdownDatabase() {
            return shutdownDatabase;
        }

        public void setShutdownDatabase(String shutdownDatabase) {
            this.shutdownDatabase = shutdownDatabase;
        }

        public String getUser() {
            return user;
        }

        public void setUser(String user) {
            this.user = user;
        }

        private void createUnderlying() throws SQLException {

            Server server = SystemContext.getContext().getSystem().getServer();
            DBServerFeature embeddedServer = server.getFeature(DBServerFeature.class);
            
            if (embeddedServer == null) {
                throw new IllegalStateException("Required feature " + DBServerFeature.TYPE + " not enabled.");
            }

            underlying = embeddedServer.getEmbeddedDataSource(isXA);

            if (isXA) {
                if (!underlying.getClass().getName().equals("org.apache.derby.jdbc.EmbeddedXADataSource")) {
                    underlying = null;
                }
            } else {
                if (!underlying.getClass().getName().equals("org.apache.derby.jdbc.EmbeddedDataSource")) {
                    underlying = null;
                }
            }

            if (underlying == null) {
                throw new SQLException("Failed creating connection to database."
                        + " Embedded server is not a derby database server.");
            } else {
                try {
                    PropertyUtils.setSimpleProperty(underlying, "user", user);
                    PropertyUtils.setSimpleProperty(underlying, "password", password);
                    PropertyUtils.setSimpleProperty(underlying, "databaseName", databaseName);
                    PropertyUtils.setSimpleProperty(underlying, "createDatabase", createDatabase);
                    PropertyUtils.setSimpleProperty(underlying, "shutdownDatabase", shutdownDatabase);
                    PropertyUtils.setSimpleProperty(underlying, "logWriter", logWriter);
                    PropertyUtils.setSimpleProperty(underlying, "loginTimeout", loginTimeout);
                } catch (Exception ex) {
                    throw new SQLException("Failed creating connection to embedded database."
                            + " Error setting properties on datasource. "
                            + ex.getMessage());
                }
            }

        }

    }

}
