/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.client.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.impl.builder.StAXBuilder;
import org.apache.axiom.om.util.StAXUtils;
import org.apache.axiom.soap.SOAPEnvelope;
import org.apache.axiom.soap.SOAPFactory;
import org.apache.axiom.soap.impl.builder.StAXSOAPModelBuilder;
import org.apache.http.ConnectionReuseStrategy;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseFactory;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.DefaultHttpResponseFactory;
import org.apache.http.impl.DefaultHttpServerConnection;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.BasicHttpProcessor;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpExecutionContext;
import org.apache.http.protocol.HttpProcessor;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.http.protocol.HttpRequestHandlerRegistry;
import org.apache.http.protocol.HttpService;
import org.apache.http.protocol.ResponseConnControl;
import org.apache.http.protocol.ResponseContent;
import org.apache.http.protocol.ResponseDate;
import org.apache.http.protocol.ResponseServer;

public class HTTPCallbackServer {

    public static final String MESSAGE_ID = "messageID";

    private Listener listener;
    private boolean shutdown = true;
    private HttpParams params;
    private HttpProcessor httpProcessor;
    private HttpResponseFactory rf;
    private ConnectionReuseStrategy crs;
    private Map<String, HTTPCallback> callbacks;

    public HTTPCallbackServer(int port, int readTimeout) throws IOException {
        ServerSocket serverSocket = new ServerSocket(port);
        serverSocket.setReuseAddress(true);
        listener = new Listener(serverSocket);
        listener.setDaemon(false);
        params = new BasicHttpParams();
        params.setIntParameter(HttpConnectionParams.SO_TIMEOUT, readTimeout);
        params.setIntParameter(HttpConnectionParams.SOCKET_BUFFER_SIZE, 8 * 1024);
        params.setBooleanParameter(HttpConnectionParams.STALE_CONNECTION_CHECK, false);
        params.setBooleanParameter(HttpConnectionParams.TCP_NODELAY, true);
        crs = new DefaultConnectionReuseStrategy();
        rf = new DefaultHttpResponseFactory();
        httpProcessor = new BasicHttpProcessor();
        ((BasicHttpProcessor)httpProcessor).addInterceptor(new ResponseConnControl());
        ((BasicHttpProcessor)httpProcessor).addInterceptor(new ResponseServer());
        ((BasicHttpProcessor)httpProcessor).addInterceptor(new ResponseDate());
        ((BasicHttpProcessor)httpProcessor).addInterceptor(new ResponseContent());
        callbacks = new HashMap<String, HTTPCallback>();
        callbacks = Collections.synchronizedMap(callbacks);
    }

    public void registerCallback(HTTPCallback callback) {
        callbacks.put(callback.getMessageID(), callback);
    }

    public synchronized void start() {
        if (shutdown) {
            listener.start();
            shutdown = false;
        }
    }

    public synchronized void shutdown() {
        if (shutdown) {
            return;
        }
        listener.shutdown();
        synchronized (listener) {
            while (!shutdown) {
                try {
                    listener.wait();
                } catch (InterruptedException ignore) {
                }
            }
        }
    }

    class Listener extends Thread {

        private ServerSocket serverSocket;

        public Listener(ServerSocket serverSocket) {
            super("CallackServerListener");
            this.serverSocket = serverSocket;
        }

        public void run() {
            while (!serverSocket.isClosed()) {
                try {
                    Socket socket = serverSocket.accept();
                    Thread t = new Thread(new Worker(socket), "CallbackServerWorker");
                    t.setDaemon(true);
                    t.start();
                } catch (SocketException se) {
                    break; // serverSocket closed
                } catch (IOException ie) {
                    ie.printStackTrace();
                    break;
                }
            }
            synchronized (this) {
                shutdown = true;
                this.notify();
            }
        }

        public void shutdown() {
            try {
                serverSocket.close();
            } catch (IOException ie) {
                ie.printStackTrace();
            }
        }

    }

    class Worker extends Thread {

        private Socket socket;

        public Worker(Socket socket) {
            this.socket = socket;
        }

        @Override
        public void run() {
            DefaultHttpServerConnection conn = null;
            try {
                conn = new DefaultHttpServerConnection();
                conn.bind(socket, params);
                HttpRequestHandlerRegistry reqistry = new HttpRequestHandlerRegistry();
                reqistry.register("*", new RequestHandlerImpl());
                HttpService httpService = new HttpService(httpProcessor, crs, rf);
                httpService.setParams(params);
                httpService.setHandlerResolver(reqistry);
                while (!interrupted() && conn.isOpen()) {
                    httpService.handleRequest(conn, new HttpExecutionContext(null));
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            } finally {
                try {
                    conn.shutdown();
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }

    }

    class RequestHandlerImpl implements HttpRequestHandler {

        public void handle(HttpRequest request, HttpResponse httpResponse, HttpContext ctxt) throws HttpException, IOException {

            String messageID = null;
            try {
                URI requestURI = new URI(request.getRequestLine().getUri());
                String queryString = requestURI.getQuery();
                if (queryString == null) {
                    return;
                }
                Properties params = new Properties();
                params.load(new StringReader(queryString));
                messageID = params.getProperty(MESSAGE_ID);
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            HTTPCallback callback = callbacks.remove(messageID);

            if (callback == null) {
                return;
            }

            XMLStreamReader reader = null;
            try {
                Header header = request.getFirstHeader(HTTP.CONTENT_TYPE);
                String contentType = header == null ? "text/plain" : header.getValue();
                SOAPFactory factory = getSOAPFactory(contentType);
                HttpEntity entity = ((HttpEntityEnclosingRequest)request).getEntity();
                InputStream in = entity.getContent();
                reader = StAXUtils.createXMLStreamReader(in, "utf-8");
                StAXBuilder builder = new StAXSOAPModelBuilder(reader, factory, null);
                SOAPEnvelope se = (SOAPEnvelope)builder.getDocumentElement();
                se.build();
                callback.setResponse(se);
            } catch (Exception ex) {
                callback.setException(ex);
            } finally {
                if (reader != null) {
                    try {
                        reader.close();
                    } catch (XMLStreamException ignore) {
                    }
                }
                callback.onComplete();
            }

        }

        private SOAPFactory getSOAPFactory(String contentType) throws HttpException {
            String mediaType = null;
            int index = contentType.indexOf(';');
            if (index > 0) {
                mediaType = contentType.substring(0, index);
            } else {
                mediaType = contentType;
            }
            if (mediaType.equals("text/xml")) {
                return OMAbstractFactory.getSOAP11Factory();
            } else if (mediaType.equals("application/soap+xml")) {
                return OMAbstractFactory.getSOAP12Factory();
            } else {
                throw new HttpException("Unrecognized media type " + mediaType);
            }
        }

    }

}
