/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system.dependency;

import java.io.File;

import org.bluestemsoftware.specification.eoa.Dependency;

public class AbstractDependency {
    
    public static final String DEFAULT_ORGANIZATION_ID = "org.mycompany.test";
    public static final String DEFAULT_VERSION = "1.0";

    protected File file = new File("");
    protected String organizationID = DEFAULT_ORGANIZATION_ID;
    protected String artifactID;
    protected String version = DEFAULT_VERSION;
    protected String extension;
    
    public AbstractDependency(String artifactID, String extension) {
        if (artifactID == null) {
            throw new IllegalArgumentException("artifactID null");
        }
        if (extension == null) {
            throw new IllegalArgumentException("extension null");
        }
        this.artifactID = artifactID;
        this.extension = extension;
    }
    
    public AbstractDependency(File file, String artifactID, String extension) {
        if (file == null) {
            throw new IllegalArgumentException("file null");
        }
        if (artifactID == null) {
            throw new IllegalArgumentException("artifactID null");
        }
        if (extension == null) {
            throw new IllegalArgumentException("extension null");
        }
        this.file = file;
        this.artifactID = artifactID;
        this.extension = extension;
    }
    
    public AbstractDependency(String organizationID, String artifactID, String extension) {
        if (artifactID == null) {
            throw new IllegalArgumentException("artifactID null");
        }
        if (extension == null) {
            throw new IllegalArgumentException("extension null");
        }
        if (organizationID != null) {
            this.organizationID = organizationID;
        }
        this.artifactID = artifactID;
        this.extension = extension;
    }
    
    public AbstractDependency(String organizationID, String artifactID, String version, String extension) {
        if (artifactID == null) {
            throw new IllegalArgumentException("artifactID null");
        }
        if (extension == null) {
            throw new IllegalArgumentException("extension null");
        }
        if (organizationID != null) {
            this.organizationID = organizationID;
        }
        this.artifactID = artifactID;
        this.extension = extension;
        if (version != null) {
            this.version = version;
        }
    }
    
    public AbstractDependency(File file, String organizationID, String artifactID, String version, String extension) {
        if (file == null) {
            throw new IllegalArgumentException("file null");
        }
        if (artifactID == null) {
            throw new IllegalArgumentException("artifactID null");
        }
        if (extension == null) {
            throw new IllegalArgumentException("extension null");
        }
        this.file = file;
        if (organizationID != null) {
            this.organizationID = organizationID;
        }
        this.artifactID = artifactID;
        this.extension = extension;
        if (version != null) {
            this.version = version;
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getRef()
     */
    public final String getRef() {
        return organizationID + "/" + artifactID + "/" + version + "/" + extension;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getVersionlessRef()
     */
    public final String getVersionlessRef() {
        return organizationID + "/" + artifactID + "/" + extension;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getShortName()
     */
    public final String getShortName() {
        if (artifactID == null) {
            throw new IllegalStateException("Error generating short name. 'artifactID' not defined");
        }
        if (version == null) {
            throw new IllegalStateException("Error generating short name. 'version' not defined");
        }
        if (extension == null) {
            throw new IllegalStateException("Error generating short name. 'extension' not defined");
        }
        return artifactID + ":" + version + ":" + extension;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getOrganizationID()
     */
    public final String getOrganizationID() {
        return organizationID;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getArtifactID()
     */
    public final String getArtifactID() {
        return artifactID;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getVersion()
     */
    public final String getVersion() {
        return version;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getExtension()
     */
    public final String getExtension() {
        return extension;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.Dependency#getFile()
     */
    public final File getFile() {
        return file;
    }

     /** ********************* Identity Methods *********************************** */

    @Override
    public final String toString() {
        return getRef();
    }

    @Override
    public final int hashCode() {
        return hashCode(getRef());
    }

    @Override
    public final boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof Dependency)) {
            return false;
        }

        if ((getRef() == ((Dependency)obj).getRef() || (getRef() != null && getRef().equals(
                ((Dependency)obj).getRef())))) {
            return true;
        }

        return false;
    }

    private int hashCode(Object object) {
        return (object == null ? 0 : object.hashCode());
    }

}
