/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system.cfg;

import java.io.File;
import java.io.FileFilter;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Appender;
import org.apache.log4j.Logger;
import org.apache.log4j.WriterAppender;
import org.apache.log4j.xml.DOMConfigurator;
import org.bluestemsoftware.open.eoa.test.system.dependency.FactoryDependencyImpl;
import org.bluestemsoftware.open.eoa.test.system.util.Constants;
import org.bluestemsoftware.open.eoa.test.system.util.FactoryDependencyDeployment;
import org.bluestemsoftware.open.eoa.test.system.util.SetupException;
import org.bluestemsoftware.specification.eoa.ComponentDependency;
import org.bluestemsoftware.specification.eoa.Deployment;
import org.bluestemsoftware.specification.eoa.DeploymentClassLoader;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.FactoryDependency;
import org.bluestemsoftware.specification.eoa.FeatureDependency;
import org.bluestemsoftware.specification.eoa.ProvidedDependency;
import org.bluestemsoftware.specification.eoa.ScopedDependency;
import org.bluestemsoftware.specification.eoa.SharedClassLoader;
import org.bluestemsoftware.specification.eoa.SharedDependency;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDependencies;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDeployment;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory.Provider;
import org.bluestemsoftware.specification.eoa.system.SystemClassLoader;

/**
 * An implementation of <code>SystemConfiguration</code> which can be extended to override
 * default set of <code>SystemDependencies</code>. Note that this instance is used by
 * default if no other service provider found.
 */
public class AbstractSystemConfiguration implements SystemConfiguration {

    protected org.bluestemsoftware.specification.eoa.system.System system;

    protected String myOrganizationID;
    protected String myArtifactID;
    protected String myExtension;
    protected String myVersion;

    protected File baseDir;
    protected File resourcesDir;
    protected File classesDir;

    protected File containerFile;
    protected File serverFile;
    protected File etcDir;
    protected Class<?> systemImplClass;
    protected transient int exitCode;

    protected Deployment deploymentUnderTest;

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.test.system.cfg.SystemConfiguration#setup()
     */
    public boolean setUp() throws Exception {

        // retrieve system properties set by surefire booter which identify
        // the 'deployment under test'

        myOrganizationID = java.lang.System.getProperty("eoa.system.dut.groupid");
        myArtifactID = java.lang.System.getProperty("eoa.system.dut.artifactid");
        myExtension = java.lang.System.getProperty("eoa.system.dut.packaging");
        myVersion = java.lang.System.getProperty("eoa.system.dut.version");

        baseDir = new File(System.getProperty("basedir"));
        classesDir = new File(baseDir, "target/classes");
        resourcesDir = new File(java.lang.System.getProperty("testClassesDirectory"));
        
        // if no tests are defined, e.g. as is the case with a newly
        // generated archetype, we'll opt out of set-up
        
        if (!testsExist(resourcesDir)) {
            return true;
        }

        Map<String, Object> testProperties = setSystemProperties();

        defineFactoryUnderTest(testProperties);
        addMockFactoryDependencies(testProperties);
        testProperties.put("mockComponentDependencies", getMockComponentDependencies());
        return bootstrapSystem();

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.test.system.cfg.SystemConfiguration#destroy()
     */
    public boolean destroy() {
        if (!testsExist(resourcesDir)) {
            return true;
        }
        if (system == null) {
            return false;
        }
        Boolean success = false;
        try {
            Method mainMethod = systemImplClass.getDeclaredMethod("shutdown", new Class[] {});
            success = (Boolean)mainMethod.invoke(system, new Object[] {});
        } catch (Exception ex) {
            System.err.println(ex.toString());
        }
        return success;
    }

    /*
     * Overriden by descendants that wish to define dependent factories that generate mock
     * extension providers
     */
    @SuppressWarnings("unchecked")
    protected Set<FactoryDependency> getMockFactoryDependencies() {
        return Collections.EMPTY_SET;
    }

    /*
     * Overriden by descendants that wish to define mock component dependencies
     */
    @SuppressWarnings("unchecked")
    protected Set<ComponentDependency> getMockComponentDependencies() {
        return Collections.EMPTY_SET;
    }

    /* ************************ private methods **************************** */

    private Map<String, Object> setSystemProperties() throws Exception {

        // alakai dir is located within test target dir
        File systemDir = new File(resourcesDir, "alakai");
        System.setProperty(Constants.SYSTEM_DIR_PROPERTY, systemDir.getAbsolutePath());

        // with a required config dir named ...
        etcDir = new File(systemDir, "etc/");
        System.setProperty(Constants.SYSTEM_ETC_DIR_PROPERTY, etcDir.getAbsolutePath());

        // and server and container configuration files ...
        containerFile = new File(etcDir, "container.xml");
        serverFile = new File(etcDir, "server.xml");

        // we'll use system impl jars in eoa installation
        String prop = System.getenv().get("ALAKAI_HOME");
        if (prop == null) {
            throw new SetupException("Environment variable ALAKAI_HOME not set.");
        }
        File homeDir = new File(new File(prop).getCanonicalPath());
        File libDir = new File(homeDir, "lib");
        System.setProperty(Constants.SYSTEM_LIB_DIR_PROPERTY, libDir.getAbsolutePath());

        // create var dir
        File varDir = new File(baseDir, "target/var");
        if (!varDir.exists()) {
            varDir.mkdir();
            varDir.deleteOnExit();
        }
        System.setProperty(Constants.SYSTEM_VAR_DIR_PROPERTY, varDir.getAbsolutePath());

        // create tmp dir
        File tmpDir = new File(baseDir, "target/tmp");
        if (!tmpDir.exists()) {
            tmpDir.mkdir();
            tmpDir.deleteOnExit();
        }
        System.setProperty(Constants.SYSTEM_TMP_DIR_PROPERTY, tmpDir.getAbsolutePath());

        // set system namespace property with fixed value
        System.setProperty(Constants.SYSTEM_NAMESPACE_PROPERTY, "http://mycompany.com/eoa/component");

        // add bucket for test related properties
        Map<String, Object> testProperties = new HashMap<String, Object>();
        System.getProperties().put("eoa.system.test.properties", testProperties);
        return testProperties;

    }

    private boolean bootstrapSystem() throws Exception {

        // first we configure the system classloader
        String prop = System.getProperty(Constants.SYSTEM_LIB_DIR_PROPERTY);
        File libDir = new File(new File(prop).getCanonicalPath());
        if (!libDir.exists()) {
            throw new SetupException("required property '"
                    + Constants.SYSTEM_LIB_DIR_PROPERTY
                    + "' points to an invalid path '"
                    + libDir.getAbsolutePath()
                    + "'.");
        }

        // scope all jars in lib dir to system classloader, i.e.
        // hide system implementation
        File[] contents = libDir.listFiles(new JarFilter());
        if (contents.length == 0) {
            throw new SetupException("no system artifacts found within dir " + libDir.toString());
        }

        URL[] artifacts = new URL[contents.length];
        for (int i = 0; i < contents.length; i++) {
            artifacts[i] = contents[i].toURI().toURL();
        }

        // retrieve parent classloader and shared loader configured by surefire
        // booter plugin and use it to create system classloader
        ClassLoader pcl = (ClassLoader)System.getProperties().get("eoa.system.parent.classloader");
        SharedClassLoader scl = (SharedClassLoader)System.getProperties().get("eoa.system.shared.classloader");
        SystemClassLoader systemClassLoader = new SystemClassLoader(pcl, artifacts);
        systemClassLoader.setSharedClassLoader(scl);

        // load the logging implementation implemented by standalone
        // system and set commons logging property which forces our
        // implementation and prevents deployments from overriding. the
        // commons logging 1.1.x artifact must be on shared classpath
        File loggingConfig = new File(etcDir, "log4j.xml");
        if (!loggingConfig.exists()) {
            throw new SetupException("missing logging configuration " + loggingConfig.getCanonicalPath());
        } else {
            ClassLoader current = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(systemClassLoader);
                DOMConfigurator.configure(loggingConfig.getCanonicalPath());
                Enumeration<?> e = Logger.getRootLogger().getAllAppenders();
                while (e.hasMoreElements()) {
                    Appender appender = (Appender)e.nextElement();
                    if (appender instanceof WriterAppender) {
                        // we override the platform default encoding
                        ((WriterAppender)appender).setEncoding("UTF-8");
                    }
                }
            } finally {
                Thread.currentThread().setContextClassLoader(current);
            }
            System.setProperty("org.apache.commons.logging.LogFactory",
                    "org.apache.commons.logging.impl.LogFactoryImpl");
        }

        // load implementation class using system classloader and retrieve
        // singleton
        try {
            systemImplClass = systemClassLoader.loadClass(Constants.SYSTEM_IMPL_CLASS);
            Method getInstance = systemImplClass.getMethod("getInstance", new Class[] {});
            Object obj = getInstance.invoke(systemImplClass, new Object[] {});
            system = org.bluestemsoftware.specification.eoa.system.System.class.cast(obj);
        } catch (Exception ex) {
            throw new SetupException(ex.toString());
        }

        // start system on a separate thread which will die when system
        // is shutdown, i.e. when this instance is destroyed

        Thread bootstrapper = new Thread() {
            public void run() {
                try {
                    Class<?>[] signature = { File.class, File.class };
                    Method mainMethod = systemImplClass.getDeclaredMethod("main", signature);
                    synchronized (system) {
                        exitCode = (Integer)mainMethod.invoke(systemImplClass, new Object[] { containerFile,
                                serverFile });
                        system.notifyAll();
                    }
                } catch (Exception ex) {
                    System.err.println(ex.toString());
                }
            }
        };

        // set system classloader as context classloader. system impl
        // will retrieve and set on system context
        bootstrapper.setContextClassLoader(systemClassLoader);
        bootstrapper.start();

        // wait for system to finish startup before returning. system
        // issues a notify on system object when complete
        Method isStarted = systemImplClass.getDeclaredMethod("isStarted", new Class[] {});
        synchronized (system) {
            while (exitCode > -1 && isStarted.invoke(system, new Object[] {}).equals(Boolean.FALSE)) {
                try {
                    system.wait();
                } catch (InterruptedException ignore) {
                }
            }
        }

        // check to see if system failed to start. if so, we cannot
        // continue
        if (exitCode < 0) {
            return false;
        }

        // execute resume all on system impl, which will begin the
        // flow of messages on all executable extensions
        Method resumeAll = systemImplClass.getDeclaredMethod("resumeAll", new Class[] {});
        resumeAll.invoke(system, new Object[] {});

        // verify that expected deployment under test was deployed, i.e.
        // to avoid situation where decendant test passes wrong
        // constructor parms and ref/versionless ref set in test props
        // is invalid

        if (myExtension.equals("eoa-component")) {
            String ref = myOrganizationID + "/" + myArtifactID + "/" + myVersion + "/" + myExtension;
            deploymentUnderTest = system.getComponentDeployment(ref);
            if (deploymentUnderTest == null) {
                throw new SetupException("Component deployment under test' " + ref + "' was never deployed.");
            }
        } else {
            String vref = myOrganizationID + "/" + myArtifactID + "/" + myExtension;
            if (system.getExtensionFactory(vref) == null) {
                throw new SetupException("Deployment for factory under test' " + vref + "' was never deployed.");
            }
            deploymentUnderTest = system.getExtensionFactory(vref).getFactoryContext().getDeployment();
        }

        return true;

    }

    /*
     * defines system properties picked up by alakai-standalone-implementation, and
     * maven-dependency-manager when deployment is loaded within context of test, i.e.
     * 'deployment under test'
     */
    private void defineFactoryUnderTest(Map<String, Object> testProperties) throws DeploymentException {
        Object dcl = System.getProperties().get("eoa.system.deployment.classloader");
        Object scl = System.getProperties().get("eoa.system.shared.classloader");
        testProperties.put("organizationID", myOrganizationID);
        testProperties.put("artifactID", myArtifactID);
        testProperties.put("version", myVersion);
        testProperties.put("extension", myExtension);
        testProperties.put("deploymentClassLoader", dcl);
        testProperties.put("sharedClassLoader", scl);
    }

    /*
     * picked up by alakai-standalone-implementation and deployed, i.e. rather than being
     * defined within dependency model
     */
    private void addMockFactoryDependencies(Map<String, Object> testProperties) {
        Set<FactoryDependency> mockFactoryDependencies = getMockFactoryDependencies();
        Set<DeploymentContext> mockFactoryDeployments = new HashSet<DeploymentContext>();
        for (FactoryDependency efd : mockFactoryDependencies) {
            Object dcl = System.getProperties().get("eoa.system.deployment.classloader");
            File file = efd.getFile();
            String organizationID = efd.getOrganizationID();
            String artifactID = efd.getArtifactID();
            String version = efd.getVersion();
            String extension = efd.getExtension();
            List<ProvidedDependency> prds = efd.getProvidedDependencies();
            List<SharedDependency> shds = efd.getSharedDependencies();
            List<ScopedDependency> scds = efd.getScopedDependencies();
            List<FeatureDependency> ffds = efd.getFeatureDependencies();
            ExtensionFactoryDependencies dd = new ExtensionFactoryDependencies(prds, shds, scds, ffds);
            Provider p = ((FactoryDependencyImpl)efd).getProvider();
            ExtensionFactoryDeployment deployment = new FactoryDependencyDeployment(organizationID, artifactID,
                    version, extension, file, dd, p);
            ExtensionFactoryContext efc = new ExtensionFactoryContext(deployment, (DeploymentClassLoader)dcl);
            mockFactoryDeployments.add(efc);
        }
        testProperties.put("mockFactoryDependencies", mockFactoryDependencies);
        testProperties.put("mockFactoryDeployments", mockFactoryDeployments);
    }
    
    private boolean testsExist(File directory) {
        boolean exists = false;
        for (File file : directory.listFiles()) {
            if (file.isDirectory()) {
                if (exists = testsExist(file)) {
                    break;
                }
            } else {
                if (file.getName().endsWith(".class")) {
                    exists = true;
                    break;
                }
            }            
        }
        return exists;
    }

    static class JarFilter implements FileFilter {
        public boolean accept(File pathname) {
            return pathname.toString().endsWith("jar");
        }
    }

}
