/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system.util;


/**
 * A multi-threaded 'test harness'.
 */
public class MultiThreadedTest {

    private int runningThreads;
    private Thread[] testThreads = null;
    private Thread waitingThread = null;
    private boolean success = true;
    private Object lock = new Object();

    /**
     * Constructor
     * 
     * @param threads the number of concurrent threads
     * @param workerFactory factory used to create workers.
     */
    public MultiThreadedTest(int threads, WorkerFactory workerFactory) {
        runningThreads = threads;
        testThreads = new Thread[threads];
        for (int i = 0; i < threads; i++) {
            testThreads[i] = new TestThread(workerFactory.createWorker());
        }
    }

    /**
     * Executes multi-threaded test.
     * 
     * @return true if successful, false if an error occurred. Error, if any, is logged 
     *         via log4j logger.
     */
    public boolean execute() {

        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                if (waitingThread != null) {
                    waitingThread.interrupt();
                }
            }
        });

        for (int i = 0; i < testThreads.length; i++) {
            testThreads[i].start();
        }

        synchronized (lock) {
            while (runningThreads > 0) {
                try {
                    waitingThread = Thread.currentThread();
                    lock.wait();
                    waitingThread = null;
                } catch (InterruptedException fallout) {
                }
            }
        }

        return success;

    }

    private class TestThread extends Thread {

        private Worker worker;

        public TestThread(Worker worker) {
            this.worker = worker;
        }

        public void run() {
            try {
                worker.doWork();
            } catch (Throwable th) {
                System.err.println("worker threw exception: " + th);
                success = false;
            } finally {
                try {
                    synchronized (lock) {
                        --runningThreads;
                        lock.notifyAll();
                    }
                } catch (Exception ignore) {
                }
            }
        }

    }

}
