/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system.dependency;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.bluestemsoftware.specification.eoa.ComponentDependency;
import org.bluestemsoftware.specification.eoa.FeatureDependency;
import org.bluestemsoftware.specification.eoa.ProvidedDependency;
import org.bluestemsoftware.specification.eoa.ScopedDependency;
import org.bluestemsoftware.specification.eoa.SharedDependency;

/**
 * Provides ability to mock a component dependency within context of an integration test.
 * See documentation for examples.
 * 
 */
public class ComponentDependencyImpl extends AbstractDependency implements ComponentDependency {

    public static final String EXTENSION = "eoa-component";

    private List<ComponentDependency> componentDependencies;
    
    /**
     * Constructs a 'root', i.e. non-transitive, non-optional, component dependency using the
     * provided deployment file and default values for organizationID and version.
     * 
     * @param file
     * @param artifactID
     * @param componentDependencies
     *        upon which this root dependency defines.
     */
    public ComponentDependencyImpl(File file, String artifactID) {
        super(file, artifactID, EXTENSION);
        this.componentDependencies = new ArrayList<ComponentDependency>();
    }

    /**
     * Constructs a 'root', i.e. non-transitive, non-optional, component dependency using the
     * provided deployment file and default values for organizationID and version.
     * 
     * @param file
     * @param artifactID
     * @param componentDependencies
     *        upon which this root dependency defines.
     */
    public ComponentDependencyImpl(File file, String artifactID, List<ComponentDependency> componentDependencies) {
        super(file, artifactID, EXTENSION);
        if (componentDependencies == null) {
            throw new IllegalArgumentException("componentDependencies null");
        } else {
            this.componentDependencies = componentDependencies;
        }
    }

    /**
     * Constructs a 'root', i.e. non-transitive, non-optional, component dependency using
     * provided deployment file and transitive component dependencies.
     * 
     * @param file
     * @param organizationID
     * @param artifactID
     * @param version
     * @param componentDependencies
     *        upon which this root dependency defines or an empty list if none defined.
     */
    public ComponentDependencyImpl(File file, String organizationID, String artifactID, String version,
            List<ComponentDependency> componentDependencies) {
        super(file, organizationID, artifactID, version, EXTENSION);
        if (componentDependencies == null) {
            throw new IllegalArgumentException("componentDependencies null");
        } else {
            this.componentDependencies = componentDependencies;
        }
    }

    /**
     * Constructs an optional, 'transitive' component dependency, i.e. no file is provided, it
     * is instead resolved using system dependency manager, i.e. the organizationID,
     * artifactID, and version MUST resolve to an artifact within an in-scope repository.
     * <p>
     * Note that because the constructed dependency is optional, transitive dependencies
     * defined by this dependency are effectively ignored.
     * 
     * @param organizationID
     * @param artifactID
     * @param version
     */
    public ComponentDependencyImpl(String organizationID, String artifactID, String version) {
        super(organizationID, artifactID, version, EXTENSION);
        this.componentDependencies = new ArrayList<ComponentDependency>();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.FactoryDependency#getProvidedDependencies()
     */
    @SuppressWarnings("unchecked")
    public List<ProvidedDependency> getProvidedDependencies() {
        return Collections.EMPTY_LIST;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.FactoryDependency#getSharedDependencies()
     */
    @SuppressWarnings("unchecked")
    public List<SharedDependency> getSharedDependencies() {
        return Collections.EMPTY_LIST;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.FactoryDependency#getScopedDependencies()
     */
    @SuppressWarnings("unchecked")
    public List<ScopedDependency> getScopedDependencies() {
        return Collections.EMPTY_LIST;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.FactoryDependency#getFeatureDependencies()
     */
    @SuppressWarnings("unchecked")
    public List<FeatureDependency> getFeatureDependencies() {
        return Collections.EMPTY_LIST;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ComponentDependency#getComponentDependencies()
     */
    public List<ComponentDependency> getComponentDependencies() {
        return componentDependencies;
    }

}