/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.test.system.binding.HTTPBindingFactoryProvider;
import org.bluestemsoftware.open.eoa.test.system.binding.SOAPBindingFactoryProvider;
import org.bluestemsoftware.open.eoa.test.system.cfg.AbstractSystemConfiguration;
import org.bluestemsoftware.open.eoa.test.system.util.MockExtensionProviderFactory;
import org.bluestemsoftware.specification.eoa.ApplicationClassLoader;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.FactoryDependency;
import org.bluestemsoftware.specification.eoa.component.binding.Binding;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference.ResponseActionReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.MEP;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.FaultContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.service.Endpoint;
import org.bluestemsoftware.specification.eoa.ext.ExecutableException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionException;
import org.bluestemsoftware.specification.eoa.ext.binding.BindingException;
import org.bluestemsoftware.specification.eoa.ext.binding.http.XMLHTTPProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.SOAP11HTTPProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.SOAP11VMTransportProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.SOAP12HTTPProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.SOAP12VMTransportProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAP11HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAP11VMBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAP12HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAP12VMBindingRT;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerResponse;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.w3c.dom.Element;

public abstract class AbstractApplicationTest extends AbstractIntegrationTest {

    protected EngineRT engineUnderTest;
    protected EngineReference engineReference;

    protected void setUp() throws Exception {
        super.setUp();
        if (system.getEngine(getEngineName()) == null) {
            throw new IllegalStateException("Engine under test " + getEngineName() + " is undefined.");
        }
        engineUnderTest = system.getEngine(getEngineName()).getRuntimeProvider();
        engineReference = engineUnderTest.getEngineReference(engineUnderTest.getName());
    }

    protected void tearDown() throws Exception {
        super.tearDown();
    }

    protected RoleUnderTest getRoleUnderTest(String roleName) {
        ServiceReference sr = engineReference.getServiceReference(roleName);
        if (sr == null) {
            throw new IllegalStateException("Role " + roleName + " is undefined.");
        }
        return new RoleUnderTest(sr);
    }

    /**
     * Gets name of <code>Engine</code> which implements application under test.
     * @return
     */
    protected abstract QName getEngineName();

    /**
     * Mocks partner behavior within context of application test. See example projects for
     * usage.
     */
    public interface MockPartnerRole {

        /**
         * Processes request and returns response.
         * 
         * @param partnerOperation
         * @param request
         * @return response or <code>null</code> if MEP in-only or if MEP 'robust-in-only'
         *         and no fault occurred.
         */
        public ActionContext handleRequest(MockPartnerOperation partnerOperation, ActionContext request);

    }

    /**
     * Mocks a runtime operation defined on mocked partner role. Provides ability to generate
     * response, if any.
     */
    public static class MockPartnerOperation {

        // because partner engine is mocked, this is a partner operation
        // ref, i.e. owned by engine under test. so ... we wrap it in
        // this class to make response action generation methods
        // available to user. nothing else is relevant

        private EndpointOperationReference eor;

        public MockPartnerOperation(EndpointOperationReference eor) {
            this.eor = eor;
        }

        /**
         * Creates normal partner response.
         * 
         * @param relatesTo
         * @return
         */
        public MessageContext createResponseAction(String relatesTo) {
            EndpointActionReference ear = eor.getResponseAction();
            if (ear == null) {
                throw new IllegalArgumentException("Response action for operation "
                        + eor.getEndpointOperationName()
                        + " is undefined.");
            }
            return (MessageContext)((ResponseActionReference)ear).createAction(relatesTo);
        }

        /**
         * Creates fault partner response.
         * 
         * @param faultName
         * @return
         */
        public FaultContext createFaultAction(String relatesTo, QName faultName) {
            EndpointActionReference ear = eor.getFaultAction(faultName);
            if (ear == null) {
                throw new IllegalArgumentException("Fault ref "
                        + faultName
                        + " is undefined on operation "
                        + eor.getEndpointOperationName()
                        + ".");
            }
            return (FaultContext)((ResponseActionReference)ear).createAction(relatesTo);
        }

        /**
         * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
         * 
         * @param faultReason
         *        fault description
         * @return FaultContext
         */
        public FaultContext createSystemFaultAction(String relatesTo, String faultReason) {
            return eor.getParent().createSystemFaultAction(relatesTo, faultReason);
        }

        /**
         * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
         * 
         * @param faultReason
         *        fault description
         * @param faultCode
         *        A specific indication of fault, i.e. a fault sub-type, within context of the
         *        general classification {@link #NAME}.
         * @return FaultContext
         */
        public FaultContext createSystemFaultAction(String relatesTo, QName faultCode, String faultReason) {
            return eor.getParent().createSystemFaultAction(relatesTo, faultCode, faultReason);
        }

        /**
         * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
         * 
         * @param faultReason
         *        fault description
         * @param cause
         *        Throwable which caused fault. Note throwable MAY be an instanceof
         *        <code>SystemFault</code>, i.e. a 'nested' fault.
         *        <p>
         *        Note that if <code>Throwable</code> is not an instance of
         *        <code>SystemFault</code>, a stacktrace is serialized to 'FaultDetail'
         *        element of message returned via {@link SystemFault#toMessage()}.
         * @return FaultContext
         */
        public FaultContext createSystemFaultAction(String relatesTo, String faultReason, Throwable cause) {
            return eor.getParent().createSystemFaultAction(relatesTo, faultReason, cause);
        }

        /**
         * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
         * 
         * @param cause
         *        Throwable which caused fault. Note throwable MAY be an instanceof
         *        <code>SystemFault</code>.
         *        <p>
         *        Note that if <code>Throwable</code> is not an instance of
         *        <code>SystemFault</code>, a stacktrace is serialized to 'FaultDetail'
         *        element of message returned via {@link SystemFault#toMessage()}.
         * @return FaultContext
         */
        public FaultContext createSystemFaultAction(String relatesTo, Throwable cause) {
            return eor.getParent().createSystemFaultAction(relatesTo, cause);
        }

        /**
         * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
         * 
         * @param systemFault
         * 
         * @return FaultContext
         */
        public FaultContext createSystemFaultAction(String relatesTo, SystemFault systemFault) {
            return eor.getParent().createSystemFaultAction(relatesTo, systemFault);
        }

    }

    /**
     * Models 'role under test'. Provides access to invokable operations on modeled role. This
     * class is threadsafe and can be re-used. See example projects for usage.
     */
    public class RoleUnderTest {

        private ServiceReference sr;

        public RoleUnderTest(ServiceReference sr) {
            this.sr = sr;
        }

        /**
         * Retrieves 'operation under test.'
         * @param operationName
         * @return
         */
        public OperationUnderTest getOperationUnderTest(QName operationName) {
            EndpointReference epr = sr.getEndpointReferences().next();
            EndpointOperationReference eor = epr.getOperationReference(operationName);
            return new OperationUnderTest(eor);
        }

    }

    /**
     * Models 'operation under test.' This class is threadsafe and can be re-used. See example
     * projects for usage.
     */
    public class OperationUnderTest {

        private EndpointOperationReference eor;

        public OperationUnderTest(EndpointOperationReference eor) {
            this.eor = eor;
        }

        /**
         * Creates request action.
         * 
         * @return
         */
        public ActionContext createRequest() {
            return eor.getRequestAction().createAction();
        }

        /**
         * Invokes 'operation under test.'
         * 
         * @param request
         * @return ActionContext response
         */
        public ActionContext sendRequest(ActionContext request) {
            Endpoint endpoint = eor.getParent().getEndpoint();
            Binding binding = endpoint.getBindingReference().getReferencedComponent();
            Extension.Provider p = binding.getRuntimeProvider().getExtensionProvider();
            if (p instanceof AbstractBindingProvider == false) {
                throw new IllegalStateException("Binding provider not instance of "
                        + AbstractBindingProvider.class);
            }
            Callback callback = new Callback(request.getMessageID());
            ((AbstractBindingProvider)p).registerCallback(callback);
            eor.getParent().receiveAction(request);
            synchronized (callback) {
                while (callback.getResponse() == null) {
                    try {
                        callback.wait();
                    } catch (InterruptedException ie) {
                        throw new IllegalStateException(ie);
                    }
                }
            }
            return callback.getResponse();
        }

    }

    /**
     * Used to construct {@link SOAPBindingFactoryProvider} or
     * {@link HTTPBindingFactoryProvider} as required to generate a mock
     * {@link FactoryDependency}.
     * <p>
     * Note, to mock partner role behavior, extend {@link AbstractSystemConfiguration} and
     * implement {@link ApplicationConfiguration.} See example projects for usage.
     * 
     */
    public static class MockBindingFactory implements MockExtensionProviderFactory {

        /*
         * (non-Javadoc)
         * @see org.bluestemsoftware.open.eoa.test.system.util.MockExtensionProviderFactory#createExtensionProvider(java.lang.reflect.Method,
         *      java.lang.Object[])
         */
        public Extension.Provider createExtensionProvider(Method method, Object[] args) throws ExtensionException {
            String protocolType = (String)args[2];
            if (protocolType.equals(SOAP11HTTPProtocol.TYPE)) {
                return new MockSOAPHTTPBindingProvider();
            } else if (protocolType.equals(SOAP12HTTPProtocol.TYPE)) {
                return new MockSOAPHTTPBindingProvider();
            } else if (protocolType.equals(SOAP11VMTransportProtocol.TYPE)) {
                return new MockSOAPVMBindingProvider();
            } else if (protocolType.equals(SOAP12VMTransportProtocol.TYPE)) {
                return new MockSOAPVMBindingProvider();
            } else if (protocolType.equals(XMLHTTPProtocol.TYPE)) {
                return new MockHTTPBindingProvider();
            } else {
                throw new IllegalArgumentException("Unsupported protocol " + protocolType);
            }
        }

    }

    @SuppressWarnings("unchecked")
    private static class AbstractBindingProvider {

        private Map<String, Callback> callbacks = new HashMap<String, Callback>();
        private Map<String, MockPartnerRole> mockPartnerRoles;

        public synchronized void registerCallback(Callback callback) {
            callbacks.put(callback.getRequestMessageID(), callback);
        }

        /**
         * Receives partner response.
         * 
         * @param actionContext
         */
        protected void sendMyServiceResponse(EndpointReference epr, ActionContext response) {
            
            // perform a santity check here, in case user retrieved reference
            // to epr, created action and sent using eoa api, i.e. as
            // opposed to via mockoperation, etc ..
            
            String action = response.getAction();
            EndpointActionReference ear = epr.getEndpointActionReference(action);
            Direction dir = ear.getReferencedComponent().getInterfaceAction().getDirection();
            if (dir != Direction.OUT) {
                throw new IllegalStateException("An attempt was made to send 'my' service"
                        + " response for operation "
                        + ear.getParent().getEndpointOperationName()
                        + ", but action "
                        + action
                        + " has message direction 'In'. Did you intend to invoke 'receiveAction'");
            }
            
            Callback callback = null;
            synchronized (callbacks) {
                callback = callbacks.remove(response.getRelatesTo());
            }
            synchronized (callback) {
                callback.setResponse(response);
                callback.notify();
            }
            
        }

        /**
         * Sends partner request.
         * @param epr
         * @param request
         */
        protected void sendPartnerServiceRequest(EndpointReference epr, ActionContext request) {
            
            // perform a santity check here, in case user retrieved reference
            // to epr, created action and sent using eoa api, i.e. as
            // opposed to via mockoperation, etc ..
            
            String action = request.getAction();
            EndpointActionReference ear = epr.getEndpointActionReference(action);
            Direction dir = ear.getReferencedComponent().getInterfaceAction().getDirection();
            if (dir != Direction.IN) {
                throw new IllegalStateException("An attempt was made to send 'partner' service"
                        + " request defined on operation "
                        + ear.getParent().getEndpointOperationName()
                        + ", but action "
                        + action
                        + " has message direction 'Out'. Did you intend to invoke 'sendAction'?");
            }
 
            
            String roleName = epr.getParent().getRoleName();
            MockPartnerRole mockPartnerRole = getMockPartnerRoles().get(roleName);
            if (mockPartnerRole == null) {
                throw new IllegalStateException("No MockPartnerRole defined for role '" + roleName + "'.");
            }
            
            ActionContext response = null;
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            try {
                // give mocked role access to classes visible to application under test,
                // and enable role to use MyApplicationContext.getApplication() which
                // requires application classloader of application under test. note that
                // current classloader is application component deployment classloader
                // which is the loader that loaded mock binding factory dependency that
                // created this mock binding
                EngineRT engine = epr.getRootComponent();
                ApplicationClassLoader acl = engine.getApplicationClassLoader();
                Thread.currentThread().setContextClassLoader(acl);
                MockPartnerOperation mockedOperation = new MockPartnerOperation(ear.getParent());
                response = mockPartnerRole.handleRequest(mockedOperation, request);
            } catch (Throwable th) {
                throw new RuntimeException("MockPartnerRole "
                        + roleName
                        + " threw unchecked exception. "
                        + th);
            } finally {
                Thread.currentThread().setContextClassLoader(cl);
            }
            
            if (response != null) {
                epr.receiveAction(response);
            } else {
                MEP mep = ear.getParent().getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();
                if (mep == MEP.IN_OUT) {
                    throw new RuntimeException("MockPartnerRole returned null response, operation "
                            + ear.getParent().getEndpointOperationName()
                            + " defines 'in-out' MEP.");
                }
            }
            
        }
        
        public void spi_destroy() {
        }

        public Set<String> spi_getRequiredBindingFeatures() {
            return new HashSet<String>();
        }

        public void spi_init(Set<ManagementContext> managementContexts) throws BindingException {
        }

        public boolean spi_isSuspended() {
            return true;
        }

        public void spi_resume() throws ExecutableException {
        }

        public void spi_suspend() throws ExecutableException {
        }

        public void spi_validateEndpointReference(EndpointReference endpointReference) throws DeploymentException {
        }

        public void spi_setConsumer(Extension consumer) {
        }

        /*
         * lazy loads mock partner roles. roles are not retrieved from application
         * configuration until surefire booter is run, i.e. which is after we've been
         * constructed
         */
        private Map<String, MockPartnerRole> getMockPartnerRoles() {

            if (mockPartnerRoles == null) {

                // mock partner roles, if any, were retrieved from user defined
                // instance of ApplicationConfiguration by SurefireBooter and
                // set within system test properties

                Map<String, Object> testProperties = (Map)System.getProperties().get("eoa.system.test.properties");
                if (testProperties != null) {
                    mockPartnerRoles = (Map)testProperties.get("mock.partner.roles");
                    if (mockPartnerRoles == null) {
                        mockPartnerRoles = new HashMap<String, MockPartnerRole>();
                    }
                }

            }

            return mockPartnerRoles;

        }

    }

    private static class MockSOAPHTTPBindingProvider extends AbstractBindingProvider implements SOAP11HTTPBindingRT.Provider, SOAP12HTTPBindingRT.Provider {

        public void spi_receiveAction(HTTPServerRequest request, HTTPServerResponse response, Direction direction, EndpointReference endpointReference) {
        }

        public void spi_sendAction(EndpointReference endpointReference, ActionContext actionContext) throws SystemFault {
            if (endpointReference.getParent().isMyService()) {
                sendMyServiceResponse(endpointReference, actionContext);
            } else {
                sendPartnerServiceRequest(endpointReference, actionContext);
            }
        }

    }

    private static class MockSOAPVMBindingProvider extends AbstractBindingProvider implements SOAP11VMBindingRT.Provider, SOAP12VMBindingRT.Provider {

        public void spi_receiveAction(EndpointReference endpointReference, Direction direction, Element envelope) {
        }

        public void spi_sendAction(EndpointReference endpointReference, ActionContext actionContext) throws SystemFault {
            if (endpointReference.getParent().isMyService()) {
                sendMyServiceResponse(endpointReference, actionContext);
            } else {
                sendPartnerServiceRequest(endpointReference, actionContext);
            }
        }

    }

    private static class MockHTTPBindingProvider extends AbstractBindingProvider implements HTTPBindingRT.Provider {

        public void spi_receiveAction(HTTPServerRequest request, HTTPServerResponse response, Direction direction, EndpointReference endpointReference) {
        }

        public void spi_sendAction(EndpointReference endpointReference, ActionContext actionContext) throws SystemFault {
            if (endpointReference.getParent().isMyService()) {
                sendMyServiceResponse(endpointReference, actionContext);
            } else {
                sendPartnerServiceRequest(endpointReference, actionContext);
            }
        }

    }

    private static class Callback {

        private String requestMessageID;
        private transient ActionContext response;

        public Callback(String requestMessageID) {
            this.requestMessageID = requestMessageID;
        }

        public ActionContext getResponse() {
            return response;
        }

        public String getRequestMessageID() {
            return requestMessageID;
        }

        public void setResponse(ActionContext response) {
            this.response = response;
        }

    }

}
