/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.system;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.log4j.xml.DOMConfigurator;
import org.bluestemsoftware.open.eoa.test.system.dependency.FactoryDependencyImpl;
import org.bluestemsoftware.open.eoa.test.system.dependency.FeatureDependencyImpl;
import org.bluestemsoftware.open.eoa.test.system.util.FactoryDependencyDeployment;
import org.bluestemsoftware.open.eoa.test.system.util.FactoryUnderTestDeployment;
import org.bluestemsoftware.specification.eoa.ComponentDependency;
import org.bluestemsoftware.specification.eoa.DeploymentClassLoader;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.FactoryDependency;
import org.bluestemsoftware.specification.eoa.FeatureDependency;
import org.bluestemsoftware.specification.eoa.OtherDependency;
import org.bluestemsoftware.specification.eoa.ProvidedDependency;
import org.bluestemsoftware.specification.eoa.ScopedDependency;
import org.bluestemsoftware.specification.eoa.SharedDependency;
import org.bluestemsoftware.specification.eoa.SystemDependencies;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.ComponentDeployment;
import org.bluestemsoftware.specification.eoa.component.ComponentManager;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.RootComponent;
import org.bluestemsoftware.specification.eoa.component.ClasspathDeployment.BuiltInComponentsDeployment;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.application.Application;
import org.bluestemsoftware.specification.eoa.component.binding.Binding;
import org.bluestemsoftware.specification.eoa.component.engine.Engine;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.policy.Policy;
import org.bluestemsoftware.specification.eoa.component.rt.ProviderReader;
import org.bluestemsoftware.specification.eoa.component.service.Service;
import org.bluestemsoftware.specification.eoa.component.types.Schema;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionException;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDependencies;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDeployment;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryManager;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory.Provider;
import org.bluestemsoftware.specification.eoa.ext.application.ApplicationFactory;
import org.bluestemsoftware.specification.eoa.ext.binding.BindingFactory;
import org.bluestemsoftware.specification.eoa.ext.connector.ConnectorFactory;
import org.bluestemsoftware.specification.eoa.ext.container.ContainerException;
import org.bluestemsoftware.specification.eoa.ext.container.ContainerFactory;
import org.bluestemsoftware.specification.eoa.ext.engine.EngineFactory;
import org.bluestemsoftware.specification.eoa.ext.expression.ExpressionFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.rt.TransportProtocolRT;
import org.bluestemsoftware.specification.eoa.ext.idl.IDLFactory;
import org.bluestemsoftware.specification.eoa.ext.management.ManagementContextFactory;
import org.bluestemsoftware.specification.eoa.ext.message.MessageFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.ext.schema.SchemaFactory;
import org.bluestemsoftware.specification.eoa.ext.server.ServerException;
import org.bluestemsoftware.specification.eoa.ext.server.ServerFactory;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.System;
import org.bluestemsoftware.specification.eoa.system.SystemClassLoader;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.container.Connector;
import org.bluestemsoftware.specification.eoa.system.container.Container;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.bluestemsoftware.specification.eoa.system.server.TransportFeature;
import org.w3c.dom.Element;

/**
 * Provides a 'lightweight', UNIT TESTING framework used to test <code>Extension</code>
 * implementations. Appropriate in situations where descendant tests DO NOT rely upon the
 * existence of other 'real' extension objects. Mock and/or stubbed extensions are used
 * instead.
 * <p>
 * Note that by convention, all unit tests SHOULD be located within src/test/java directory.
 */
public class AbstractUnitTest implements System {

    public static final String SYS_ORGANIZATION_ID = "com.mycompany.eoa";
    public static final String SYS_ARTIFACT_ID = "mysystem";
    public static final String SYS_VERSION = "1.0";

    protected File baseDir;
    protected File classesDir;
    protected File resourcesDir;
    protected SystemContext systemContext = new SystemContextImpl();
    protected ComponentManager componentManager;
    protected static ExtensionFactoryManager factoryManager;
    protected File systemVarDir;
    protected File systemTmpDir;
    protected File systemEtcDir;
    protected URI systemNamespace;
    protected String myOrganizationID;
    protected String myArtifactID;
    protected String myExtension;
    protected String myVersion;
    protected boolean factoryUnderTest;
    protected boolean initializeContainer = true;
    protected boolean initializeServer = true;
    private String versionlessRef;

    protected void setUp() throws Exception {

        // retrieve system properties set by surefire booter which identify
        // the 'deployment under test'

        myOrganizationID = java.lang.System.getProperty("eoa.system.dut.groupid");
        myArtifactID = java.lang.System.getProperty("eoa.system.dut.artifactid");
        myExtension = java.lang.System.getProperty("eoa.system.dut.packaging");

        baseDir = new File(java.lang.System.getProperty("basedir"));

        versionlessRef = myOrganizationID + "/" + myArtifactID + "/" + myExtension;

        classesDir = new File(baseDir, "target/classes");

        if (java.lang.System.getProperty("testClassesDirectory") != null) {
            resourcesDir = new File(java.lang.System.getProperty("testClassesDirectory"));
        } else {
            resourcesDir = new File(baseDir, "target/test-classes");
        }

        factoryManager = new ExtensionFactoryManager(null);
        componentManager = new ComponentManager(getLog(this.getClass()), null, new HashSet<ManagementContext>());
        Properties properties = java.lang.System.getProperties();
        DeploymentClassLoader dcl = (DeploymentClassLoader)properties.get("eoa.system.deployment.classloader");
        ComponentContext ctx = new ComponentContext(new BuiltInComponentsDeployment(dcl), dcl);
        componentManager.insert(ctx);
        DOMConfigurator.configure(Thread.currentThread().getContextClassLoader().getResource("log4j.xml"));

        SystemDependencies sd = getSystemDependencies();
        for (FactoryDependency efd : sd.getFactoryDependencies()) {
            deployExtensionFactory(efd);
        }

        if (!factoryUnderTest) {
            throw new Exception("System dependencies must define 'factory under test' as a dependency");
        }

        // concrete test may be testing a server and or container deployment
        // in which case we may need to skip initilization of partially
        // functional server/container

        if (initializeServer) {
            getServer().init(new HashSet<ManagementContext>());
        }
        if (initializeContainer) {
            getContainer().init(new HashSet<ManagementContext>());
        }

    }

    protected void tearDown() throws Exception {
        getContainer().destroy();
        getServer().destroy();
    }

    /*
     * Adds factory under test as a dependency. Subclasses may override to add additional
     * dependencies.
     */
    public SystemDependencies getSystemDependencies() {
        List<SharedDependency> shds = new ArrayList<SharedDependency>();
        List<FactoryDependency> efds = new ArrayList<FactoryDependency>();
        List<ComponentDependency> cds = new ArrayList<ComponentDependency>();
        FactoryDependency efd = null;
        if (myExtension.equals(FactoryDependencyImpl.EXTENSION)) {
            efd = new FactoryDependencyImpl(myOrganizationID, myArtifactID);
        } else if (myExtension.equals(FeatureDependencyImpl.EXTENSION)) {
            efd = new FeatureDependencyImpl(myOrganizationID, myArtifactID, true, null);
        } else {
            throw new IllegalStateException("Unit test not supported for project type " + myExtension);
        }
        efds.add(efd);
        return new SystemDependencies(SYS_ORGANIZATION_ID, SYS_ARTIFACT_ID, SYS_VERSION, shds, efds, cds);
    }

    /** ************************* system impl ********************************* */

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getComponent(org.bluestemsoftware.specification.eoa.component.FragmentIdentifier)
     */
    public RootComponent getComponent(FragmentIdentifier fragmentID) {
        if (fragmentID == null) {
            return null;
        }
        return componentManager.getComponent(fragmentID);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getComponentDeployment(java.lang.String)
     */
    public ComponentDeployment getComponentDeployment(String deploymentRef) {
        return componentManager.getComponentDeployment(deploymentRef);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getApplication(javax.xml.namespace.QName)
     */
    public Application getApplication(QName componentName) {
        return (Application)componentManager.getComponent(ComponentType.APPLICATION, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getBinding(javax.xml.namespace.QName)
     */
    public Binding getBinding(QName componentName) {
        return (Binding)componentManager.getComponent(ComponentType.BINDING, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getEngine(javax.xml.namespace.QName)
     */
    public Engine getEngine(QName componentName) {
        return (Engine)componentManager.getComponent(ComponentType.ENGINE, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getInterface(javax.xml.namespace.QName)
     */
    public Interface getInterface(QName componentName) {
        return (Interface)componentManager.getComponent(ComponentType.INTERFACE, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getMessage(javax.xml.namespace.QName)
     */
    public InterfaceMessage getMessage(QName componentName) {
        return (InterfaceMessage)componentManager.getComponent(ComponentType.MESSAGE, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getPolicy(javax.xml.namespace.QName)
     */
    public Policy getPolicy(QName componentName) {
        return (Policy)componentManager.getComponent(ComponentType.POLICY, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getSchema(javax.xml.namespace.QName)
     */
    public Schema getSchema(QName componentName) {
        return (Schema)componentManager.getComponent(ComponentType.SCHEMA, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getService(javax.xml.namespace.QName)
     */
    public Service getService(QName componentName) {
        return (Service)componentManager.getComponent(ComponentType.SERVICE, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getProviderReader(java.lang.String)
     */
    public ProviderReader<?> getProviderReader(String providerNamespace) {
        return factoryManager.getProviderReader(providerNamespace);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getExtensionFactory(java.lang.String)
     */
    public ExtensionFactory getExtensionFactory(String deploymentRef) {
        return factoryManager.getExtensionFactory(deploymentRef);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getExtensionFactories()
     */
    public List<ExtensionFactory> getExtensionFactories() {
        return factoryManager.getExtensionFactories();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getApplicationFactory(java.lang.String)
     */
    public final ApplicationFactory getApplicationFactory(String ref) {
        return factoryManager.getApplicationFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getExpressionFactory(java.lang.String)
     */
    public final ExpressionFactory getExpressionFactory(String ref) {
        return factoryManager.getExpressionFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getContainerFactory(java.lang.String)
     */
    public ContainerFactory getContainerFactory(String ref) {
        if (ref.equals(ContainerImpl.TYPE)) {
            return ContainerFactoryImpl.getInstance();
        }
        return factoryManager.getContainerFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getServerFactory(java.lang.String)
     */
    public ServerFactory getServerFactory(String ref) {
        if (ref.equals(ServerImpl.TYPE)) {
            return ServerFactoryImpl.getInstance();
        }
        return factoryManager.getServerFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getEngineFactory(java.lang.String)
     */
    public final EngineFactory getEngineFactory(String ref) {
        return factoryManager.getEngineFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getBindingFactory(java.lang.String)
     */
    public final BindingFactory getBindingFactory(String ref) {
        return factoryManager.getBindingFactory(ref);
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getBindingFactories()
     */
    public Set<BindingFactory> getBindingFactories() {
        return factoryManager.getBindingFactories();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getSchemaFactory(java.lang.String)
     */
    public final SchemaFactory getSchemaFactory(String ref) {
        return factoryManager.getSchemaFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getConnectorFactory(java.lang.String,
     *      java.lang.String)
     */
    public final ConnectorFactory getConnectorFactory(String ref, String impl) {
        return factoryManager.getConnectorFactory(ref, impl);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getWSDLFactory(java.lang.String)
     */
    public final IDLFactory getIDLFactory(String ref) {
        return factoryManager.getIDLFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getPolicyFactories()
     */
    public Set<PolicyFactory> getPolicyFactories() {
        return factoryManager.getPolicyFactories();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getSchemaFactories()
     */
    public Set<SchemaFactory> getSchemaFactories() {
        return factoryManager.getSchemaFactories();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getPolicyFactory(java.lang.String)
     */
    public final PolicyFactory getPolicyFactory(String ref) {
        return factoryManager.getPolicyFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getMessageFactory(java.lang.String)
     */
    public MessageFactory getMessageFactory(String ref) {
        return factoryManager.getMessageFactory(ref);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getFeatureFactory(java.lang.Class,
     *      java.lang.String)
     */
    public FeatureFactory getFeatureFactory(String featureType, String impl) {
        return factoryManager.getFeatureFactory(featureType, impl);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getContainer()
     */
    public Container getContainer() {
        return ContainerImpl.getInstance();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.System#getServer()
     */
    public Server getServer() {
        return ServerImpl.getInstance();
    }

    public Log getLog(Class<?> clazz) {
        return new NoOpLog();
    }

    public String getRef() {
        return "http://test/system";
    }

    public File getSystemVarDir() {
        if (systemVarDir == null) {
            systemVarDir = new File(baseDir, "target/var");
            if (!systemVarDir.exists()) {
                systemVarDir.mkdir();
                systemVarDir.deleteOnExit();
            }
        }
        return systemVarDir;
    }

    public File getSystemTmpDir() {
        if (systemTmpDir == null) {
            systemTmpDir = new File(baseDir, "target/tmp");
            if (!systemTmpDir.exists()) {
                systemTmpDir.mkdir();
                systemTmpDir.deleteOnExit();
            }
        }
        return systemTmpDir;
    }
    
    public File getSystemEtcDir() {
        if (systemEtcDir == null) {
            systemEtcDir = new File(baseDir, "target/etc");
            if (!systemEtcDir.exists()) {
                systemEtcDir.mkdir();
                systemEtcDir.deleteOnExit();
            }
        }
        return systemEtcDir;
    }

    /** *********************** private methods ******************************* */

    private void deployExtensionFactory(FactoryDependency efd) throws DeploymentException {
        for (FeatureDependency ffd : efd.getFeatureDependencies()) {
            deployExtensionFactory(ffd);
        }
        if (efd.getVersionlessRef().equals(versionlessRef)) {
            deployFactoryUnderTest(efd);
        } else {
            deployFactoryDependency(efd);
        }
    }

    private void deployFactoryUnderTest(FactoryDependency efd) throws DeploymentException {
        Properties properties = java.lang.System.getProperties();
        DeploymentClassLoader dcl = (DeploymentClassLoader)properties.get("eoa.system.deployment.classloader");
        File file = classesDir; // use instead of empty file defined on efd
        String organizationID = efd.getOrganizationID();
        String artifactID = efd.getArtifactID();
        String version = efd.getVersion();
        String extension = efd.getExtension();
        List<ProvidedDependency> prds = efd.getProvidedDependencies();
        List<SharedDependency> shds = efd.getSharedDependencies();
        List<ScopedDependency> scds = efd.getScopedDependencies();
        List<FeatureDependency> ffds = efd.getFeatureDependencies();
        Map<String, OtherDependency> ods = efd.getOtherDependencies();
        ExtensionFactoryDependencies dd = new ExtensionFactoryDependencies(prds, shds, scds, ffds, ods);
        ExtensionFactoryDeployment dep = new FactoryUnderTestDeployment(organizationID, artifactID, version,
                extension, file, dd);
        ExtensionFactoryContext factoryContext = new ExtensionFactoryContext(dep, dcl);
        factoryManager.deploy(factoryContext);
        factoryUnderTest = true;
    }

    private void deployFactoryDependency(FactoryDependency efd) throws DeploymentException {
        Properties properties = java.lang.System.getProperties();
        DeploymentClassLoader dcl = (DeploymentClassLoader)properties.get("eoa.system.deployment.classloader");
        File file = efd.getFile();
        String organizationID = efd.getOrganizationID();
        String artifactID = efd.getArtifactID();
        String version = efd.getVersion();
        String extension = efd.getExtension();
        List<ProvidedDependency> prds = efd.getProvidedDependencies();
        List<SharedDependency> shds = efd.getSharedDependencies();
        List<ScopedDependency> scds = efd.getScopedDependencies();
        List<FeatureDependency> ffds = efd.getFeatureDependencies();
        Map<String, OtherDependency> ods = efd.getOtherDependencies();
        ExtensionFactoryDependencies dd = new ExtensionFactoryDependencies(prds, shds, scds, ffds, ods);
        Provider p = ((FactoryDependencyImpl)efd).getProvider();
        ExtensionFactoryDeployment deployment = new FactoryDependencyDeployment(organizationID, artifactID,
                version, extension, file, dd, p);
        ExtensionFactoryContext factoryContext = new ExtensionFactoryContext(deployment, dcl);
        factoryManager.deploy(factoryContext);
    }

    /** ************************ inner classes ******************************** */

    static class NoOpLog implements Log {

        public void debug(String message, Throwable t) {
            return;
        }

        public void debug(String message) {
            return;
        }

        public void error(String message, Throwable cause) {
            return;
        }

        public void error(String message) {
            return;
        }

        public void fatal(String message, Throwable t) {
            return;
        }

        public void fatal(String message) {
            return;
        }

        public void info(String message, Throwable cause) {
            return;
        }

        public void info(String message) {
            return;
        }

        public boolean isDebugEnabled() {
            return false;
        }

        public boolean isInfoEnabled() {
            return false;
        }

        public boolean isTraceEnabled() {
            return false;
        }

        public void trace(String message, Throwable cause) {
            return;
        }

        public void trace(String message) {
            return;
        }

        public void warn(String message, Throwable cause) {
            return;
        }

        public void warn(String message) {
            return;
        }

    }

    class SystemContextImpl extends SystemContext {

        private SystemClassLoader systemClassLoader;

        SystemContextImpl() {
            systemContext = this;
        }

        @Override
        public org.bluestemsoftware.specification.eoa.system.System getSystem() {
            return AbstractUnitTest.this;
        }

        @Override
        public SystemClassLoader getSystemClassLoader() {
            if (systemClassLoader == null) {
                systemClassLoader = new SystemClassLoader(new URL[] {});
            }
            return systemClassLoader;
        }

    }

    protected static class ServerImpl extends Server {

        public static final String TYPE = "http://test/server";

        private static Server singleton;
        private Map<Class<?>, Feature> features = new HashMap<Class<?>, Feature>();
        private Map<String, Feature> featuresByTypeRef = new HashMap<String, Feature>();

        public ServerImpl(ExtensionFactory factory, Provider provider, File extensionEtcDir, File extensionVarDir) {
            super(factory, provider, extensionEtcDir, extensionVarDir);
        }

        public static Server getInstance() {
            if (singleton == null) {
                singleton = new ServerImpl(ServerFactoryImpl.getInstance(), new ServerProviderImpl(), null, null);
            }
            return singleton;
        }

        public void init(Set<ManagementContext> managementContexts) throws ExtensionException {
            for (ExtensionFactory ef : factoryManager.getExtensionFactories()) {
                if (ef instanceof FeatureFactory) {
                    Feature temp = ((FeatureFactory)ef).createFeature();
                    if (features.containsKey(temp.getClass())) {
                        throw new ExtensionException("AbstractUnitTest supports only one impl per feature type");
                    }
                    temp.init(managementContexts);
                    features.put(temp.getClass(), temp);
                    featuresByTypeRef.put(temp.getExtensionType(), temp);
                }
            }
        }

        public void destroy() {
            features.clear();
        }

        public Provider getProvider() {
            return null;
        }

        public List<Feature> getFeatures() {
            return new ArrayList<Feature>();
        }

        public void updateFeatureConfiguration(Feature feature, Element configuration) {
        }

        public Feature getFeature(String featureType) {
            return featuresByTypeRef.get(featureType);
        }

        public TransportFeature getTransportFeature(Class<? extends TransportProtocolRT> arg0) {
            return null;
        }

        @Override
        public <T extends Feature> T getFeature(Class<T> featureType) {
            return featureType.cast(features.get(featureType));
        }

        private static class ServerProviderImpl implements Server.Provider {

            public void spi_setConsumer(Extension consumer) {
            }

        }

    }

    protected static class ContainerImpl extends Container {

        public static final String TYPE = "http://test/container";
        private Map<String, Connector> connectors = new HashMap<String, Connector>();
        private Map<String, Connector> defaultConnectors = new HashMap<String, Connector>();

        private static ContainerImpl singleton;

        public static Container getInstance() {
            if (singleton == null) {
                singleton = new ContainerImpl(ContainerFactoryImpl.getInstance(), new ContainerProviderImpl(), null, null);
            }
            return singleton;
        }

        private ContainerImpl(ExtensionFactory factory, Provider provider, File extensionEtcDir, File extensionVarDir) {
            super(factory, provider, extensionEtcDir, extensionVarDir);
        }

        public Connector getConnector(String name) {
            return connectors.get(name);
        }

        public Set<Connector> getConnectors() {
            return new HashSet<Connector>(connectors.values());
        }

        public Connector getDefaultConnector(String typeRef) {
            return defaultConnectors.get(typeRef);
        }
        
        public Provider getProvider() {
            return null;
        }

        public void addConnector(Connector connector) {
            connectors.put(connector.getConnectorName(), connector);
            defaultConnectors.put(connector.getExtensionType(), connector);
        }

        public void destroy() {
            connectors.clear();
            defaultConnectors.clear();
        }

        public void init(Set<ManagementContext> managementContexts) throws ExtensionException {
        }

        private static class ContainerProviderImpl implements Container.Provider {

            public void spi_setConsumer(Extension consumer) {
            }

        }

        @Override
        public Connector getSystemDataSource() {
            return null;
        }
    }

    private static class ServerFactoryImpl extends ServerFactory {

        private static ServerFactory singleton = new ServerFactoryImpl();

        public static ServerFactory getInstance() {
            return singleton;
        }

        @Override
        public Server createServer(DOMSource domSource) throws ServerException {
            return ServerImpl.getInstance();
        }

        @Override
        public String getExtensionType() {
            return ServerImpl.TYPE;
        }

    }

    private static class ContainerFactoryImpl extends ContainerFactory {

        private static ContainerFactory singleton = new ContainerFactoryImpl();

        public static ContainerFactory getInstance() {
            return singleton;
        }

        @Override
        public String getExtensionType() {
            return ContainerImpl.TYPE;
        }

        @Override
        public Container createContainer(DOMSource domSource) throws ContainerException {
            return ContainerImpl.getInstance();
        }

    }

    public MessageFactory getDefaultMessageFactory() {
        return null;
    }

    public Set<ManagementContextFactory> getManagementContextFactories() {
        return new HashSet<ManagementContextFactory>();
    }

    public Set<ManagementContext> getManagementContexts() {
        return new HashSet<ManagementContext>();
    }

}
