/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.jms.ConnectionFactory;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.management.ObjectName;
import javax.sql.DataSource;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMOutputFormat;
import org.apache.axiom.om.impl.dom.DocumentImpl;
import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.apache.commons.io.IOUtils;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.ContextInfo;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.bluestemsoftware.open.eoa.commons.util.URIUtils;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.mgmt.Binding;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.Constants;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.FileUtils;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.HTTPFaultContext;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.HTTPRequestEntity;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeModule;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MessageReader;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MessageWriter;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MethodUtil;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.Pool;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil.InstanceData;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.DAOException;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.MessageDAO;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.MessageModuleDAO;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms.JMSReceiver;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms.JMSRequestor;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms.JMSRequestorFactory;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms.JMSSender;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.binding.BindingAction;
import org.bluestemsoftware.specification.eoa.component.binding.BindingFault;
import org.bluestemsoftware.specification.eoa.component.binding.BindingFaultReference;
import org.bluestemsoftware.specification.eoa.component.binding.BindingOperation;
import org.bluestemsoftware.specification.eoa.component.binding.rt.AcceptedResponseError;
import org.bluestemsoftware.specification.eoa.component.binding.rt.BindingFaultRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceFault;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceFaultReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference.ContentModel;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.MEP;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.BusinessFault;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.FaultContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageModule;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext.MessageModules;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.message.MessagePart;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentFactory;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.bluestemsoftware.specification.eoa.component.service.BindingReference;
import org.bluestemsoftware.specification.eoa.component.service.EndpointOperation;
import org.bluestemsoftware.specification.eoa.component.service.Service;
import org.bluestemsoftware.specification.eoa.ext.ExecutableException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBinding;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBindingException;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBindingMessageReference;
import org.bluestemsoftware.specification.eoa.ext.binding.http.XMLHTTPProtocol;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingConfiguration;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPReceiverFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPSenderFault;
import org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest.Method;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest.RequestEntity;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerResponse;
import org.bluestemsoftware.specification.eoa.ext.feature.jms.server.JMSServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAMessageModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA.WSA10;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.appdata.ApplicationDataModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.appdata.soap.ApplicationDataFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPHeader;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPLocationInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPMethod;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HyperTextTransferProtocol;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationJSON;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.container.Container;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class HTTPBindingImpl implements HTTPBindingRT.Provider {

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPBinding.class);

    protected HTTPBindingRT consumer;
    protected HTTPBindingConfiguration configuration;
    protected HTTPBinding binding;
    protected XMLHTTPProtocol underlyingProtocol;
    protected ExtensionFactoryContext factoryContext;
    protected MessageDAO messageDAO;
    protected MessageModuleDAO messageModuleDAO;
    protected Map<String, EndpointReference> endpointReferences;
    protected Queue receiveQueue;
    protected File dataDirectory;
    protected JMSRequestorFactory jmsRequestorFactory;
    protected Set<JMSReceiver> jmsReceivers;
    protected Pool<JMSRequestor> jmsRequestors;
    protected Pool<JMSSender> jmsSenders;

    private boolean isSuspended = true;
    private QueueConnection receiveConnection;
    private QueueConnection sendConnection;
    private ObjectName objectName;
    private JMXServerFeature jmxServerFeature;
    private String hostAddress;
    private File tempDirectory;
    private Map<String, Map<String, String>> httpHeaders;

    public HTTPBindingImpl(HTTPBindingConfiguration configuration) {
        this.configuration = configuration;
        this.endpointReferences = new HashMap<String, EndpointReference>();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.rt.UnderlyingProtocolRT$Provider#spi_init()
     */
    public void spi_init() throws HTTPBindingException {

        log.debug("init begin");

        // retrieve static binding component which has same name as runtime
        // binding provider

        binding = (HTTPBinding)consumer.getRuntimeProvidable();
        underlyingProtocol = (XMLHTTPProtocol)binding.getUnderlyingProtocol();

        // initialize dao objects. each will verify that underlying tables exist
        // within default data source, if not, they log a warning and create them.
        // note that all binding instances of this type will work off the same
        // table, implying an AtMostOnce delivery assurance, i.e. without ws-rm

        String connectorName = configuration.getDataSourceConnectorName();
        Container container = SystemContext.getContext().getSystem().getContainer();
        DataSource ds = null;
        try {
            ds = (DataSource)container.getConnector(connectorName);
        } catch (ClassCastException ce) {
            throw new HTTPBindingException("Connector '"
                    + connectorName
                    + "' specified within configuration is not an instance of "
                    + DataSourceConnector.class.getName());
        }

        if (ds == null) {
            throw new HTTPBindingException("Connector '"
                    + connectorName
                    + "' specified within configuration is undefined");
        }

        messageDAO = MessageDAO.getInstance(ds).init();
        messageModuleDAO = MessageModuleDAO.getInstance(ds).init();

        // TODO: we need to define a parameter that will allow us to archive
        // the message modules table (message table is insert only, i.e. no
        // performance penalty for size), i.e. archive messages older than
        // x days. i don't think deleting completed meps is appropriate b/c
        // for outgoing messages, the engine cannot log modules via an app
        // module. we're the only one that can do that

        // note that jms server feature is a required feature, i.e. we require
        // that it is embedded. this allows us to stream message content to a
        // file rather than reading entire message into memory (limitation of
        // jms api). and it insures that sender and receiver are both on same
        // machine, i.e. have access to same file system. because connection
        // is not managed, i.e. no transaction or pooling support, we can
        // retrieve directly from feature

        Server server = SystemContext.getContext().getSystem().getServer();
        JMSServerFeature jmsServer = server.getFeature(JMSServerFeature.class);
        ConnectionFactory connFactory = jmsServer.getEmbeddedConnectionFactory(false);

        try {
            receiveConnection = (QueueConnection)connFactory.createConnection();
        } catch (JMSException je) {
            throw new HTTPBindingException(je);
        }

        // construct a unique queue name. we'll use fragment id, because
        // binding qname may clash with other component types defined w/in
        // same namespace

        String queueName = null;
        try {
            URI temp = new URI(binding.getFragmentIdentifier().toString());
            queueName = URIUtils.toFile(temp, null).getName();
            queueName = queueName.replace(".", "_"); // else activemq hangs
            receiveQueue = jmsServer.createQueue(queueName);
        } catch (Exception ex) {
            throw new HTTPBindingException("Error creating receive queue. " + ex);
        }

        // create data directory which we will use to stream message content.
        // this prevents us from having to buffer entire message to memory,
        // i.e. jms api doesn't support handing off inputstream/outpstream

        try {
            dataDirectory = getDataDirectory();
        } catch (Exception ex) {
            throw new HTTPBindingException("Error creating data directory. " + ex);
        }

        // set up our cache of jms objects. note that all collections are
        // constrained by same size parm, i.e. because http binding does
        // not support async messaging, loose coupling between requesting
        // and responding node is not possible

        int maxPayloadProcessors = configuration.getMaxPayloadProcessors();

        try {
            jmsRequestorFactory = new JMSRequestorFactory(receiveConnection);
            jmsRequestors = new Pool<JMSRequestor>();
            for (int i = 0; i < maxPayloadProcessors; i++) {
                jmsRequestors.addObject((JMSRequestor)jmsRequestorFactory.makeObject());
            }
        } catch (Exception ex) {
            throw new HTTPBindingException(ex);
        }

        try {
            sendConnection = (QueueConnection)connFactory.createConnection();
            jmsSenders = new Pool<JMSSender>();
            for (int i = 0; i < maxPayloadProcessors; i++) {
                jmsSenders.addObject(new JMSSender(sendConnection));
            }
            sendConnection.start();
        } catch (Exception ex) {
            throw new HTTPBindingException(ex);
        }

        try {
            jmsReceivers = new HashSet<JMSReceiver>(maxPayloadProcessors);
            for (int i = 0; i < maxPayloadProcessors; i++) {
                jmsReceivers.add(new JMSReceiver(this, receiveConnection, receiveQueue, dataDirectory));
            }
        } catch (Exception ex) {
            throw new HTTPBindingException(ex);
        }

        // if jmx server feature is enabled, expose management methods defined
        // on our mbean interface

        jmxServerFeature = server.getFeature(JMXServerFeature.class);
        if (jmxServerFeature != null) {
            String name = HTTPBinding.class.getSimpleName();
            try {
                objectName = jmxServerFeature.registerMBean(consumer, new Binding(this), null, name);
            } catch (JMXServerFeatureException je) {
                throw new HTTPBindingException(je);
            }
        }

        // derive local ip address which may be used in place of machine
        // name and/or localhost when addressing a message

        try {
            hostAddress = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException ue) {
            throw new HTTPBindingException(ue);
        }

        tempDirectory = SystemContext.getContext().getSystem().getSystemTmpDir();

        // cache values for required http headers and optionally defined
        // header values

        httpHeaders = new HashMap<String, Map<String, String>>();
        if (!binding.isReusable()) {
            Interface intrface = binding.getInterfaceReference().getReferencedComponent();
            for (InterfaceAction ia : intrface.getInterfaceActions()) {
                BindingAction ba = binding.getBindingAction(ia.getAction());
                if (ba == null) {
                    continue;
                } else {
                    cacheRequiredHTTPHeaders(ia.getAction(), ba);
                }
            }
        }

        log.debug("init end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.binding.http.XMLHTTPProtocol$Provider#spi_receiveAction(HTTPServerRequest,
     *      HTTPServerResponse,
     *      org.bluestemsoftware.specification.eoa.component.engine.EndpointReference)
     */
    public void spi_receiveAction(HTTPServerRequest request, HTTPServerResponse response, Direction direction, EndpointReference epr) {

        log.debug("receiveAction begin");

        // the goal of the logic within this method is to 'accept' the message for
        // processing, see javadoc on XMLHTTPBindingRT for a detailed description of
        // rules which define the concept

        MessageModules requestModules = null;
        WSAMessageModule wsammIn = null;
        OMElement payload = null;

        try {

            // evaluate method value in order of most commonly used to
            // avoid unnecessary comparisons

            String method = request.getMethod();
            boolean hasRequestBody = false;
            if (method.equals("POST")) {
                hasRequestBody = true;
            } else if (method.equals("GET")) {
                hasRequestBody = false;
            } else if (method.equals("PUT")) {
                hasRequestBody = true;
            } else if (method.equals("DELETE")) {
                hasRequestBody = false;
            } else {
                response.setStatus(Constants.SC_METHOD_NOT_ALLOWED);
                response.trace();
                return;
            }

            // extract media type and encoding from content type header

            String mediaType = null;
            String encoding = null;
            String ctype = request.getContentType();
            ctype = ctype == null ? "" : ctype;
            encoding = getCharacterSetEncoding(ctype);
            int index = ctype.indexOf(';');
            if (index > 0) {
                mediaType = ctype.substring(0, index);
            } else {
                mediaType = ctype;
            }

            // if method has no request body, then the implied media type
            // is url encoded, i.e. content type header should be undefined

            if (mediaType.equals("")) {
                if (hasRequestBody) {
                    mediaType = Constants.MEDIA_TYPE_DEFAULT; // unsupported
                } else {
                    mediaType = Constants.MEDIA_TYPE_X_WWW_FORM;
                }
            }

            MediaTypeUtil mediaTypeUtil = MediaTypeUtil.getMediaTypeUtil(mediaType);

            if (mediaTypeUtil == null) {
                if (log.isDebugEnabled()) {
                    log.debug("unsupported media type " + mediaType);
                }
                response.setStatus(Constants.SC_UNSUPPORTED_MEDIA_TYPE);
                response.trace();
                return;
            }

            // check if content is encoded using gzip, which is currently the
            // only transport encoding we support

            boolean isGZIP = false;
            String ce = request.getHeader(Constants.HEADER_CONTENT_ENCODING);
            if (ce != null) {
                if (ce.equalsIgnoreCase("gzip")) {
                    isGZIP = true;
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Content-Encoding '" + ce + "' not supported.");
                    }
                    response.setStatus(Constants.SC_BAD_REQUEST);
                    response.trace();
                    return;
                }
            }

            if (mediaTypeUtil.requiresRequestBody() && hasRequestBody == false) {
                if (log.isDebugEnabled()) {
                    log.debug("method " + method + " inconsistent with media type " + mediaType);
                }
                response.setStatus(Constants.SC_METHOD_NOT_ALLOWED);
                response.trace();
                return;
            }

            // relativize request url against epr address. the resulting
            // value should match a declared binding operation. note that
            // uri api requires scheme and authority to relativize, but
            // all we care about are the paths

            URI eprAddress = new URI(epr.getEndpoint().getAddress());
            String host = eprAddress.getHost();
            int port = eprAddress.getPort();
            String path = request.getRequestURI();
            URI requestURI = new URI("http", null, host, port, path, null, null);
            String location = eprAddress.relativize(requestURI).toString();

            HTTPLocationInfo locationInfo = underlyingProtocol.getHTTPLocationInfo(location);

            if (locationInfo == null) {
                if (log.isDebugEnabled()) {
                    log.debug("location '"
                            + location
                            + "' relativized from requestURI '"
                            + request.getRequestURI()
                            + "' fails to map to a binding operation.");
                }
                response.setStatus(Constants.SC_NOT_FOUND);
                response.trace();
                return;
            }

            // retrieve settings which constrain way in which instance data can
            // be sent. note that because this is 'my' service, we know it's a
            // request

            QName operationName = locationInfo.getOperationName();
            String separator = underlyingProtocol.getHTTPQueryParameterSeparator(operationName);
            separator = separator == null ? "&" : separator;
            EndpointOperation eo = epr.getEndpoint().getEndpointOperation(operationName);
            InterfaceMessageReference im = eo.getInterfaceOperation().getInputMessageReference();
            Map<String, Style> styles = im.getApplicableStyles();

            // now that we have the operation metadata, compare the actual
            // method used with the expected method

            HTTPMethod expectedMethod = underlyingProtocol.getMethod(operationName);
            if (expectedMethod == null) {
                expectedMethod = underlyingProtocol.getMethodDefault();
                if (expectedMethod == null) {
                    if (eo.getInterfaceOperation().isSafe()) {
                        expectedMethod = HTTPMethod.GET;
                    } else {
                        expectedMethod = HTTPMethod.POST;
                    }
                }
            }

            if (!expectedMethod.toString().equals(method)) {
                if (log.isDebugEnabled()) {
                    log.debug("request addressed to location '"
                            + location
                            + "' used incorrect http method '"
                            + method
                            + "'. metadata for operation '"
                            + operationName
                            + "' indicates httpMethod '"
                            + expectedMethod
                            + "' is expected.");
                }
                response.setStatus(Constants.SC_METHOD_NOT_ALLOWED);
                response.trace();
                return;
            }

            // because message has no headers, we must generate implied values for
            // ws-addressing module

            wsammIn = new WSAMessageModule(im.getAction());
            wsammIn.setTo(epr.getEndpoint().getAddress());
            wsammIn.setReplyTo(WSA10.DEFAULT_REPLY_TO);
            requestModules = new MessageModules(wsammIn);

            // if user accepts json as a response serialization, we must persist
            // this as a message module, i.e. because response is processed on
            // a different thread. we perform poor man's content type negotiation,
            // i.e. we do not examine q prefs, etc. we check for the various
            // supported json media types. we ignore other media types and default
            // to application/xml if no accept header defined or no match found

            String acceptHeader = request.getHeader(Constants.HEADER_ACCEPT);
            if (acceptHeader != null) {
                String responseMediaType = null;
                if (acceptHeader.contains(Constants.MEDIA_TYPE_APPLICATION_JSON_BADGER)) {
                    responseMediaType = Constants.MEDIA_TYPE_APPLICATION_JSON_BADGER;
                } else if (acceptHeader.contains(Constants.MEDIA_TYPE_APPLICATION_JSON)) {
                    responseMediaType = Constants.MEDIA_TYPE_APPLICATION_JSON;
                } else if (acceptHeader.contains(Constants.MEDIA_TYPE_TEXT_JAVASCRIPT)) {
                    responseMediaType = Constants.MEDIA_TYPE_TEXT_JAVASCRIPT;
                } else if (acceptHeader.contains(Constants.MEDIA_TYPE_TEXT_JSON)) {
                    responseMediaType = Constants.MEDIA_TYPE_TEXT_JSON;
                }
                if (responseMediaType != null) {
                    MediaTypeModule mtm = new MediaTypeModule(responseMediaType);
                    requestModules.put(MediaTypeModule.class, mtm);
                }
            }

            // now we parse data, i.e. payload from request uri and/or the
            // request body

            payload = mediaTypeUtil.readData(request, styles, location, separator, encoding, isGZIP);

        } catch (Throwable t) {

            // if we end up here, this implies that message was NOT 'accepted' for
            // processing. the status code MUST be set to 4xx and the response body
            // must be empty

            String statusReason = null;

            if (!(t instanceof HTTPFault)) {
                statusReason = "Unable to read request";
                log.error("caught throwable while receiving action addressed to epr "
                        + epr.getFragmentIdentifier()
                        + ". "
                        + t);
            } else {
                statusReason = ((HTTPFault)t).getFaultReason();
                if (log.isDebugEnabled()) {
                    log.debug("caught http fault while receiving action addressed to epr "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + ((HTTPFault)t).toString());
                }
            }

            try {
                response.sendError(Constants.SC_BAD_REQUEST, statusReason);
                response.trace();
            } catch (IOException ie) {
                log.error("error sending http error response. " + ie);
            }

            return;

        }

        // if we make it this far, we have, according to MEP state model,
        // successfully transitioned from the 'init' state to the 'receiving'
        // state, i.e. the message has been 'accepted for processing.' b/c
        // http binding does not support async messaging, we know that this
        // action represents a request addressed to 'my' service

        receiveMyServiceRequest(request, response, epr, requestModules, payload);
        response.trace();

        log.debug("receiveAction end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingRT$Provider#spi_sendAction(org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference,
     *      org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext)
     */
    public void spi_sendAction(EndpointReference epr, ActionContext actionContext) throws SystemFault {
        log.debug("sendAction begin");

        ApplicationRT app = epr.getRootComponent().getApplication();
        EndpointActionReference ear = epr.getEndpointActionReference(actionContext.getAction());

        Direction direction = null;
        if (ear == null) {
            if (actionContext.getAction().equals(SystemFault.ACTION)) {
                direction = Direction.OUT;
            } else {
                throw new SystemFault(app, "Action '"
                        + actionContext.getAction()
                        + "' is undefined on referenced endpoint "
                        + epr.getEndpoint().getFragmentIdentifier());
            }
        } else {
            direction = ear.getReferencedComponent().getInterfaceAction().getDirection();
        }

        try {

            if (direction == Direction.OUT) {
                sendMyServiceResponse(epr, actionContext);
            } else {
                sendPartnerServiceRequest(epr, (MessageContext)actionContext);
            }

        } catch (Throwable th) {

            // if we end up here, the implication is that message was not accepted
            // by receiver. we translate into a system fault and throw back to
            // engine and allow it to handle it

            SystemFault systemFault = null;

            if (th instanceof SystemFault) {
                if (log.isDebugEnabled()) {
                    log.debug("caught system fault while sending action "
                            + actionContext.getAction()
                            + " defined on epr "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + ((SystemFault)th).getFaultReason());
                }
                systemFault = (SystemFault)th;
            } else if (th instanceof BindingFaultRT) {
                if (log.isDebugEnabled()) {
                    log.debug("caught binding fault while sending action "
                            + actionContext.getAction()
                            + " defined on epr "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + ((BindingFaultRT)th).getFaultReason());
                }
                QName faultCode = ((BindingFaultRT)th).getFaultName();
                systemFault = new SystemFault(app, faultCode, ((BindingFaultRT)th).getFaultReason());
            } else {
                log.error("caught throwable while sending action "
                        + actionContext.getAction()
                        + " defined on epr "
                        + epr.getFragmentIdentifier()
                        + ". "
                        + th);
                systemFault = new SystemFault(app, "Caught throwable sending action.", th);
            }

            throw systemFault;

        }

        log.debug("sendAction end");
    }

    /*
     * action represents a request addressed to a service hosted by target engine. message
     * direction is in and all supported meps are valid
     */
    private void receiveMyServiceRequest(HTTPServerRequest request, HTTPServerResponse response, EndpointReference epr, MessageModules requestModules, OMElement payload) {

        log.debug("receiveMyServiceRequest begin");

        // note that invocation of this method implies that we have 'accepted'
        // the request for processing

        EndpointActionReference ear = null;
        EndpointOperationReference eor = null;
        WSAMessageModule wsammIn = null;
        MEP mep = null;
        JMSRequestor jmsRequestor = null;
        org.bluestemsoftware.specification.wsa.EndpointReference replyToEPR = null;

        try {

            wsammIn = requestModules.getModule(WSAMessageModule.class);
            replyToEPR = wsammIn.getReplyTo();

            // register message. we know message id is unique, because we
            // generated it, i.e. duplicate requests cannot be prevented

            messageDAO.insertMessage(wsammIn.getMessageID(), true);
            String action = requestModules.getAction();
            ear = epr.getEndpointActionReference(action);
            eor = (EndpointOperationReference)ear.getParent();
            mep = eor.getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();

            // verify that if binding config contains abstract http header decls
            // marked as required, that they exist and if user has defined
            // a value within provider configuration, verify it matches

            checkRequiredHTTPHeaders(action, null, request);

            // serialize message content to a file to avoid buffering entire
            // message to memory which is limitation of jms api. we use
            // message id as file name, which will be used by jms receiver
            // to retrieve from file system

            String fileName = FileUtils.createFileName(wsammIn.getMessageID());
            File requestContent = new File(dataDirectory, fileName);

            // if payload is null, i.e. it's an empty request. we create an
            // empty file, i.e. which will allow user to manage the request
            // files within the binding var dir, i.e. browsing/deleting,
            // etc ... if jms receiver can't find file, it quietly logs a
            // debug message

            if (payload != null) {
                OutputStream out = null;
                XMLStreamWriter writer = null;
                try {
                    out = new FileOutputStream(requestContent);
                    writer = STAXUtils.createXMLStreamWriter(out, Constants.UTF_8);
                    payload.buildNext(); // kids won't serialize otherwise
                    payload.serializeAndConsume(writer);
                } finally {
                    if (out != null) {
                        out.flush();
                        IOUtils.closeQuietly(out);
                    }
                    if (writer != null) {
                        writer.close();
                    }
                }
            } else {
                requestContent.createNewFile();
            }

            // overwrite anonymous replyTo with name of temporary queue used
            // by requestor before persisting the module to dbms, which will
            // be used to return response to requestor

            jmsRequestor = jmsRequestors.getObject();
            String replyTo = "jms:" + jmsRequestor.getTemporaryQueue().getQueueName();
            org.bluestemsoftware.specification.wsa.EndpointReference temp;
            temp = new org.bluestemsoftware.specification.wsa.EndpointReference(replyTo, null);
            wsammIn.setReplyTo(temp);

            // persist message modules to dbms before passing execution off
            // to a separate thread

            for (MessageModule mm : requestModules.values()) {
                messageModuleDAO.insertMessageModule(wsammIn.getMessageID(), true, mm);
            }

            // write message to queue where it will be picked up by jms
            // receiver on separate thread and processed by read message
            // method on super

            JMSSender jmsSender = null;
            try {
                jmsSender = jmsSenders.getObject();
                javax.jms.Message msg = jmsSender.createMessage();
                msg.setJMSCorrelationID(wsammIn.getMessageID());
                msg.setStringProperty(Constants.MESSAGE_ID_PROPERTY, wsammIn.getMessageID());
                msg.setBooleanProperty(Constants.MY_SERVICE_PROPERTY, true);
                jmsSender.send(msg, receiveQueue, 0, DeliveryMode.PERSISTENT);
            } finally {
                jmsSenders.addObject(jmsSender);
            }

            // if mep is in only, we return an accepted response code as
            // required by eoa spec

            if (mep == MEP.IN_ONLY) {
                response.setStatus(Constants.SC_ACCEPTED);
                return;
            }

            // block for response using jmsRequestor. the response is plain xml.
            // note that content type is passed as a message property which
            // indicates char encoding used to serialize chars to bytes

            String jmsCorrellationID = wsammIn.getMessageID();
            int timeout = configuration.getRequestorTimeout(eor.getEndpointOperationName());
            long current = System.currentTimeMillis();
            javax.jms.Message msg = jmsRequestor.receive(jmsCorrellationID, timeout);
            long waited = System.currentTimeMillis() - current;

            // if we timed out waiting for response and mep is in-out, throw
            // a fault so that client will receive notification. if mep is
            // robust in-only and we timeout, we return an accepted code, i.e.
            // we assume no fault response is forthcoming (if it is, it is
            // silently discarded by jms provider). note that although the
            // spec requires a 204 status code when no fault is returned and
            // mep is robust-in-only, as of version 1.3, axis writes a
            // transport error to the log 'unable to sendViaXXX'

            if (waited >= timeout) {
                if (mep == MEP.IN_OUT) {
                    throw new HTTPReceiverFault("Binding timed-out waiting for response.");
                } else {
                    response.setStatus(Constants.SC_NO_CONTENT);
                    return;
                }
            }

            boolean gzipResponse = msg.getBooleanProperty(Constants.GZIP_RESPONSE_PROPERTY);
            boolean isFaultResponse = msg.getBooleanProperty(Constants.IS_FAULT_RESPONSE_PROPERTY);
            String responseAction = msg.getStringProperty(Constants.ACTION_PROPERTY);

            // add user defined header values which are optionally defined within
            // binding configuration

            Map<String, String> userDefinedHeaders = httpHeaders.get(responseAction);
            if (userDefinedHeaders != null) {
                for (Map.Entry<String, String> entry : userDefinedHeaders.entrySet()) {
                    String name = entry.getKey();
                    String value = entry.getValue();
                    if (!response.containsHeader(name)) {
                        response.addHeader(name, value);
                    } else {
                        log.debug("not overwriting existing http header '"
                                + name
                                + "' with user defined value '"
                                + value
                                + "'.");
                    }
                }
            }

            // the wsdl 2.0 spec requires that all responses where content
            // model defined on abstract message ref is #any or #element
            // use application/xml as response serialization. we bend the
            // rules a bit here and allow response to be serialized as
            // a json string if so indicated within accept header

            MediaTypeModule mtm = requestModules.getModule(MediaTypeModule.class);
            if (mtm == null) {
                response.setContentType(Constants.APPLICATION_XML_CONTENT_TYPE);
            } else {
                response.setContentType(MediaTypeUtil.getContentType(mtm.getMediaType()));
            }

            // set the response status. note that if response contains a
            // fault, spec requires that we set status to 500. in-out
            // requires a 200 status code

            if (isFaultResponse) {
                response.setStatus(Constants.SC_INTERNAL_SERVER_ERROR);
            } else {
                if (mep == MEP.ROBUST_IN_ONLY) {
                    throw new HTTPReceiverFault("Normal response not allowed with robust-in-only MEP.");
                } else {
                    response.setStatus(Constants.SC_OK);
                }
            }

            String responseFileName = msg.getStringProperty(Constants.FILE_NAME_PROPERTY);
            File responseMessageContent = new File(responseFileName);

            OutputStream out = response.getOutputStream();
            if (gzipResponse) {
                response.addHeader(Constants.HEADER_CONTENT_ENCODING, Constants.COMPRESSION_GZIP);
                out = new GZIPOutputStream(response.getOutputStream());
            }

            InputStream in = null;
            try {
                in = new FileInputStream(responseMessageContent);
                IOUtils.copy(in, out);
            } finally {
                IOUtils.closeQuietly(in);
                if (!responseMessageContent.delete()) {
                    log.error("error deleting processed response "
                            + responseMessageContent.getAbsolutePath()
                            + " from file system");
                }
            }

        } catch (Throwable th) {

            HTTPFault hf = null;

            if (th instanceof HTTPFault) {
                hf = (HTTPFault)th;
                if (log.isDebugEnabled()) {
                    log.debug("caught http fault while receiving request action "
                            + wsammIn.getAction()
                            + " defined on endpoint referenced by component "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + ((HTTPFault)th).getFaultReason());
                }
            } else {
                if (configuration.isEmbeddStackTraceInFaults()) {
                    hf = new HTTPReceiverFault(th);
                } else {
                    hf = new HTTPReceiverFault(th.getMessage());
                }
                log.error("caught throwable while receiving request action "
                        + wsammIn.getAction()
                        + " defined on endpoint referenced by component "
                        + epr.getFragmentIdentifier()
                        + ". "
                        + th);
            }

            ear = epr.getEndpointActionReference(wsammIn.getAction());

            if (ear != null) {
                eor = (EndpointOperationReference)ear.getParent();
                mep = eor.getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();
            }

            // eoa spec requires an empty body in response to one way mep,
            // with an accepted code, even if we failed to process the request

            if (mep != null && mep == MEP.IN_ONLY) {
                log.debug("MEP is in-only. No fault response allowed.");
                response.setStatus(Constants.SC_ACCEPTED);
                return;
            }

            // set replyTo epr back to anonymous uri so sendMyServiceResponse
            // will not attempt to return response to requestor via jms

            wsammIn.setReplyTo(replyToEPR);

            // run the binding fault response through handle response logic. http
            // response is passed by reference, i.e. binding fault response is
            // processed on same thread as request

            try {
                WSAMessageModule wsammOut = new WSAMessageModule(hf.getAction());
                wsammOut.setRelatesTo(wsammIn.getMessageID());
                MessageModules responseModules = new MessageModules(wsammOut);
                HTTPFaultContext hfc = new HTTPFaultContext(response, requestModules, responseModules, hf);
                sendMyServiceResponse(epr, hfc);
            } catch (Throwable thr) {
                log.error("caught throwable while processing http fault. " + thr);
                response.setStatus(Constants.SC_INTERNAL_SERVER_ERROR);
            }

        } finally {
            if (jmsRequestor != null) {
                try {
                    jmsRequestors.addObject(jmsRequestor);
                } catch (Exception fatchance) {
                }
            }
        }

        log.debug("receiveMyServiceRequest end");

    }

    /*
     * action represents a response being returned by a locally hosted service. message
     * direction is out and mep is in-out or robust in-only
     */
    private void sendMyServiceResponse(EndpointReference epr, ActionContext actionContext) throws Exception {

        log.debug("sendMyServiceResponse begin");

        // if action context is an instanceof HTTPFaultContext, then request
        // failed to enter application layer, in which case request modules
        // are passed by reference. otherwise look-up using relates to value

        ApplicationRT app = ((EngineRT)epr.getRootComponent()).getApplication();
        WSAMessageModule wsammOut = actionContext.getMessageModule(WSAMessageModule.class);
        String requestMessageID = wsammOut.getRelatesTo();
        MessageModules requestModules = null;
        ContentSerialization ser = null;
        String contentType = null;

        WSAMessageModule wsammIn = null;
        if (actionContext instanceof HTTPFaultContext) {
            requestModules = ((HTTPFaultContext)actionContext).getRequestModules();
        } else {
            requestModules = messageModuleDAO.selectMessageModules(requestMessageID, true);
            if (requestModules == null) {
                throw new SystemFault(app, "Failed to correllate request.");
            }
        }

        // the wsdl 2.0 spec requires that all responses where content
        // model defined on abstract message ref is #any or #element
        // use application/xml as response serialization. we bend the
        // rules a bit here and allow response to be serialized as
        // a json string if so indicated within accept header

        wsammIn = requestModules.getModule(WSAMessageModule.class);
        MediaTypeModule mtm = requestModules.getModule(MediaTypeModule.class);
        if (mtm == null) {
            ser = ContentSerializationXML.getInstance();
            contentType = Constants.APPLICATION_XML_CONTENT_TYPE;
        } else {
            ser = MediaTypeUtil.getContentSerialization(mtm.getMediaType());
            contentType = MediaTypeUtil.getContentType(mtm.getMediaType());
        }

        // attempt to persist the message. if id is not unique, a sql
        // exception is thrown

        try {
            messageDAO.insertMessage(wsammOut.getMessageID(), true);
        } catch (SQLIntegrityConstraintViolationException se) {
            throw new SystemFault(app, "Response messageID not unique.");
        }

        Content responsePayload = null;
        String responseAction = actionContext.getAction();
        boolean gzipResponse = false;
        boolean isFaultResponse = false;

        HyperTextTransferProtocol http = ((HyperTextTransferProtocol)binding.getUnderlyingProtocol());
        EndpointActionReference responseEAR = epr.getEndpointActionReference(responseAction);

        if (responseEAR != null) {

            EndpointOperationReference eor = (EndpointOperationReference)responseEAR.getParent();
            MEP mep = eor.getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();

            if (mep == MEP.IN_ONLY) {
                throw new SystemFault(app, "MEP is 'in-only'. No response is allowed.");
            }

            InterfaceAction ia = responseEAR.getReferencedComponent().getInterfaceAction();
            if (ia instanceof InterfaceFaultReference) {
                isFaultResponse = true;
                QName faultName = ((InterfaceFaultReference)ia).getFault().getName();
                String encodings = http.getFaultHTTPContentEncoding(faultName);
                if (encodings != null) {
                    gzipResponse = encodings.contains("gzip");
                }
                if (actionContext.getMessage() == null) {
                    throw new SystemFault(app, "Fault response " + faultName + " was empty.");
                }
                responsePayload = actionContext.getMessage().getContent();
                if (responsePayload == null) {
                    throw new SystemFault(app, "Fault response " + faultName + " was empty.");
                }
                responsePayload = importContent(responsePayload);
            } else {
                if (mep == MEP.ROBUST_IN_ONLY) {
                    throw new SystemFault(app, "MEP is 'robust-in-only'. Normal response not allowed.");
                }
                QName operationName = eor.getEndpointOperationName();
                String encodings = http.getMessageReferenceHTTPContentEncoding(operationName, Direction.OUT);
                if (encodings != null) {
                    gzipResponse = encodings.contains("gzip");
                }
                InterfaceMessage am = ((InterfaceMessageReference)ia).getReferencedComponent();
                responsePayload = MessageWriter.writeMessage(responseEAR, am, (MessageContext)actionContext);
            }

        } else {

            // the action is undeclared. we run through default set of binding
            // modules so that required response headers will be generated

            if (actionContext instanceof HTTPFaultContext) {

                // an http fault was generated before it made it into application
                // layer. convert to system fault schema instance

                HTTPFault hf = ((HTTPFaultContext)actionContext).getHTTPFault();
                SystemFault sf = new SystemFault(app, hf.getFaultName(), hf.getFaultReason());
                responsePayload = importContent(sf.toMessage().getContent());

            } else {

                // the only undeclared action we support is the system fault
                // action. throw it back to engine if that's not the case

                if (!actionContext.getAction().equals(SystemFault.ACTION)) {
                    throw new SystemFault(app, "Action " + actionContext.getAction() + " is undefined.");
                }

                // payload is an instance of system fault schema
                responsePayload = importContent(actionContext.getMessage().getContent());

            }

            isFaultResponse = true;

        }

        // persist the response modules to dbms
        for (MessageModule mm : actionContext.getMessageModules().values()) {
            messageModuleDAO.insertMessageModule(wsammOut.getMessageID(), true, mm);
        }

        // if replyTo is anonymous uri, then we are returning a response
        // generated by this binding, i.e. an http fault was thrown before
        // the request was written to receive queue. no jms requestor is
        // blocking. serialize the response to http outputstream

        String replyTo = wsammIn.getReplyTo().getAddress();

        if (replyTo.equals(WSA10.ANONYMOUS_URI)) {

            HTTPServerResponse response = ((HTTPFaultContext)actionContext).getResponse();
            response.setContentType(contentType);

            // when returning a fault response, the status code must be set
            // to 500 as required by eoa spec. note that as of jettison
            // version 1.0.1 we must serialize document, i.e. because it
            // flushes written content within end document event

            response.setStatus(Constants.SC_INTERNAL_SERVER_ERROR);

            XMLStreamWriter writer = null;
            try {
                OutputStream out = response.getOutputStream();
                writer = STAXUtils.createXMLStreamWriter(ser, out, Constants.UTF_8);
                if (ser instanceof ContentSerializationJSON) {
                    ContextInfo contextInfo = null;
                    try {
                        contextInfo = STAXUtils.adjustContext(responsePayload);
                        DocumentImpl temp = (DocumentImpl)responsePayload.getParentNode();
                        temp.internalSerializeAndConsume(writer);
                    } finally {
                        STAXUtils.restoreContext(contextInfo, responsePayload);
                    }
                } else {
                    ((OMElement)responsePayload).serializeAndConsume(writer);
                }
            } catch (Throwable th) {
                log.error("Error serializing binding fault response. " + th);
            } finally {
                if (writer != null) {
                    writer.close();
                }
            }

            return;

        }

        // retrieve a reference to temp queue by name. note that we use
        // this approach rather than dynamically creating using session
        // on sender. activemq overwrites the existing queue with a new
        // destination and registers with mbean, etc ... not what we want

        String tempQueueName = replyTo.substring(replyTo.indexOf(":") + 1);
        Queue temporaryQueue = jmsRequestorFactory.getTemporaryQueue(tempQueueName);

        // if the requestor timed out waiting for response, and cache was
        // soft, then requestor may have been garbage collected, in which
        // case, the temp queue is null

        if (temporaryQueue == null) {
            log.debug("temporary queue was destroyed. discarding response.");
            return;
        }

        // stream the response to a temporary file which gets deleted
        // automatically when jvm dies, i.e. no sense in preserving it
        // beyond the life of jvm because servlet response object will
        // no longer exist nor will the blocking jms requestor. note
        // that as of version 1.0.1 jettison requires a document for
        // ser (the end document event flushes the written content)

        File responseContent = null;
        OutputStream out = null;
        XMLStreamWriter writer = null;
        try {
            responseContent = File.createTempFile("eoa-", ".xml", tempDirectory);
            responseContent.deleteOnExit();
            out = new FileOutputStream(responseContent);
            writer = STAXUtils.createXMLStreamWriter(ser, out, Constants.UTF_8);
            if (ser instanceof ContentSerializationJSON) {
                ContextInfo contextInfo = null;
                try {
                    contextInfo = STAXUtils.adjustContext(responsePayload);
                    DocumentImpl temp = (DocumentImpl)responsePayload.getParentNode();
                    temp.serializeAndConsume(writer);
                } finally {
                    STAXUtils.restoreContext(contextInfo, responsePayload);
                }
            } else {
                ((OMElement)responsePayload).serializeAndConsume(writer);
            }
        } finally {
            if (out != null) {
                out.flush();
                IOUtils.closeQuietly(out);
            }
            if (writer != null) {
                writer.close();
            }
        }

        JMSSender jmsSender = null;
        try {
            jmsSender = (JMSSender)jmsSenders.getObject();
            javax.jms.Message msg = jmsSender.createMessage();
            msg.setStringProperty(Constants.FILE_NAME_PROPERTY, responseContent.getAbsolutePath());
            msg.setBooleanProperty(Constants.GZIP_RESPONSE_PROPERTY, gzipResponse);
            msg.setBooleanProperty(Constants.IS_FAULT_RESPONSE_PROPERTY, isFaultResponse);
            msg.setStringProperty(Constants.ACTION_PROPERTY, responseAction);
            msg.setJMSCorrelationID(wsammIn.getMessageID());
            jmsSender.send(msg, temporaryQueue, 0, DeliveryMode.NON_PERSISTENT);
        } finally {
            jmsSenders.addObject(jmsSender);
        }

        log.debug("sendMyServiceResponse end");

    }

    /*
     * action represents a request addressed to a partner service. message direction is in and
     * all supported meps are valid
     */
    private void sendPartnerServiceRequest(EndpointReference epr, MessageContext messageContext) throws SystemFault {

        log.debug("sendPartnerServiceRequest begin");

        // as required by spec, if an error occurs before we transition to
        // the 'sending+receiving' state, we throw a fault to spi_sendAction
        // method where it will be converted into a system fault and thrown
        // back to engine as an indication action was not accepted by partner

        ApplicationRT app = ((EngineRT)epr.getRootComponent()).getApplication();
        MessageModules requestModules = messageContext.getMessageModules();
        WSAMessageModule wsammIn = requestModules.getModule(WSAMessageModule.class);

        EndpointActionReference requestAction = epr.getEndpointActionReference(wsammIn.getAction());

        // because http binding uses plain old xml, i.e. no message
        // headers, we cannot support the application data feature.
        // this, unfortunately, is a leaky abstraction, i.e. an
        // application should not care which endpoint is used to
        // invoke a partner service. wsa spec, etc should allow msg
        // headers to be transmitted as transport headers, i.e. as
        // http headers

        if (messageContext.getMessageModule(ApplicationDataModule.class).getSize() > 0) {
            throw new SystemFault(app, "Binding type '"
                    + HTTPBinding.BINDING_TYPE
                    + "' cannot support the feature '"
                    + ApplicationDataFeature.TYPE
                    + "'.");
        }

        // attempt to persist the message. if id is not unique, a sql
        // exception is thrown

        try {
            messageDAO.insertMessage(wsammIn.getMessageID(), false);
        } catch (SQLIntegrityConstraintViolationException sve) {
            throw new SystemFault(app, "Request messageID " + wsammIn.getMessageID() + " not unique.");
        } catch (SQLException se) {
            throw new SystemFault(app, "Error persisting request. " + se);
        }

        // validate payload against expected type and content model. if invalid
        // a system fault is thrown

        Content requestPayload = writeMessage(epr, (MessageContext)messageContext);

        // set ws-addressing attribs on message module with implied properties,
        // i.e. which will not actually be serialized to message, but will be
        // used to handle response

        String to = epr.getEndpoint().getAddress();
        wsammIn.setTo(to);
        wsammIn.setReplyTo(WSA10.DEFAULT_REPLY_TO);

        // persist the request modules to the dbms
        for (MessageModule mm : requestModules.values()) {
            try {
                messageModuleDAO.insertMessageModule(wsammIn.getMessageID(), false, mm);
            } catch (Exception ex) {
                throw new SystemFault(app, "Error inserting message modules. " + ex);
            }
        }

        EndpointOperationReference eor = (EndpointOperationReference)requestAction.getParent();
        MEP mep = eor.getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();

        // determine if content is encoded by inspecting algorithms for
        // a supported encoding, i.e. just gzip currently

        QName operationName = eor.getEndpointOperationName();
        String encodings = underlyingProtocol.getMessageReferenceHTTPContentEncoding(operationName, Direction.IN);
        boolean gzipRequest = false;
        if (encodings != null) {
            gzipRequest = encodings.contains("gzip");
        }

        // if exchange indicates a response is warranted, inspect algorithms
        // defined on response action with message direction Out

        boolean gzipResponse = false;
        if (mep != MEP.IN_ONLY) {
            encodings = underlyingProtocol.getMessageReferenceHTTPContentEncoding(operationName, Direction.OUT);
            if (encodings != null) {
                gzipResponse = encodings.contains("gzip");
            }
        }

        // retreive http method used to send request. logic defined within
        // wsdl 20 spec applies

        HTTPMethod method = underlyingProtocol.getMethod(eor.getEndpointOperationName());
        InterfaceOperation io = eor.getEndpointOperation().getInterfaceOperation();
        if (method == null) {
            method = underlyingProtocol.getMethodDefault();
            if (method == null) {
                if (io.isSafe()) {
                    method = HTTPMethod.GET;
                } else {
                    method = HTTPMethod.POST;
                }
            }
        }

        // determine media type used to send request. we will use default
        // values defined by wsdl 2.0 spec which map to http method used
        // unless content is modeled using #other in which case it we
        // validated request has a boty and it's an alternate serialization
        // the we support, e.g. one of the json serializations

        Map<String, String> requestHeaders = new HashMap<String, String>();
        MediaTypeUtil mt = null;
        String ct = null;
        boolean hasRequestBody = false;
        if (method == HTTPMethod.POST || method == HTTPMethod.PUT) {
            ContentModel cm = io.getInputMessageReference().getContentModel();
            if (cm == ContentModel.OTHER) {
                String ser = underlyingProtocol.getInputSerialization(io.getName());
                if (ser == null) {
                    ser = Constants.MEDIA_TYPE_APPLICATION_XML;
                }
                ct = MediaTypeUtil.getContentType(ser);
                mt = MediaTypeUtil.getMediaTypeUtil(ser);
            } else {
                ct = Constants.APPLICATION_XML_CONTENT_TYPE;
                mt = MediaTypeUtil.getMediaTypeUtil(Constants.MEDIA_TYPE_APPLICATION_XML);
            }
            requestHeaders.put(Constants.HEADER_CONTENT_TYPE, ct);
            hasRequestBody = true;
        } else {
            ct = Constants.X_WWW_FORM_CONTENT_TYPE;
            mt = MediaTypeUtil.getMediaTypeUtil(Constants.MEDIA_TYPE_X_WWW_FORM);
            requestHeaders.put(Constants.HEADER_CONTENT_TYPE, ct);
            hasRequestBody = false;
        }

        // retrieve values which constrain underlying protocol and which will
        // be used by util to format instance data.

        Map<String, Style> styles = io.getInputMessageReference().getApplicableStyles();
        String address = epr.getEndpoint().getAddress();
        String sep = underlyingProtocol.getHTTPQueryParameterSeparator(operationName);
        String location = underlyingProtocol.getHTTPLocationInfo(operationName).getLocation();
        boolean iu = underlyingProtocol.getHTTPLocationInfo(operationName).isIgnoreUncited();

        // using media type util, format request payload and/or request uri
        // according to rules defined within wsdl 2.0 spec

        InstanceData id = null;
        try {
            id = mt.writeData(requestPayload, styles, address, location, sep, iu);
        } catch (Exception ex) {
            throw new SystemFault(app, "Error writing instance data." + ex.getMessage());
        }

        // add http headers which indicate encoding employed on request, if
        // any, and encoding supported on response, if any

        if (gzipRequest) {
            requestHeaders.put(Constants.HEADER_CONTENT_ENCODING, Constants.COMPRESSION_GZIP);
        }

        if (gzipResponse) {
            requestHeaders.put(Constants.HEADER_ACCEPT_ENCODING, Constants.COMPRESSION_GZIP);
        }

        // add user defined header values which are optionaly defined within
        // binding configuration

        Map<String, String> userDefinedHeaders = httpHeaders.get(wsammIn.getAction());
        if (userDefinedHeaders != null) {
            for (Map.Entry<String, String> entry : userDefinedHeaders.entrySet()) {
                String name = entry.getKey();
                String value = entry.getValue();
                if (requestHeaders.get(name) == null) {
                    requestHeaders.put(name, value);
                } else {
                    log.debug("not overwriting existing http header '"
                            + name
                            + "' with user defined value '"
                            + value
                            + "'.");
                }
            }
        }

        // if request has no body, we write payload as uri query string. else
        // query string is written to body

        String requestURI = id.getRequestURI().toString();

        String queryString = null;
        if (hasRequestBody == false) {
            queryString = (String)id.getPayload();
            requestURI = requestURI + "?" + queryString;
        }

        // attempt to send the request. if module is unable to send, e.g. due
        // to a communication error, etc ..., it throws a transport fault, which
        // we propogate back to engine as a system fault. note that timeout is
        // configured via options on client

        ClassLoader cloader = factoryContext.getClassLoader();
        RequestEntity re = null;
        if (hasRequestBody) {
            OMOutputFormat omOutputFormat = new OMOutputFormat();
            omOutputFormat.setCharSetEncoding("UTF-8");
            re = new HTTPRequestEntity(cloader, id.getPayload(), omOutputFormat, gzipRequest);
        }
        Method transportMethod = MethodUtil.valueOf(method);
        HTTPClientRequest request = new HTTPClientRequest(requestURI, transportMethod, requestHeaders, re);
        HTTPClientResponse clientResponse = null;

        try {

            // note that we set handleRedirects to true which implies we do not
            // have to worry about handling 3xx codes, i.e. if redirects are not
            // successful, module will throw a transport fault, i.e. which will
            // be converted to a system fault and re-thrown to engine within
            // spi_sendAction method

            HTTPTransportModule tm = (HTTPTransportModule)requestAction.getTransportModule();
            clientResponse = tm.doSend(request, true);

            // according to spec, if response status is 4xx, the request was
            // not accepted by receiver. we must throw a system fault

            int sc = clientResponse.getStatusCode();

            if (sc >= 400 && sc < 500) {
                String faultReason = clientResponse.getStatusReasonPhrase();
                throw new SystemFault(app, faultReason);
            }

            // from this point forward, the request is considered to have been
            // 'accepted for processing' by responding http node, i.e. we now
            // transition from the 'requesting' state to the 'sending+receiving'
            // state. pass response to receivePartnerServiceResponse method where
            // response will be processed and passed to engine on a separate
            // thread

            receivePartnerServiceResponse(eor, requestModules, clientResponse);

        } finally {

            // response envelope has been serialized to file system by this
            // point, so it's safe to close the underlying stream, i.e. by
            // releasing connection

            if (clientResponse != null) {
                clientResponse.releaseConnection();
            }

        }

        log.debug("sendPartnerServiceRequest end");
    }

    /*
     * action represents a response from a partner service referenced by target engine. message
     * direction is out and mep is in-out or robust in-only
     */
    private void receivePartnerServiceResponse(EndpointOperationReference eor, MessageModules requestModules, HTTPClientResponse clientResponse) {

        log.debug("receivePartnerServiceResponse begin");

        // invocation of this method implies message has been accepted by receiver,
        // i.e. call originates via sendPartnerServiceRequest method and it's an anon
        // response, partner is receiver and has implicitly accepted the request.
        // do NOT throw any faults

        EndpointReference epr = eor.getParent();
        ApplicationRT app = ((EngineRT)epr.getRootComponent()).getApplication();

        MessageModules responseModules = null;
        WSAMessageModule wsammIn = null;
        WSAMessageModule wsammOut = null;
        OMElement responsePayload = null;

        try {

            responsePayload = parseHTTPClientResponse(clientResponse);

            InterfaceOperation interfaceOperation = eor.getEndpointOperation().getInterfaceOperation();
            MEP mep = interfaceOperation.getMessageExchangePattern();
            if (mep == MEP.IN_ONLY || (responsePayload == null && mep == MEP.ROBUST_IN_ONLY)) {
                return;
            }

            // because response has no wsa headers, we must derive the
            // implied action. wsdl 2.0 spec does not require a 2xx
            // response for in-out mep, but eoa spec does (which should
            // be consistent across other platforms.) if it's not 2xx,
            // we assume it's a fault and attempt to match payload type
            // to fault message types. note that normal response can be
            // empty. a fault response can't

            // note also that status code defined on binding fault is
            // no help, i.e. because not all faults are declared and
            // spec allows #any

            int statusCode = clientResponse.getStatusCode();
            String action = null;
            if (statusCode > 199 && statusCode < 300) {
                action = interfaceOperation.getOutputMessageReference().getAction();
            } else {
                if (responsePayload != null) {
                    QName payloadType = responsePayload.getQName();
                    for (InterfaceFaultReference ifr : interfaceOperation.getFaultReferences()) {
                        InterfaceMessage im = ifr.getFault().getReferencedComponent();
                        MessagePart messagePart = im.getPart(MessagePart.PAYLOAD);
                        if (messagePart.getSchemaComponentQName().equals(payloadType)) {
                            action = ifr.getAction();
                            break;
                        }
                    }
                }
            }

            boolean isActionUndeclared = false;
            if (action == null) {
                action = SystemFault.ACTION;
                isActionUndeclared = true;
            }

            // build up the ws addressing message module for response using
            // implied values. set value of wsa:To header with the epr
            // fragment identifier. super will use it to map message
            // pulled off of queue with an epr, i.e. so that it can be
            // bound within the readMessage method

            wsammIn = requestModules.getModule(WSAMessageModule.class);
            wsammOut = new WSAMessageModule(action);
            wsammOut.setTo(epr.getFragmentIdentifier().toString());
            wsammOut.setRelatesTo(wsammIn.getMessageID());
            responseModules = new MessageModules(wsammOut);

            // register the message. we know id is unique, because we
            // generated it
            messageDAO.insertMessage(wsammOut.getMessageID(), false);

            if (isActionUndeclared) {
                QName faultCode = BindingFaultRT.RECEIVER_ERROR;
                String faultReason = null;
                List<Content> content = null;
                if (responsePayload == null) {
                    faultReason = "Partner returned an empty fault response.";
                } else {
                    faultReason = "Partner fault response typed as "
                            + responsePayload.getQName()
                            + ", fails to map to a fault message type thrown by operation "
                            + eor.getEndpointOperationName()
                            + ", i.e. the implied action is undefined.";
                    content = Arrays.asList((Content)responsePayload);
                }
                throw new AcceptedResponseError(faultCode, faultReason, content);
            }

            // verify that if binding config contains abstract http header decls
            // marked as required, that they exist and if user has defined
            // a value within provider configuration, verifiy it matches

            try {
                checkRequiredHTTPHeaders(action, clientResponse, null);
            } catch (HTTPSenderFault hs) {
                QName faultCode = BindingFaultRT.SENDER_ERROR;
                List<Content> content = null;
                if (responsePayload != null) {
                    content = Arrays.asList((Content)responsePayload);
                }
                throw new AcceptedResponseError(faultCode, hs.getMessage(), content);
            }

            // so if we make it to this point, we have a declared action which
            // maps to a payload. note that payload could be a normal message,
            // a fault message, or empty. payload will be verified by our msg
            // reader when pulled of queue by super

        } catch (Throwable th) {

            // because response has been accepted for processing, any exceptions
            // caught here must be communicated to engine via receive action method.
            // convert the exception into an accepted response error which will
            // be used to generate an instance of system fault

            AcceptedResponseError are = null;

            if (th instanceof AcceptedResponseError) {
                are = (AcceptedResponseError)th;
            } else {
                QName faultCode = BindingFaultRT.RECEIVER_ERROR;
                String faultReason = "Caught throwable while processing partner response. " + th.getMessage();
                log.error("caught throwable while processing partner response. " + th);
                List<Content> content = null;
                if (responsePayload != null) {
                    content = Arrays.asList((Content)responsePayload);
                }
                are = new AcceptedResponseError(faultCode, faultReason, content);
            }

            // if partner is an eoa compliant engine, they may have thrown a
            // system fault, which is typically undeclared. if so, use existing
            // payload, rather than embedding in another system fault instance

            boolean useExistingPayload = false;
            if (responsePayload != null) {
                if (responsePayload.getQName().equals(SystemFault.MESSAGE_NAME)) {
                    useExistingPayload = true;
                }
            }

            // translate are fault into a system fault so that we can use it
            // to generate an instanceof system fault message which contains
            // partner response within fault detail

            if (useExistingPayload == false) {
                SystemFault systemFault = SystemFault.valueOf(app, are);
                responsePayload = (OMElement)importContent(systemFault.toMessage().getContent());
            }

            // payload is a runtime instance of system fault message.
            // change action to system fault action, i.e. engine
            // should never see an undeclared action uri

            wsammOut.setAction(SystemFault.ACTION);

        }

        // persist message modules to dbms before passing execution off
        // to a separate thread

        for (MessageModule mm : responseModules.values()) {
            try {
                messageModuleDAO.insertMessageModule(wsammOut.getMessageID(), false, mm);
            } catch (Exception ex) {
                log.error("error persisting message modules. " + ex);
                return;
            }
        }

        // write message to queue where it will be picked up by jms
        // receiver on separate thread and processed by read message
        // method on super

        JMSSender jmsSender = null;
        try {

            // serialize message content to a file to avoid buffering entire
            // message to memory as required by jms api. we use message id as
            // file name, which will be used by jms receiver to retrieve from
            // file system

            String fileName = FileUtils.createFileName(wsammOut.getMessageID());
            File responseContent = new File(dataDirectory, fileName);

            // if payload is null, i.e. it's an empty response. we create an
            // empty file, i.e. which will allow user to manage the request
            // files within the binding var dir, i.e. browsing/deleting,
            // etc ... if jms receiver can't find file, it quietly logs a
            // debug message

            if (responsePayload != null) {
                OutputStream out = null;
                XMLStreamWriter writer = null;
                try {
                    out = new FileOutputStream(responseContent);
                    writer = STAXUtils.createXMLStreamWriter(out, Constants.UTF_8);
                    responsePayload.buildNext(); // kids won't serialize otherwise
                    responsePayload.serializeAndConsume(writer);
                } finally {
                    if (out != null) {
                        out.flush();
                        IOUtils.closeQuietly(out);
                    }
                    if (writer != null) {
                        writer.close();
                    }
                }
            } else {
                responseContent.createNewFile();
            }

            jmsSender = jmsSenders.getObject();
            javax.jms.Message msg = jmsSender.createMessage();
            msg.setStringProperty(Constants.MESSAGE_ID_PROPERTY, wsammOut.getMessageID());
            msg.setBooleanProperty(Constants.MY_SERVICE_PROPERTY, false);
            jmsSender.send(msg, receiveQueue, 0, DeliveryMode.PERSISTENT);

        } catch (Throwable th) {
            log.error("caught throwable while receiving partner response. " + th);
        } finally {
            try {
                jmsSenders.addObject(jmsSender);
            } catch (Exception fatchance) {
            }
        }

        log.debug("receivePartnerServiceResponse end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT$Provider#spi_getRequiredBindingFeatures()
     */
    @SuppressWarnings("unchecked")
    public Set<String> spi_getRequiredBindingFeatures() {
        return Collections.EMPTY_SET; // binding modules not supported on http binding
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.Binding$Provider#spi_destroy()
     */
    public void spi_destroy() {
        if (sendConnection != null) {
            try {
                sendConnection.close();
            } catch (JMSException je) {
                log.error("error closing send connection. " + je);
            }
        }
        if (receiveConnection != null) {
            try {
                spi_suspend();
                receiveConnection.close();
            } catch (Exception ex) {
                log.error("error closing receive connection. " + ex);
            }
        }
        // interrupt blocking threads, each of which will throw an exception
        if (jmsRequestors != null) {
            jmsRequestors.destroy();
        }
        // interrupt blocking threads, each of which will throw an exception
        if (jmsSenders != null) {
            jmsSenders.destroy();
        }
        binding = null;
        consumer = null;
        if (messageDAO != null) {
            messageDAO.destroy();
        }
        if (messageModuleDAO != null) {
            messageModuleDAO.destroy();
        }
        if (endpointReferences != null) {
            endpointReferences.clear();
        }
        if (objectName != null) {
            try {
                jmxServerFeature.unRegisterMBean(objectName);
            } catch (JMXServerFeatureException ignore) {
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT$Provider#spi_isSuspended()
     */
    public synchronized boolean spi_isSuspended() {
        return isSuspended;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT$Provider#spi_resume()
     */
    public synchronized void spi_resume() throws ExecutableException {
        if (receiveConnection != null) {
            if (!isSuspended) {
                return;
            }
            try {
                receiveConnection.start();
                isSuspended = false;
            } catch (JMSException je) {
                throw new ExecutableException("Error starting receive connection. " + je);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT$Provider#spi_suspend()
     */
    public synchronized void spi_suspend() throws ExecutableException {
        if (receiveConnection != null) {
            if (isSuspended) {
                return;
            }
            try {
                receiveConnection.stop();
                isSuspended = true;
            } catch (JMSException je) {
                throw new ExecutableException("Error stopping receive connection. " + je);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (HTTPBindingRT)consumer;
        this.factoryContext = consumer.getExtensionFactory().getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.binding.UnderlyingProtocol$Provider#spi_validateEndpointReference(EndpointReference)
     */
    public void spi_validateEndpointReference(EndpointReference endpointReference) throws DeploymentException {

        // map address to epr object such that we can retrieve component when
        // we deserialize a message instance from message queue. if it's a my
        // service request, epr is mapped to address. if it's a partner svc
        // response, epr is mapped to fragment identifier

        ServiceReference serviceReference = (ServiceReference)endpointReference.getParent();
        if (serviceReference.isMyService()) {
            URI address = null;
            try {
                address = new URI(endpointReference.getEndpoint().getAddress());
            } catch (URISyntaxException fatchance) {
            }
            endpointReferences.put(address.toString(), endpointReference);
            String replyToAddress = endpointReference.getFragmentIdentifier().toString();
            endpointReferences.put(replyToAddress, endpointReference); // self invocation
            String scheme = address.getScheme();
            int port = address.getPort();
            String path = address.getPath();
            try {
                address = new URI(scheme, null, "localhost", port, path, null, null);
                endpointReferences.put(address.toString(), endpointReference);
                address = new URI(scheme, null, hostAddress, port, path, null, null);
                endpointReferences.put(address.toString(), endpointReference);
            } catch (URISyntaxException fatchance) {
            }
        } else {
            String replyToAddress = endpointReference.getFragmentIdentifier().toString();
            if (endpointReferences.get(replyToAddress) == null) {
                endpointReferences.put(replyToAddress, endpointReference);
            } else {
                // multiple response actions with same replyTo
            }
        }

        // we can't use instance var, i.e. because we may not have been
        // initialized as of yet. retrieve binding via epr

        BindingReference br = endpointReference.getEndpoint().getBindingReference();
        HTTPBinding httpBinding = (HTTPBinding)br.getReferencedComponent();

        // reusable http bindings are not supported, i.e. we require
        // that all abstract operations are bound, so that we can
        // utilize location to dispatch message

        if (serviceReference.isMyService() && httpBinding.isReusable()) {
            throw new DeploymentException("Binding "
                    + httpBinding.getName()
                    + " referenced by endpoint defined on 'my' service "
                    + serviceReference.getReferencedComponentName()
                    + " is invalid. Reuseable bindings not supported for local services.");
        }

        // verify verify that all operations are bound and that if content model
        // #other is specified, that the bound endpoint belongs to a partner
        // service. the thinking here is that because runtime message instances
        // use dom as the object model, with an option to serialize\parse content
        // using alternate serializations, e.g. JSON, definining their interface
        // using #other (which we interpret as other than xml) would preclude
        // sending receiving xml messages which would not correctly model 'my'
        // service. we instead use the content-type and accept header to negotiate
        // alternate serializations for 'my' service. if the service is a partner
        // service, however, and the application truly only supports a serialization
        // other than xml, e.g. JSON, then using #other as the content model is
        // the only way to support this

        XMLHTTPProtocol up = (XMLHTTPProtocol)httpBinding.getUnderlyingProtocol();
        Service service = serviceReference.getReferencedComponent();
        Interface intrface = service.getInterfaceReference().getReferencedComponent();

        for (InterfaceOperation io : intrface.getOperations()) {
            if (httpBinding.getOperation(io.getName()) == null) {
                throw new DeploymentException("HTTP Binding "
                        + httpBinding.getName()
                        + " which binds interface "
                        + intrface.getName()
                        + " is invalid. Abstract operation "
                        + io.getName()
                        + " is not bound. Unbound operations are not supported by this provider.");
            }
            InterfaceMessageReference imr = io.getInputMessageReference();
            if (imr != null && imr.getContentModel() != null) {
                if (imr.getContentModel() == ContentModel.OTHER) {
                    if (serviceReference.isMyService()) {
                        throw new DeploymentException("HTTP Binding "
                                + httpBinding.getName()
                                + " which binds interface "
                                + intrface.getName()
                                + " referenced by endpoint '"
                                + endpointReference.getEndpointName()
                                + "' is invalid. ContentModel #other is not supported for "
                                + " 'my' service endpoints. Expected #element, #any or #none."
                                + " Client must indicate non-xml serializations, e.g. JSON,"
                                + " via HTTP Content-Type and Accept header.");
                    }
                    String serialization = up.getInputSerialization(io.getName());
                    if (serialization == null) {
                        serialization = Constants.MEDIA_TYPE_APPLICATION_XML;
                    }
                    if (MediaTypeUtil.getContentSerialization(serialization) == null) {
                        throw new DeploymentException("HTTP Binding "
                                + httpBinding.getName()
                                + " which binds interface "
                                + intrface.getName()
                                + " referenced by endpoint '"
                                + endpointReference.getEndpointName()
                                + "' is invalid. ContentModel #other is specified, but "
                                + " input serialization "
                                + serialization
                                + " is not supported by this binding.");
                    }
                    HTTPMethod method = up.getMethod(io.getName());
                    if (method == null) {
                        method = up.getMethodDefault();
                        if (method == null) {
                            if (io.isSafe()) {
                                method = HTTPMethod.GET;
                            } else {
                                method = HTTPMethod.POST;
                            }
                        }
                    }
                    if (method == HTTPMethod.GET || method == HTTPMethod.DELETE) {
                        throw new DeploymentException("HTTP Binding "
                                + httpBinding.getName()
                                + " which binds interface "
                                + intrface.getName()
                                + " referenced by endpoint '"
                                + endpointReference.getEndpointName()
                                + "' is invalid. ContentModel #other is specified"
                                + " with input serialization "
                                + serialization
                                + ", but "
                                + " explicit/implied HTTP method "
                                + method.toString()
                                + " requires "
                                + Constants.MEDIA_TYPE_X_WWW_FORM
                                + " serialization.");
                    }
                }
            }
            imr = io.getOutputMessageReference();
            if (imr != null && imr.getContentModel() != null) {
                if (imr.getContentModel() == ContentModel.OTHER) {
                    if (serviceReference.isMyService()) {
                        throw new DeploymentException("HTTP Binding "
                                + httpBinding.getName()
                                + " which binds interface "
                                + intrface.getName()
                                + " referenced by endpoint '"
                                + endpointReference.getEndpointName()
                                + "' is invalid. ContentModel #other is not supported for "
                                + " 'my' service endpoints. Expected #element, #any or #none."
                                + " Client must indicate non-xml serializations, e.g. JSON,"
                                + " via HTTP Content-Type and Accept header.");
                    } else {
                        // while these values should be used to set the
                        // request Accept header, we will simply ignore
                        // the static definition and use the content type
                        // header to retrieve the appropriate media type
                        // util to read the response (no accept header
                        // implies that application xml should be used
                        // according to the spec)
                    }
                }
            }

        }

        // we do not allow location templating for services hosted locally. this is
        // not something application should care about, i.e. how instance data is
        // serialized over 'the wire', i.e. they 'see' the same payload regardless.
        // we require that requestURI relativized against epr address match location,
        // i.e. which would not work if requestURI contains instance data serialized
        // to URI path (instance data in query string, is o.k.)

        if (serviceReference.isMyService()) {
            for (BindingOperation bo : httpBinding.getOperations()) {
                HTTPLocationInfo locInfo = up.getHTTPLocationInfo(bo.getName());
                if (locInfo == null) {
                    throw new DeploymentException("HTTP Binding "
                            + httpBinding.getName()
                            + " referenced by endpoint defined on 'my' service "
                            + serviceReference.getReferencedComponentName()
                            + " is invalid. Operation "
                            + bo.getName()
                            + " must define whttp:location info.");
                }
                String location = locInfo.getLocation();
                if (location == null || location.equals("")) {
                    throw new DeploymentException("HTTP Binding "
                            + httpBinding.getName()
                            + " referenced by endpoint defined on 'my' service "
                            + serviceReference.getReferencedComponentName()
                            + " is invalid. Operation "
                            + bo.getName()
                            + " must define whttp:location value.");
                }
                if (location.contains("{")) {
                    throw new DeploymentException("HTTP Binding "
                            + httpBinding.getName()
                            + " referenced by endpoint defined on 'my' service "
                            + serviceReference.getReferencedComponentName()
                            + " is invalid. Operation "
                            + bo.getName()
                            + " which defines whttp:location value '"
                            + location
                            + "' uses location templating. Location templating not supported"
                            + " for locally hosted services.");
                }
            }
        }

    }

    /**
     * Parses message payload retrieved from jms queue. This method is invoked by instances of
     * {@link JMSReceiver}.
     * 
     * @param messageID
     * @param myService
     *        required in addition to messageid to make primary key unique, i.e. when partner
     *        engine is defined on same machine. true if receiving my service request or
     *        sending my service response. false if sending partner service request or
     *        receiving partner service response.
     * @param inputStream
     *        payload content or <code>null</code> if ##none
     * @return runtime abstract message instance
     * @throws org.bluestemsoftware.specification.eoa.component.binding.rt.BindingFault
     */
    public void readMessage(String messageID, boolean myService, InputStream inputStream) throws org.bluestemsoftware.specification.eoa.component.binding.rt.BindingFaultRT {

        EndpointReference epr = null;
        EndpointActionReference ear = null;
        ActionContext actionContext = null;
        MessageModules messageModules = null;
        WSAMessageModule wsamm = null;
        Message message = null;
        Direction direction = null;

        try {

            messageModules = messageModuleDAO.selectMessageModules(messageID, myService);
            wsamm = messageModules.getModule(WSAMessageModule.class);

            epr = endpointReferences.get(wsamm.getTo());
            ear = epr.getEndpointActionReference(wsamm.getAction());

            // if we're processing a partner response, we need to allow for the
            // fact that it may be a fault. note that caller insures that action
            // is declared. all undeclared actions are converted to a sys fault
            // message

            InterfaceMessage am = null;
            QName faultName = null;
            if (ear != null) {
                InterfaceAction ia = ear.getReferencedComponent().getInterfaceAction();
                direction = ia.getDirection();
                if (ia instanceof InterfaceMessageReference) {
                    am = ((InterfaceMessageReference)ia).getReferencedComponent();
                } else {
                    InterfaceFault interfaceFault = ((InterfaceFaultReference)ia).getFault();
                    am = interfaceFault.getReferencedComponent();
                    faultName = interfaceFault.getName();
                }
            } else {
                FragmentIdentifier fragmentID = SystemFault.MESSAGE_FRAGMENT_ID;
                am = (InterfaceMessage)SystemContext.getContext().getSystem().getComponent(fragmentID);
                direction = Direction.OUT;
            }

            message = MessageReader.readMessage(epr, am, messageModules, inputStream);

            if (ear == null) {
                ApplicationRT app = ((EngineRT)epr.getRootComponent()).getApplication();
                actionContext = new FaultContext(messageModules, SystemFault.valueOf(app, message));
            } else {
                if (faultName == null) {
                    actionContext = new MessageContext(messageModules, message);
                } else {
                    actionContext = new FaultContext(messageModules, new BusinessFault(faultName, message));
                }
            }

        } catch (Throwable th) {

            // msg reader should only throw http sender and/or http receiver
            // faults. anything else is a programming error that should be
            // logged

            if (th instanceof HTTPFault) {
                if (log.isDebugEnabled()) {
                    log.debug("caught http fault while binding received action "
                            + wsamm.getAction()
                            + " defined on endpoint referenced by component "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + ((HTTPFault)th).getFaultReason());
                }
            } else {
                if (th instanceof SQLException || th instanceof DAOException) {
                    log.error("error reading message modules. " + th.getMessage());
                } else {
                    log.error("caught throwable while binding received action "
                            + wsamm.getAction()
                            + " defined on endpoint referenced by component "
                            + epr.getFragmentIdentifier()
                            + ". "
                            + th);
                }
            }

            if (direction == Direction.IN) {

                // we encountered an error while receiving a request targeted to
                // locally hosted service. if mep is in-out or robust-in-only,
                // we return a response to client, i.e. request failed within
                // binding layer and will not be passed to engine

                EndpointOperationReference eor = (EndpointOperationReference)ear.getParent();
                MEP mep = eor.getEndpointOperation().getInterfaceOperation().getMessageExchangePattern();

                if (mep == MEP.IN_ONLY) {

                    // wsdl 20 spec requires an empty body in response to one way mep,
                    // with an accepted code, even if we failed to process the request

                    log.debug("MEP is in-only. No fault response allowed.");
                    return;

                }

                HTTPFault hf = null;
                if (th instanceof HTTPFault) {
                    hf = (HTTPFault)th;
                } else {
                    if (configuration.isEmbeddStackTraceInFaults()) {
                        hf = new HTTPReceiverFault(th);
                    } else {
                        hf = new HTTPReceiverFault(th.getMessage());
                    }
                }

                // pass response to sendMyServiceResponse method which will handle
                // module processing, etc ...

                try {
                    WSAMessageModule wsammOut = new WSAMessageModule(hf.getAction());
                    wsammOut.setRelatesTo(wsamm.getMessageID());
                    MessageModules responseModules = new MessageModules(wsammOut);
                    HTTPFaultContext hfc = new HTTPFaultContext(null, messageModules, responseModules, hf);
                    sendMyServiceResponse(epr, hfc);
                } catch (Throwable thr) {
                    log.error("caught throwable while returning http fault response. " + thr);
                }

                return;

            } else {

                // we're processing a partner response. convert fault to an instance
                // of accepted response fault containing partner response, if avail,
                // and then convert to a system fault, which will be passed to app

                QName faultCode = null;
                String faultReason = null;

                if (th instanceof HTTPFault) {
                    faultCode = ((HTTPFault)th).getFaultName();
                    faultReason = ((HTTPFault)th).getFaultReason();
                } else {
                    faultReason = "Caught throwable while processing partner response. " + th.toString();
                }

                List<Content> payload = null;
                if (th instanceof HTTPSenderFault) {
                    payload = Arrays.asList(((HTTPSenderFault)th).getPayload());
                }

                AcceptedResponseError are = new AcceptedResponseError(faultCode, faultReason, payload);
                ApplicationRT app = ((EngineRT)ear.getRootComponent()).getApplication();
                SystemFault systemFault = SystemFault.valueOf(app, are);
                actionContext = new FaultContext(messageModules, systemFault);

            }

        }

        // message has successfully traversed through the 'binding layer'. now we
        // pass the abstract modules and abstract message to engine which is
        // responsible for guiding it through the 'application layer'

        try {

            epr.receiveAction(actionContext);

        } catch (Throwable th) {

            // engine should not propogate exceptions. all we can do here is log
            // an error, i.e. if this is a partner svc response, invoking engine
            // again would probably cause same error. if this is a my service
            // request, engine may have successfully processed the request and
            // allowed an exception, i.e. a system fault, to propogate from the
            // sendAction method while returning response - attempting to unwravel
            // this would be ugly at best

            log.error("engine threw/propogated an exception while receiving action "
                    + wsamm.getAction()
                    + " defined on endpoint referenced by component "
                    + epr.getFragmentIdentifier()
                    + ". "
                    + th);

            // note that the jms requestor will eventually timeout and return a
            // timeout error message to client (as a system fault)

        }

    }

    /*
     * parses optional char set parameter from content type, e.g.
     * application/soap+xml;action="http://foo"[;charset=utf-8]
     */
    private String getCharacterSetEncoding(String contentType) {

        String answer = contentType;

        // locate name value pair. if undefined, default value to utf-8
        int index = answer.indexOf("charset");
        if (index == -1) {
            return Constants.UTF_8;
        }

        // name value pair may be the last pair or an intermediate pair
        index = answer.indexOf("=", index);
        if (answer.indexOf(";", index) == -1) {
            answer = answer.substring(index + 1, answer.length()).trim();
        } else {
            answer = answer.substring(index + 1, answer.indexOf(";", index)).trim();
        }

        // trim double quotes which surround encoding, if any
        if (answer.startsWith("\"") && answer.endsWith("\"")) {
            answer = answer.substring(1, answer.length());
        }

        return answer.trim();

    }

    private OMElement parseHTTPClientResponse(HTTPClientResponse clientResponse) throws Exception {

        // as of version 1.3, in some situations, axis writes garbage to
        // http response body and returns without a content-type header
        // when response status is accepted (202). so if status is accepted,
        // do not attempt to parse, i.e. because decendant will throw an
        // unsupported media type exception

        InputSource responseContent = null;
        if (clientResponse.getStatusCode() == 202) {
            return null;
        } else {
            responseContent = clientResponse.getContent();
        }

        if (responseContent == null) {
            return null;
        }

        Map<String, String> responseHeaders = clientResponse.getHeaders();

        // retrieve input stream from request. check if content is encoded
        // using gzip, which is the only encoding we support currently

        InputStream is = null;
        String ce = responseHeaders.get(Constants.HEADER_CONTENT_ENCODING);
        if (ce != null) {
            if (ce.equalsIgnoreCase("gzip")) {
                try {
                    is = new GZIPInputStream(responseContent.getByteStream());
                } catch (IOException ie) {
                    if (configuration.isEmbeddStackTraceInFaults()) {
                        throw new HTTPReceiverFault("Error decoding content.", ie);
                    } else {
                        log.warn("Error decoding content.", ie);
                        throw new HTTPReceiverFault("Error decoding content. " + ie.getMessage());
                    }
                }
            } else {
                throw new HTTPSenderFault("Content-Encoding '" + ce + "' not supported.");
            }
        } else {
            is = responseContent.getByteStream();
        }

        // extract media type from content type header. if media type is un-
        // expected, descendant throws a soap fault

        String responseCType = responseHeaders.get(Constants.HEADER_CONTENT_TYPE);
        responseCType = responseCType == null ? Constants.DEFAULT_CONTENT_TYPE : responseCType;
        String responseMediaType = null;
        int index = responseCType.indexOf(';');
        if (index > 0) {
            responseMediaType = responseCType.substring(0, index);
        } else {
            responseMediaType = responseCType;
        }

        // if no matching media type util found, we do not support the
        // returned type. if a match is found and media type not supported 
        // for response, e.g. url encoded, media type util throws an http
        // sender fault
        
        MediaTypeUtil mt = MediaTypeUtil.getMediaTypeUtil(responseMediaType);

        if (mt == null) {
            throw new HTTPSenderFault("Unsupported media type '"
                    + responseMediaType
                    + "' employed on response.");
        }

        // the response contains plain old xml, i.e. no query string
        // formatted text is allowed within responses

        return mt.readResponse(is, responseContent.getEncoding());

    }

    private Content writeMessage(EndpointReference epr, MessageContext messageContext) throws SystemFault {
        EndpointActionReference ear = epr.getEndpointActionReference(messageContext.getAction());
        InterfaceAction ia = ear.getReferencedComponent().getInterfaceAction();
        InterfaceMessage am = ((InterfaceMessageReference)ia).getReferencedComponent();
        return MessageWriter.writeMessage(ear, am, messageContext);
    }

    private File getDataDirectory() throws Exception {
        File systemVarDir = SystemContext.getContext().getSystem().getSystemVarDir();
        File temp = URIUtils.toFile(new URI(binding.getFragmentIdentifier().toString()), null);
        File result = new File(systemVarDir, temp.toString());
        if (!result.exists()) {
            result.mkdir();
        }
        return result;
    }

    private static Content importContent(Content content) throws SystemFault {
        ContentFactory cf = (ContentFactory)new OMDOMFactory().createOMDocument();
        Element imported = null;
        try {
            imported = cf.importContent(content);
        } catch (Exception ex) {
            imported = (Element)cf.importNode(content, true);
        }
        return (Content)imported;
    }

    private void checkRequiredHTTPHeaders(String action, HTTPClientResponse response, HTTPServerRequest request) throws HTTPSenderFault {

        if (binding.isReusable()) {
            return; // no abstract header decls allowed
        }

        BindingAction ba = binding.getBindingAction(action);
        if (ba == null) {
            return;
        }

        HyperTextTransferProtocol http = (HyperTextTransferProtocol)binding.getUnderlyingProtocol();

        if (ba instanceof HTTPBindingMessageReference) {
            QName operationName = ((HTTPBindingMessageReference)ba).getParent().getName();
            Direction direction = ((HTTPBindingMessageReference)ba).getDirection();
            for (HTTPHeader httpHeader : http.getMessageReferenceHTTPHeaders(operationName, direction)) {
                String name = httpHeader.getName();
                String value = null;
                if (request == null) {
                    value = response.getHeaders().get(name);
                } else {
                    value = request.getHeader(name);
                }
                if (value == null) {
                    if (httpHeader.isRequired()) {
                        throw new HTTPSenderFault("Required http header '" + name + "' is missing.");
                    }
                } else {
                    String expected = configuration.getMessageReferenceHTTPHeaderValue(operationName, direction,
                            name);
                    if (expected != null) {
                        if (!expected.equals(value)) {
                            throw new HTTPSenderFault("HTTP header '"
                                    + name
                                    + "' failed to match expected value '"
                                    + expected
                                    + "'.");
                        }
                    }
                }
            }
        } else {
            BindingFault bf = ((BindingFaultReference)ba).getFault();
            QName faultName = bf.getFaultName();
            for (HTTPHeader httpHeader : http.getFaultHTTPHeaders(faultName)) {
                String name = httpHeader.getName();
                String value = null;
                if (request == null) {
                    value = response.getHeaders().get(name);
                } else {
                    value = request.getHeader(name);
                }
                if (value == null) {
                    if (httpHeader.isRequired()) {
                        throw new HTTPSenderFault("Required http header '" + name + "' is missing.");
                    }
                } else {
                    String expected = configuration.getFaultHTTPHeaderValue(faultName, name);
                    if (expected != null) {
                        if (!expected.equals(value)) {
                            throw new HTTPSenderFault("HTTP header '"
                                    + name
                                    + "' failed to match expected value '"
                                    + expected
                                    + "'.");
                        }
                    }
                }
            }
        }

    }

    private Map<String, String> cacheRequiredHTTPHeaders(String action, BindingAction ba) throws HTTPBindingException {

        Map<String, String> headerMap = new HashMap<String, String>();
        httpHeaders.put(action, headerMap);

        HyperTextTransferProtocol http = (HyperTextTransferProtocol)binding.getUnderlyingProtocol();

        if (ba instanceof HTTPBindingMessageReference) {
            QName operationName = ((HTTPBindingMessageReference)ba).getParent().getName();
            Direction direction = ((HTTPBindingMessageReference)ba).getDirection();
            for (HTTPHeader httpHeader : http.getMessageReferenceHTTPHeaders(operationName, direction)) {
                String name = httpHeader.getName();
                String value = configuration.getMessageReferenceHTTPHeaderValue(operationName, direction, name);
                if (value == null) {
                    if (httpHeader.isRequired()) {
                        throw new HTTPBindingException("Value for required http header '"
                                + name
                                + "' not defined within binding provider configuration.");
                    }
                } else {
                    headerMap.put(name, value);
                }
            }
        } else {
            BindingFault bf = ((BindingFaultReference)ba).getFault();
            QName faultName = bf.getFaultName();
            for (HTTPHeader httpHeader : http.getFaultHTTPHeaders(faultName)) {
                String name = httpHeader.getName();
                String value = configuration.getFaultHTTPHeaderValue(faultName, name);
                if (value == null) {
                    if (httpHeader.isRequired()) {
                        throw new HTTPBindingException("Value for required http header '"
                                + name
                                + "' not defined within binding provider configuration.");
                    }
                } else {
                    headerMap.put(name, value);
                }
            }
        }

        return headerMap;

    }

    static class RequiredHeaderException extends Exception {

        private static final long serialVersionUID = 1L;

        public RequiredHeaderException() {
            super();
        }

        public RequiredHeaderException(String message) {
            super(message);
        }

    }

}
