/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.InputStream;
import java.net.URI;
import java.util.Map;
import java.util.zip.GZIPInputStream;

import org.apache.axiom.om.OMElement;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationMappedJSON;

/**
 * Utility class used to handle messages formatted using 'multipart/form-data' media type.
 */
public class ApplicationMappedJSONUtil extends MediaTypeUtil {

    private static ContentSerialization MAPPED = ContentSerializationMappedJSON.getInstance();
    private static ApplicationMappedJSONUtil singleton = new ApplicationMappedJSONUtil();

    private ApplicationMappedJSONUtil() {
    }

    static ApplicationMappedJSONUtil getInstance() {
        return singleton;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#requiresRequestBody()
     */
    @Override
    public boolean requiresRequestBody() {
        return true;
    }

    @Override
    public OMElement readData(HTTPServerRequest request, Map<String, Style> styles, String location, String separator, String encoding, boolean isGZIP) throws Exception {
        
        InputStream in = null;
        if (isGZIP) {
            in = new GZIPInputStream(request.getInputStream());
        } else {
            in = request.getInputStream();
        }
        
        return parseData(MAPPED, in, encoding); // streams plain old xml
        
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#writeData(org.bluestemsoftware.specification.eoa.component.message.rt.Content, java.util.Map, java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    @Override
    public InstanceData writeData(Content payload, Map<String, Style> styles, String address, String location, String separator, boolean ignoreUncited) throws Exception {
        
        // all instance data is serialized to request body as plain
        // old xml. note that when resolving location, two styles
        // are appropriate, e.g.
        // address = http://foo/myEP location = myEP/myLocation
        // address = http://foo/myEP/ location = myLocation
        //
        // wsdl author is responsible for understanding difference
        // else it won't work, e.g.
        // address = http://foo/myEP location = myLocation
        // resolves to:
        // http://foo/myLocation

        URI requestURI = new URI(address).resolve(location);
        return new InstanceData(requestURI, payload.serializeContent(MAPPED, "UTF-8"));
        
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#readResponse(java.io.InputStream, java.lang.String)
     */
    @Override
    public OMElement readResponse(InputStream in, String encoding) throws Exception {
        return parseData(MAPPED, in, encoding);
    }

       
}
