/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

import javax.xml.namespace.QName;

import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.om.OMOutputFormat;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPSenderFault;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle.ChildElement;

/**
 * Utility class used to handle messages formatted using 'application/xml' media type.
 * <p>
 * Note that the code within this class was largely adapted/copied from several classes within
 * version 1.3 of axis-kernel artifact including:
 * 
 * org.apache.axis2.builder.XFormURLEncodedBuilder
 * org.apache.axis2.transport.http.XFormURLEncodedFormatter
 * 
 */
public class ApplicationXMLUtil extends MediaTypeUtil {

    private static ContentSerialization XML = ContentSerializationXML.getInstance();
    private static ApplicationXMLUtil singleton = new ApplicationXMLUtil();

    private ApplicationXMLUtil() {
    }

    static ApplicationXMLUtil getInstance() {
        return singleton;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#requiresRequestBody()
     */
    @Override
    public boolean requiresRequestBody() {
        return true;
    }

    public OMElement readData(HTTPServerRequest request, Map<String, Style> styles, String location, String separator, String encoding, boolean isGZIP) throws Exception {

        InputStream in = null;
        if (isGZIP) {
            in = new GZIPInputStream(request.getInputStream());
        } else {
            in = request.getInputStream();
        }

        // extract the 'instance data', i.e. payload. note this method
        // applies to requests sent to a 'my' service. we do not support
        // serialization of instance data to path of uri (using location
        // templating) for 'my' services

        IRIStyle style = (IRIStyle)styles.get(IRIStyle.STYLE_IRI);
        if (style == null) {
            return parseData(XML, in, encoding); // streams plain old xml
        }

        // if iri style applies, we retrieve data formatted as a query
        // string from body of request

        Map<String, List<String>> params = extractParameters(null, separator, encoding, in);

        QName wrapperName = style.getElementName();
        OMElement wrapperElement = omFactory.createOMElement(style.getElementName());
        OMNamespace ns = omFactory.createOMNamespace(wrapperName.getNamespaceURI(), wrapperName.getPrefix());
        for (ChildElement childElement : style.getChildElements()) {
            int minOccurs = childElement.getMinOccurs();
            String childName = childElement.getName();
            for (String value : params.get(childName)) {
                omFactory.createOMElement(childName, ns, wrapperElement).setText(value);
            }
            minOccurs--;
            if (minOccurs > 0) {
                if (childElement.getDefaultValue() != null) {
                    String defaultValue = childElement.getDefaultValue();
                    omFactory.createOMElement(childName, ns, wrapperElement).setText(defaultValue);
                } else if (childElement.isNillable()) {
                    OMNamespace xsiNS = omFactory.createOMNamespace(Constants.XSI_NS, Constants.XSI_PREFIX);
                    OMAttribute att = omFactory.createOMAttribute("nil", xsiNS, "true");
                    omFactory.createOMElement(childName, ns, wrapperElement).addAttribute(att);
                } else {
                    throw new HTTPSenderFault("Missing parameter for required child element " + childName);
                }
            }
        }

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OMOutputFormat format = new OMOutputFormat();
        format.setCharSetEncoding(encoding);
        wrapperElement.serializeAndConsume(baos, format);
        return parseData(XML, new ByteArrayInputStream(baos.toByteArray()), encoding);

    }

    public InstanceData writeData(Content payload, Map<String, Style> styles, String address, String location, String separator, boolean ignoreUncited) throws Exception {

        // this media type is only used with put/post, i.e. we know
        // we have request body

        IRIStyle style = (IRIStyle)styles.get(IRIStyle.STYLE_IRI);

        if (style == null) {

            // all instance data is serialized to request body as plain
            // old xml. note that when resolving location, two styles
            // are appropriate, e.g.
            // address = http://foo/myEP location = myEP/myLocation
            // address = http://foo/myEP/ location = myLocation
            //
            // wsdl author is responsible for understanding difference
            // else it won't work, e.g.
            // address = http://foo/myEP location = myLocation
            // resolves to:
            // http://foo/myLocation

            URI requestURI = new URI(address).resolve(location);
            return new InstanceData(requestURI, payload);

        }

        // instance data may be serialized to path of request
        // uri

        separator = separator == null ? separator = "&" : separator;

        // using location template encode child elements to path
        // of request uri. serialized child elements are detached
        // from payload. note that we set detachChildElements to
        // false. spec indicates that ALL instance data is written
        // to body when media type is app/xml (not sure why someone
        // would serialize some of the payload to uri, when all of
        // it must be serialized to body ...)

        URI requestURI = getTemplatedURI(payload, separator, address, location, false);

        // format ALL child elements as a query string which will
        // be serialized to request body, ignore uncited is only
        // applicable to the url encoded format

        return new InstanceData(requestURI, toQueryString(payload, separator));

    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#readResponse(java.io.InputStream, java.lang.String)
     */
    @Override
    public OMElement readResponse(InputStream in, String encoding) throws Exception {
        return parseData(XML, in, encoding);
    }

}
