/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

import javax.xml.namespace.QName;

import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.om.OMOutputFormat;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPSenderFault;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle.ChildElement;

/**
 * Utility class used to handle messages formatted using 'application/x-www-form-urlencoded'
 * media type.
 * <p>
 * Note that the code within this class was largely adapted/copied from several classes within
 * version 1.3 of axis-kernel artifact including:
 * 
 * org.apache.axis2.builder.XFormURLEncodedBuilder
 * org.apache.axis2.transport.http.XFormURLEncodedFormatter
 * 
 */
public class FormURLEncodedUtil extends MediaTypeUtil {

    private static ContentSerialization XML = ContentSerializationXML.getInstance();
    private static FormURLEncodedUtil singleton = new FormURLEncodedUtil();

    private FormURLEncodedUtil() {
    }

    static FormURLEncodedUtil getInstance() {
        return singleton;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#readData(org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest, java.util.Map, java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    public OMElement readData(HTTPServerRequest request, Map<String, Style> styles, String location, String separator, String encoding, boolean isGZIP) throws Exception {

        InputStream in = null;
        if (isGZIP && request.getInputStream() != null) {
            in = new GZIPInputStream(request.getInputStream());
        } else {
            in = request.getInputStream();
        }

        IRIStyle style = (IRIStyle)styles.get(IRIStyle.STYLE_IRI);
        if (style == null) {
            throw new HTTPSenderFault("Media type "
                    + Constants.MEDIA_TYPE_X_WWW_FORM
                    + " requires operation style "
                    + IRIStyle.STYLE_IRI);
        }

        // extract instance data from query string and optional request body.
        // note that this method applies to requests sent to a 'my' service.
        // we do not support serialization of instance data to path of uri
        // (using location templating) for 'my' services. this was enforced
        // within valid epr on binding. all instance data is contained within
        // the query string

        String query = decode(request.getQueryString() == null ? "" : request.getQueryString());
        Map<String, List<String>> params = extractParameters(query, separator, encoding, in);

        QName wrapperName = style.getElementName();
        OMNamespace ns = omFactory.createOMNamespace(wrapperName.getNamespaceURI(), "ns");
        OMElement wrapperElement = omFactory.createOMElement(wrapperName.getLocalPart(), ns);
        for (ChildElement childElement : style.getChildElements()) {
            int minOccurs = childElement.getMinOccurs();
            String childName = childElement.getName();
            List<String> values = params.get(childName);
            if (values != null) {
                for (String value : values) {
                    omFactory.createOMElement(childName, ns, wrapperElement).setText(value);
                    minOccurs--;
                }
            }
            if (minOccurs > 0) {
                if (childElement.getDefaultValue() != null) {
                    String defaultValue = childElement.getDefaultValue();
                    omFactory.createOMElement(childName, ns, wrapperElement).setText(defaultValue);
                } else if (childElement.isNillable()) {
                    OMNamespace xsiNS = omFactory.createOMNamespace(Constants.XSI_NS, Constants.XSI_PREFIX);
                    OMAttribute att = omFactory.createOMAttribute("nil", xsiNS, "true");
                    omFactory.createOMElement(childName, ns, wrapperElement).addAttribute(att);
                } else {
                    throw new HTTPSenderFault("Missing parameter for required child element '" + childName + "'.");
                }
            }
        }

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OMOutputFormat format = new OMOutputFormat();
        format.setCharSetEncoding(encoding);
        wrapperElement.serializeAndConsume(baos, format);
        return parseData(XML, new ByteArrayInputStream(baos.toByteArray()), encoding);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#writeData(org.bluestemsoftware.specification.eoa.component.message.rt.Content, java.util.Map, java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    public InstanceData writeData(Content payload, Map<String, Style> styles, String address, String location, String separator, boolean ignoreUncited) throws Exception {

        if (styles.get(IRIStyle.STYLE_IRI) == null) {
            throw new HTTPSenderFault("Media type "
                    + Constants.MEDIA_TYPE_X_WWW_FORM
                    + " requires operation style "
                    + IRIStyle.STYLE_IRI);
        }

        separator = separator == null ? separator = "&" : separator;

        // using location template encode child elements to PATH
        // of request uri. serialized child elements are detached
        // from payload. note that we detach child elements from
        // payload that are serialized to URI, i.e. spec indicates
        // that only remaining child elements are serialized to
        // body as query string

        URI requestURI = getTemplatedURI(payload, separator, address, location, true);

        // encode remaining child elements to uri query string
        // unless ignoreUncited is true

        String queryString = null;
        if (ignoreUncited == false) {
            queryString = toQueryString(payload, separator);
        }

        return new InstanceData(requestURI, queryString);

    }
    
    @Override
    public OMElement readResponse(InputStream in, String encoding) throws Exception {
        throw new HTTPSenderFault("Unsupported response media type " + Constants.MEDIA_TYPE_X_WWW_FORM);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.MediaTypeUtil#requiresRequestBody()
     */
    @Override
    public boolean requiresRequestBody() {
        return false;
    }

}
