/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.zip.GZIPOutputStream;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMOutputFormat;
import org.apache.axiom.om.impl.dom.ElementImpl;
import org.apache.commons.io.IOUtils;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest.RequestEntity;

public class HTTPRequestEntity implements RequestEntity {

    private ClassLoader classLoader;
    private Object payload;
    private OMOutputFormat omFormat;
    private boolean isGZIP;

    public HTTPRequestEntity(ClassLoader classLoader, Object payload, OMOutputFormat omFormat, boolean isGZIP) {
        this.classLoader = classLoader;
        this.payload = payload;
        this.omFormat = omFormat;
        this.isGZIP = isGZIP;
    }

    public void writeRequest(OutputStream outputStream) throws IOException {
        
        if (payload == null) {
            return; // empty request
        }

        if (isGZIP) {
            outputStream = new GZIPOutputStream(outputStream);
        }

        // set context classloader to our deployment classload, i.e.
        // because we are invoked by http client feature

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        XMLStreamWriter writer = null;
        try {
            thread.setContextClassLoader(classLoader);
            String encoding = omFormat.getCharSetEncoding();
            if (payload instanceof String) {
                String s = (String)payload;
                ByteArrayInputStream in = new ByteArrayInputStream(s.getBytes(encoding));
                IOUtils.copy(in, outputStream);
            } else {
                writer = STAXUtils.createXMLStreamWriter(outputStream, encoding);
                ((ElementImpl)payload).serializeAndConsume(writer, omFormat);
            }
            if (outputStream instanceof GZIPOutputStream) {
                ((GZIPOutputStream)outputStream).finish();
            }
        } catch (XMLStreamException xe) {
            throw new IOException(xe);
        } finally {
            if (writer != null) {
                try {
                    writer.flush();
                    writer.close();
                } catch (XMLStreamException ignore) {
                }
            }
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest$RequestEntity#isEmpty()
     */
    public boolean isEmpty() {
        return payload == null;
    }

}
