/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMException;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.apache.woden.wsdl20.extensions.http.HTTPLocation;
import org.apache.woden.wsdl20.extensions.http.HTTPLocationTemplate;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerRequest;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationBadgerJSON;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationMappedJSON;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * Abstract utility class used to handle messages formatted according to a specific media type.
 * <p>
 * Note that the code within this class was largely adapted/copied from several classes within
 * version 1.3 of axis-kernel artifact including:
 * 
 * org.apache.axis2.transport.http.util.URLTemplatingUtil org.apache.axis2.util.WSDL20Util
 * org.apache.axis2.transport.http.util.URIEncoderDecoder
 * 
 */
public abstract class MediaTypeUtil {

    protected static OMFactory omFactory = OMAbstractFactory.getOMFactory();

    /*
     * note that we use contains rather than equals, because the
     * string could be a media type range production
     */
    public static MediaTypeUtil getMediaTypeUtil(String mediaType) {
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_XML)) {
            return ApplicationXMLUtil.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_X_WWW_FORM)) {
            return FormURLEncodedUtil.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_MULTIPART_FORM_DATA)) {
            return MultipartFormDataUtil.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON_BADGER)) {
            return ApplicationBadgerJSONUtil.getInstance();
        }
        // all other json media types we associate with the mapped format
        // which is the 'spec compliant' format (w/ no namespace support)
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON)) {
            return ApplicationMappedJSONUtil.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JSON)) {
            return ApplicationMappedJSONUtil.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JAVASCRIPT)) {
            return ApplicationMappedJSONUtil.getInstance();
        }
        return null;
    }
    
    /*
     * note that we use contains rather than equals, because the
     * string could be a media type range production
     */
    public static ContentSerialization getContentSerialization(String mediaType) {
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_XML)) {
            return ContentSerializationXML.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON_BADGER)) {
            return ContentSerializationBadgerJSON.getInstance();
        }
        // all other json media types we associate with the mapped format
        // which is the 'spec compliant' format (w/ no namespace support)
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON)) {
            return ContentSerializationMappedJSON.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JSON)) {
            return ContentSerializationMappedJSON.getInstance();
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JAVASCRIPT)) {
            return ContentSerializationMappedJSON.getInstance();
        }
        throw new IllegalArgumentException("Unsupported media type " + mediaType);
    }
    
    /*
     * note that we use contains rather than equals, because the
     * string could be a media type range production
     */
    public static String getContentType(String mediaType) {
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_XML)) {
            return Constants.APPLICATION_XML_CONTENT_TYPE;
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON_BADGER)) {
            return Constants.APPLICATION_JSON_BADGER_CONTENT_TYPE;
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_APPLICATION_JSON)) {
            return Constants.APPLICATION_JSON_CONTENT_TYPE;
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JSON)) {
            return Constants.TEXT_JSON_CONTENT_TYPE;
        }
        if (mediaType.contains(Constants.MEDIA_TYPE_TEXT_JAVASCRIPT)) {
            return Constants.TEXT_JAVASCRIPT_CONTENT_TYPE;
        }
        throw new IllegalArgumentException("Unsupported media type " + mediaType);
    }
    
    public abstract OMElement readData(HTTPServerRequest request, Map<String, Style> styles, String location, String separator, String encoding, boolean isGZIP) throws Exception;

    public abstract OMElement readResponse(InputStream in, String encoding) throws Exception;

    public abstract InstanceData writeData(Content payload, Map<String, Style> styles, String address, String location, String separator, boolean ignoreUncited) throws Exception;

    public abstract boolean requiresRequestBody();
    
    protected static URI getTemplatedURI(Content payload, String queryParameterSeparator, String address, String location, boolean detachChildElements) throws Exception {
        String replacedQuery = location;
        int separator = location.indexOf('{');
        if (separator > -1) {
            HTTPLocation httpLocation = new HTTPLocation(location);
            HTTPLocationTemplate[] templates = httpLocation.getTemplates();
            for (int i = 0; i < templates.length; i++) {
                HTTPLocationTemplate template = templates[i];
                String localName = template.getName();
                String elementValue = getElementValue(localName, payload, detachChildElements);
                if (template.isEncoded()) {
                    if (template.isQuery()) {
                        template.setValue(encode(elementValue, Constants.LEGAL_CHARACTERS_IN_QUERY.replaceAll(
                                queryParameterSeparator, "")));
                    } else {
                        template.setValue(encode(elementValue, Constants.LEGAL_CHARACTERS_IN_PATH));
                    }
                } else {
                    template.setValue(elementValue);
                }
            }
            replacedQuery = httpLocation.getFormattedLocation();
            replacedQuery = encode(replacedQuery, Constants.LEGAL_CHARACTERS_IN_URL);
        }
        URI appendedURI = null;
        if (replacedQuery.charAt(0) == '?') {
            appendedURI = new URI(address + replacedQuery);
        } else {
            if (!address.endsWith("/")) {
                address = address + "/";
            }
            appendedURI = new URI(address).resolve(replacedQuery);
        }
        return appendedURI;
    }

    protected OMElement parseData(ContentSerialization ser, InputStream in, String encoding) throws Exception {
        XMLStreamReader parser = STAXUtils.createXMLStreamReader(ser, in, encoding);
        StAXOMBuilder builder = new StAXOMBuilder(new OMDOMFactory(), parser);
        try {
            builder.next();
        } catch (OMException oe) {
            Throwable th = oe.getCause();
            if (th != null && th instanceof com.ctc.wstx.exc.WstxEOFException) {
                return null; // empty request
            }
            throw oe;
        }
        return builder.getDocumentElement();
    }

    protected static String encode(String s, String legal) throws Exception {
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < s.length(); i++) {
            char ch = s.charAt(i);
            if ((ch >= 'a' && ch <= 'z')
                    || (ch >= 'A' && ch <= 'Z')
                    || (ch >= '0' && ch <= '9')
                    || legal.indexOf(ch) > -1) {
                buf.append(ch);
            } else {
                byte[] bytes = new String(new char[] { ch }).getBytes("UTF-8");
                for (int j = 0; j < bytes.length; j++) {
                    buf.append('%');
                    buf.append(Constants.DIGITS.charAt((bytes[j] & 0xf0) >> 4));
                    buf.append(Constants.DIGITS.charAt(bytes[j] & 0xf));
                }
            }
        }
        return buf.toString();
    }

    protected Map<String, List<String>> extractParameters(String query, String paramSeparator, String encoding, InputStream in) throws Exception {
        Map<String, List<String>> params = new HashMap<String, List<String>>();
        if (query != null && !"".equals(query)) {
            String parts[] = query.split(paramSeparator);
            for (int i = 0; i < parts.length; i++) {
                int separator = parts[i].indexOf("=");
                if (separator > 0) {
                    String name = parts[i].substring(0, separator);
                    List<String> values = params.get(name);
                    if (values == null) {
                        values = new ArrayList<String>();
                        params.put(name, values);
                    }
                    values.add(parts[i].substring(separator + 1));
                }
            }
        }
        if (in != null) {
            InputStreamReader inputStreamReader = new InputStreamReader(in, encoding);
            BufferedReader bufferedReader = new BufferedReader(inputStreamReader);
            StringBuilder sb = new StringBuilder();
            String line = bufferedReader.readLine();
            while (line != null) {
                sb.append(line);
                line = bufferedReader.readLine();
            }
            if (sb.length() > 0) {
                String decoded = decode(sb.toString());
                String parts[] = decoded.split("&");
                for (int i = 0; i < parts.length; i++) {
                    int separator = parts[i].indexOf("=");
                    String name = parts[i].substring(0, separator);
                    List<String> values = params.get(name);
                    if (values == null) {
                        values = new ArrayList<String>();
                        params.put(name, values);
                    }
                    values.add(parts[i].substring(separator + 1));
                }
            }
        }
        return params;
    }

    protected static String decode(String s) throws Exception {
        StringBuffer result = new StringBuffer();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        for (int i = 0; i < s.length();) {
            char c = s.charAt(i);
            if (c == '%') {
                out.reset();
                do {
                    if (i + 2 >= s.length()) {
                        throw new IllegalArgumentException("Incomplete % sequence at " + i);
                    }
                    int d1 = Character.digit(s.charAt(i + 1), 16);
                    int d2 = Character.digit(s.charAt(i + 2), 16);
                    if (d1 == -1 || d2 == -1) {
                        throw new IllegalArgumentException("Invalid % sequence"
                                + s.substring(i, i + 3)
                                + "at "
                                + String.valueOf(i));
                    }
                    out.write((byte)((d1 << 4) + d2));
                    i += 3;
                } while (i < s.length() && s.charAt(i) == '%');
                result.append(out.toString("UTF-8"));
                continue;
            }
            result.append(c);
            i++;
        }
        return result.toString();
    }

    protected String toQueryString(Content payload, String separator) throws Exception {
        String queryString = "";
        String legalCharacters = Constants.LEGAL_CHARACTERS_IN_QUERY.replaceAll(separator, "");
        NodeList nodeList = payload.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            queryString = queryString
                    + (i > 0 ? separator : "")
                    + encode(element.getLocalName(), legalCharacters)
                    + "="
                    + encode(getText(element), legalCharacters);
        }
        return queryString;
    }

    private static String getElementValue(String elementName, Content parentElement, boolean detachChildElements) {
        Element httpURLParam = null;
        NodeList nodeList = parentElement.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            if (element.getLocalName().equals(elementName)) {
                httpURLParam = element;
                break;
            }
        }
        if (httpURLParam != null) {
            if (detachChildElements) {
                parentElement.removeChild(httpURLParam);
            }
            String text = getText(httpURLParam);
            return text == null ? "" : text;
        }
        return "";
    }

    private static String getText(Element element) {
        if (element == null || !element.hasChildNodes()) {
            return null;
        }
        try {
            Text textNode = (Text)element.getFirstChild();
            return textNode.getData();
        } catch (ClassCastException ce) {
        }
        return null;
    }

    public static class InstanceData {

        private URI requestURI;
        private Object payload; // Content or String

        public InstanceData(URI requestURI, Object payload) {
            this.requestURI = requestURI;
            this.payload = payload;
        }

        public Object getPayload() {
            return payload;
        }

        public URI getRequestURI() {
            return requestURI;
        }

    }

}
