/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.io.IOException;
import java.io.InputStream;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference.RequestActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference.ResponseActionReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference.ContentModel;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext.MessageModules;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPReceiverFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPSenderFault;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAMessageModule;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class MessageReader {

    private static final Log log = SystemContext.getContext().getSystem().getLog(MessageReader.class);

    public static Message readMessage(EndpointReference epr, InterfaceMessage am, MessageModules messageModules, InputStream in) throws HTTPFault {

        WSAMessageModule wsamm = messageModules.getModule(WSAMessageModule.class);
        EndpointActionReference ear = epr.getEndpointActionReference(wsamm.getAction());

        Message message = null;

        if (ear != null) {

            // the action is declared, i.e. it's an action with direction 'in' for
            // my service, or it's a declared action with direction 'out' for a
            // partner service, i.e. a normal partner response message or a declared
            // fault response

            if (ear instanceof RequestActionReference) {
                message = ((RequestActionReference)ear).createAction().getMessage();
            } else {
                String rt = wsamm.getRelatesTo();
                message = ((ResponseActionReference)ear).createAction(rt).getMessage();
            }

        } else {

            // it's an undeclared partner response action which was converted to
            // a system fault message by binding, i.e. content is a runtime
            // instance of system fault message. clone definition and overwrite
            // payload with parsed content

            message = ((Message)am.getRuntimeProvider()).cloneDefinition();

            Content content = null;
            try {
                content = message.parseContent(in);
            } catch (Exception ex) {
                log.error("Error reading partner response. " + ex);
                throw new HTTPReceiverFault("Error reading partner response.");
            }

            message.setContent(content);

            return message;

        }

        InterfaceAction ia = ear.getReferencedComponent().getInterfaceAction();

        Content content = null;
        if (message != null) {
            try {
                content = message.parseContent(in);
            } catch (Throwable th) {
                log.error("Error reading content. " + th);
                throw new HTTPReceiverFault("Error reading content.");
            }
        } else {
            try {
                if (in.read() > -1) {
                    throw new HTTPSenderFault("Expected an empty payload, but payload has content.");
                }
            } catch (IOException ignore) {
            }
        }

        if (ia instanceof InterfaceMessageReference) {

            // it's a normal message. wsdl 20 allows for different content
            // models which we must enforce. note that an exception was
            // thrown during validation if #other is specified

            ContentModel contentModel = ((InterfaceMessageReference)ia).getContentModel();
            if (contentModel == ContentModel.NONE) {
                if (content != null) {
                    throw new HTTPSenderFault("Expected an empty payload, but payload has content.", content);
                }
            } else if (contentModel == ContentModel.ANY) {
                if (content == null) {
                    // if root tag is malformed, woodstox throws eof exception which we
                    // interpret as an empty request
                    throw new HTTPSenderFault("ContentModel #any, but payload empty or malformed.");
                } else {
                    message.setContent(content);
                }
            }  else if (contentModel == ContentModel.OTHER) {
                if (content == null) {
                    // if root tag is malformed, woodstox throws eof exception which we
                    // interpret as an empty request
                    throw new HTTPSenderFault("ContentModel #other, but payload empty or malformed.");
                } else {
                    message.setContent(content);
                }
            } else {
                if (content == null) {
                    // if root tag is malformed, woodstox throws eof exception which we
                    // interpret as an empty request
                    throw new HTTPSenderFault("Invalid payload. Expected type "
                            + am.getName()
                            + ", but payload empty or malformed.");
                }
                QName payloadName = new QName(content.getNamespaceURI(), content.getLocalName());
                if (!payloadName.equals(am.getName())) {
                    throw new HTTPSenderFault("Invalid payload. Expected type "
                            + am.getName()
                            + ", but found type "
                            + payloadName
                            + ".", content);
                } else {
                    message.setContent(content);
                }
            }

        } else {

            // it's a fault message. content model is undefined, but is assumed to
            // be #element. note that content represents content extracted from
            // soap version specific fault detail element by binding impl

            if (content == null) {
                throw new HTTPSenderFault("Invalid fault message. Expected type "
                        + am.getName()
                        + ", but payload empty.");
            }

            QName payloadName = new QName(content.getNamespaceURI(), content.getLocalName());
            if (!payloadName.equals(am.getName())) {
                throw new HTTPSenderFault("Invalid fault message. Expected type "
                        + am.getName()
                        + ", but found type "
                        + payloadName
                        + ".", content);
            } else {
                message.setContent(content);
            }

        }

        return message;

    }

}
