/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import javax.xml.namespace.QName;

import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference.ContentModel;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentFactory;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.w3c.dom.Element;

/**
 * Writes message, i.e. because message has no envelope, validates payload against expected
 * type and content model and returns same.
 */
public class MessageWriter {

    public static Content writeMessage(EndpointActionReference ear, InterfaceMessage am, MessageContext messageContext) throws SystemFault {

        Message message = messageContext.getMessage();
        InterfaceMessageReference imr = (InterfaceMessageReference)ear.getReferencedComponent()
                .getInterfaceAction();
        ContentModel contentModel = imr.getContentModel();

        if (am == null) {

            // if the abstract message is null it's a wsdl 20 style message and
            // content model is either #none or #any or #other (it will not be null)

            if (contentModel == ContentModel.NONE) {
                if (message != null) {
                    ApplicationRT app = ear.getRootComponent().getApplication();
                    throw new SystemFault(app, "ContentModel #none, but runtime message found.");
                }
                return null;
            } else if (contentModel == ContentModel.ANY) {
                Content content = message.getContent();
                if (content == null) {
                    ApplicationRT app = ear.getRootComponent().getApplication();
                    throw new SystemFault(app, "ContentModel #any, but payload has no content.");
                }
                return importContent(content);
            } else if (contentModel == ContentModel.OTHER) {
                Content content = message.getContent();
                if (content == null) {
                    ApplicationRT app = ear.getRootComponent().getApplication();
                    throw new SystemFault(app, "ContentModel #other, but payload has no content.");
                }
                return importContent(content);
            } else {
                throw new IllegalStateException("Unexpected content model " + contentModel);
            }

        } else {

            ApplicationRT app = ear.getRootComponent().getApplication();
            Content content = message.getContent();
            if (content == null) {
                throw new SystemFault(app, "Invalid payload. Expected type "
                        + am.getName()
                        + ", but payload is empty.");
            }
            QName payloadType = new QName(content.getNamespaceURI(), content.getLocalName());
            if (!payloadType.equals(am.getName())) {
                throw new SystemFault(app, "Invalid payload. Expected type "
                        + am.getName()
                        + ", but found type "
                        + payloadType
                        + ".");
            } else {
                return importContent(content);
            }

        }
    }

    private static Content importContent(Content content) throws SystemFault {
        ContentFactory cf = (ContentFactory)new OMDOMFactory().createOMDocument();
        Element imported = null;
        try {
            imported = cf.importContent(content);
        } catch (Exception ex) {
            imported = (Element)cf.importNode(content, true);
        }
        return (Content)imported;
    }

}
