/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

/**
 * Simple pool implementation that blocks until an object is available. Only objects added to
 * pool are available to be retreived from pool, i.e. pool must be seeded with required number
 * of objects. Objects are not created by pool. Threads blocking for an available object can be
 * interrupted.
 * 
 * @param <T>
 *        Type of pooled object
 */
public class Pool<T> {

    private Stack<T> pool;
    private List<Thread> blocked;

    public Pool() {
        pool = new Stack<T>();
        blocked = new ArrayList<Thread>();
    }

    /**
     * Gets object from pool. If none available, blocks indefinitely, or until pool is
     * destroyed in which case, blocking threads throw an interrupted exception.
     * 
     * @return
     * @throws InterruptedException
     */
    public synchronized T getObject() throws InterruptedException {
        while (pool.size() == 0) {
            blocked.add(Thread.currentThread());
            this.wait();
        }
        return pool.pop();
    }

    /**
     * Used to seed the pool with desired number of objects and/or used to return a borrowed
     * object back to pool, i.e. used within finally block.
     * 
     * @param object
     */
    public synchronized void addObject(T object) {
        pool.push(object);
        this.notifyAll();
    }

    /**
     * Interrupts all blocking threads.
     */
    public synchronized void destroy() {
        for (Thread thread : blocked) {
            thread.interrupt();
        }
    }

}
