/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.util.Date;

import javax.sql.DataSource;

import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.MessageDAO;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBindingException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

/**
 * Abstracts jdbc interaction with eoa_message table. Note that transaction mgmnt is handled by
 * caller. If no transaction is in effect when DML issued, autocommit is set to true by
 * default, which implies that each stmt is considered atomic and is committed/rolled back
 * automatically.
 */

public class DerbyMessageDAO extends MessageDAO {

    private static Log log = SystemContext.getContext().getSystem().getLog(MessageDAO.class);

    private static DerbyMessageDAO singleton = null;

    private DataSource dataSource = null;

    // ****************************************************************************
    // if you change a where clause, use RuntimeStatistics on connection and check
    // the query plan to make sure that index(es) are being used appropriately.
    // always use column indexes when working with result sets rather than column
    // names, otherwise result set impl must do a lookup for each column in each
    // row returned!
    // ****************************************************************************

    private static final String CREATE_TABLE = "CREATE TABLE EOA_MESSAGE "
            + "(message_id VARCHAR(8192) NOT NULL, "
            + "my_service CHAR, "
            + "create_time BIGINT, "
            + "is_archived CHAR, "
            + "PRIMARY KEY (message_id, my_service))";

    private static final String INSERT_MESSAGE = "INSERT INTO EOA_MESSAGE "
            + "(message_id, my_service, is_archived, create_time) "
            + "VALUES (?, ?, ?, ?)";

    private DerbyMessageDAO(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public static MessageDAO getInstance(DataSource dataSource) {
        if (singleton == null) {
            singleton = new DerbyMessageDAO(dataSource);
        }
        return singleton;
    }

   /*
    * (non-Javadoc)
    * @see org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.MessageDAO#insertMessage(java.lang.String, boolean)
    */
    public void insertMessage(String messageID, boolean myService) throws SQLException {

        log.debug("insertMessage begin");

        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(INSERT_MESSAGE);
            stmt.setString(1, messageID);
            stmt.setBoolean(2, myService);
            stmt.setBoolean(3, false);
            stmt.setLong(4, new Date().getTime());
            stmt.executeUpdate();
        } catch (SQLException se) {
            if (se instanceof SQLIntegrityConstraintViolationException) {
                throw se;
            }
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = INSERT_MESSAGE;
            } else {
                stmtAsString = stmt.toString();
            }
            if (se.getCause() != null) {
                log.error("error executing insertMessage: " + stmtAsString + ". " + se.getCause());
            } else {
                log.error("error executing insertMessage: " + stmtAsString + ". " + se);
            }            
            throw se;
        } finally {
            try {
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException se) {
                log.error(se.toString());
            }
        }

        log.debug("insertMessage end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageDAO#init()
     */
    @Override
    public MessageDAO init() throws HTTPBindingException {

        ResultSet rs = null;
        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            rs = conn.getMetaData().getTables(null, null, "EOA_MESSAGE", new String[] { "TABLE" });
            if (!rs.next()) {
                log.warn("Table EOA_MESSAGE does not exist. Creating ...");
                stmt = (PreparedStatement)conn.prepareStatement(CREATE_TABLE);
                stmt.executeUpdate();
            }
        } catch (Throwable th) {
            throw new HTTPBindingException(th.getMessage());
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException ignore) {
            }
        }

        return this;

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageDAO#destroy()
     */
    @Override
    public void destroy() {
        dataSource = null;
    }

}