/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.derby;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import javax.sql.DataSource;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.Constants;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.DAOException;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.dao.MessageModuleDAO;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageModule;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext.MessageModules;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBindingException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAMessageModule;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.container.Container;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Abstracts jdbc interaction with ai_role table. Note that transaction mgmnt is handled by
 * caller using server TransactionManager. If no transaction is in effect when DML issued,
 * autocommit is set to true by default, which implies that each stmt is considered atomic and
 * is committed/rolled back automatically.
 */

public class DerbyMessageModuleDAO extends MessageModuleDAO {

    private static Log log = SystemContext.getContext().getSystem().getLog(MessageModuleDAO.class);

    private static DerbyMessageModuleDAO singleton = null;

    private DataSource dataSource = null;

    // ****************************************************************************
    // if you change a where clause, use RuntimeStatistics on connection and check
    // the query plan to make sure that index(es) are being used appropriately.
    // always use column indexes when working with result sets rather than column
    // names, otherwise result set impl must do a lookup for each column in each
    // row returned!
    // ****************************************************************************

    private static final String CREATE_TABLE = "CREATE TABLE EOA_MESSAGE_MODULE "
            + "(message_id VARCHAR(8192) NOT NULL, "
            + "my_service CHAR(1), "
            + "module_type VARCHAR(8192), "
            + "module VARCHAR(8192), "
            + "PRIMARY KEY (message_id, my_service, module_type))";

    private static final String CREATE_ARCHIVE = "CREATE TABLE EOA_MESSAGE_MODULE_ARCHIVE "
            + "(message_id VARCHAR(8192) NOT NULL, "
            + "my_service CHAR(1), "
            + "module_type VARCHAR(8192), "
            + "module VARCHAR(8192), "
            + "PRIMARY KEY (message_id, my_service, module_type))";

    private static final String INSERT_MODULE = "INSERT INTO EOA_MESSAGE_MODULE "
            + "(message_id, my_service, module_type, module) "
            + "VALUES (?, ?, ?, ?)";

    private static final String SELECT_MODULES = "SELECT message_id, my_service, module_type, module "
            + "FROM EOA_MESSAGE_MODULE "
            + "WHERE message_id = ? AND my_service = ?";

    private static final String ARCHIVE_INSERT = "INSERT INTO EOA_MESSAGE_MODULE_ARCHIVE "
            + "SELECT * "
            + "FROM EOA_MESSAGE_MODULE "
            + "WHERE message_id = ?";

    private static final String DELETE_MODULE = "DELETE FROM EOA_MESSAGE_MODULE WHERE message_id = ? AND my_service = ? AND module_type = ?";

    private static final String DELETE_MODULES = "DELETE FROM EOA_MESSAGE_MODULE WHERE message_id = ? AND my_service = ?";

    private DerbyMessageModuleDAO(DataSource dataSource) {
        super();
        this.dataSource = dataSource;
    }

    public static MessageModuleDAO getInstance(DataSource dataSource) {
        if (singleton == null) {
            singleton = new DerbyMessageModuleDAO(dataSource);
        }
        return singleton;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageModuleDAO#insertMessageModule(java.lang.String,
     *      org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageModule)
     */
    public void insertMessageModule(String messageID, boolean myService, MessageModule messageModule) throws SQLException, DAOException {

        log.debug("insertMessageModule begin");

        XMLStreamWriter writer = null;
        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(INSERT_MODULE);
            stmt.setString(1, messageID);
            stmt.setBoolean(2, myService);
            stmt.setString(3, messageModule.getType());
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            Document ownerDocument = (Document)new OMDOMFactory().createOMDocument();
            OMElement moduleElement = (OMElement)messageModule.toElement(ownerDocument);
            if (moduleElement != null) {
                writer = STAXUtils.createXMLStreamWriter(out, Constants.UTF_8);
                moduleElement.serializeAndConsume(writer);
                stmt.setString(4, new String(out.toByteArray(), Constants.UTF_8));
                stmt.executeUpdate();
            }
        } catch (SQLException se) {
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = INSERT_MODULE;
            } else {
                stmtAsString = stmt.toString();
            }
            if (se.getCause() != null) {
                log.error("error executing insertMessageModule: " + stmtAsString + ". " + se.getCause());
            } else {
                log.error("error executing insertMessageModule: " + stmtAsString + ". " + se);
            }
            throw se;
        } catch (Exception ex) {
            throw new DAOException("Error executing insertMessageModule. " + ex);
        } finally {
            try {
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
                if (writer != null) {
                    try {
                        writer.close();
                    } catch (XMLStreamException ignore) {
                    }
                }
            } catch (SQLException se) {
                log.error(se.toString());
            }
        }

        log.debug("insertMessageModule end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageModuleDAO#selectMessageModules(java.lang.String)
     */
    public MessageModules selectMessageModules(String messageID, boolean myService) throws SQLException, DAOException {

        log.debug("selectMessageModules begin");

        MessageModules messageModules = null;
        Connection conn = null;
        ResultSet rs = null;
        PreparedStatement stmt = null;
        try {

            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(SELECT_MODULES);
            stmt.setString(1, messageID);
            stmt.setBoolean(2, myService);
            rs = (ResultSet)stmt.executeQuery();

            Map<Class<?>, MessageModule> modules = null;

            while (rs.next()) {
                if (modules == null) {
                    modules = new HashMap<Class<?>, MessageModule>();
                }
                String moduleType = rs.getString(3);
                String module = rs.getString(4);
                ByteArrayInputStream in = new ByteArrayInputStream(module.getBytes(Constants.UTF_8));
                XMLStreamReader parser = STAXUtils.createXMLStreamReader(in, Constants.UTF_8);
                StAXOMBuilder builder = new StAXOMBuilder(new OMDOMFactory(), parser);
                Element element = (Element)builder.getDocumentElement();
                MessageModule messageModule = MessageModule.valueOf(moduleType, element);
                modules.put(messageModule.getClass(), messageModule);
            }

            if (modules != null) {
                WSAMessageModule wsamm = (WSAMessageModule)modules.remove(WSAMessageModule.class);
                messageModules = new MessageModules(wsamm);
                messageModules.putAll(modules);
            }

        } catch (SQLException se) {
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = SELECT_MODULES;
            } else {
                stmtAsString = stmt.toString();
            }
            if (se.getCause() != null) {
                log.error("error executing selectMessageModules: " + stmtAsString + ". " + se.getCause());
            } else {
                log.error("error executing selectMessageModules: " + stmtAsString + ". " + se);
            }
            throw se;
        } catch (Exception ex) {
            throw new DAOException("Error executing selectMessageModules. " + ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException sqlx) {
                log.error(sqlx.toString());
            }
        }

        log.debug("selectMessageModules end");
        return messageModules;

    }

    public void archiveMessageModules(String messageID) throws SQLException {

        log.debug("archiveInsert begin");

        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(ARCHIVE_INSERT);
            stmt.setString(1, messageID);
            stmt.executeUpdate();
        } catch (SQLException se) {
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = ARCHIVE_INSERT;
            } else {
                stmtAsString = stmt.toString();
            }
            log.error("error executing archiveInsert: " + stmtAsString + " " + se);
            throw se;
        } finally {
            try {
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException sqlx) {
                log.error(sqlx.toString());
            }
        }

        log.debug("archiveInsert end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageModuleDAO#deleteMessageModules(java.lang.String)
     */
    public void deleteMessageModules(String messageID, boolean myService) throws SQLException {

        log.debug("deleteModules begin");

        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(DELETE_MODULES);
            stmt.setString(1, messageID);
            stmt.setBoolean(2, myService);
            stmt.executeUpdate();
        } catch (SQLException se) {
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = DELETE_MODULES;
            } else {
                stmtAsString = stmt.toString();
            }
            if (se.getCause() != null) {
                log.error("error executing deleteModules: " + stmtAsString + ". " + se.getCause());
            } else {
                log.error("error executing deleteModules: " + stmtAsString + ". " + se);
            }
            throw se;
        } finally {
            try {
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException sqlx) {
                log.error(sqlx.toString());
            }
        }

        log.debug("deleteModules end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageModuleDAO#deleteMessageModule(java.lang.String,
     *      java.lang.String)
     */
    public void deleteMessageModule(String messageID, boolean myService, String moduleType) throws SQLException {

        log.debug("deleteModule begin");

        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = dataSource.getConnection();
            stmt = (PreparedStatement)conn.prepareStatement(DELETE_MODULE);
            stmt.setString(1, messageID);
            stmt.setBoolean(2, myService);
            stmt.setString(3, moduleType);
            stmt.executeUpdate();
        } catch (SQLException se) {
            String stmtAsString = null;
            if (stmt == null) {
                stmtAsString = DELETE_MODULE;
            } else {
                stmtAsString = stmt.toString();
            }
            if (se.getCause() != null) {
                log.error("error executing deleteModule: " + stmtAsString + ". " + se.getCause());
            } else {
                log.error("error executing deleteModule: " + stmtAsString + ". " + se);
            }
            throw se;
        } finally {
            try {
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException sqlx) {
                log.error(sqlx.toString());
            }
        }

        log.debug("deleteModule end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageModuleDAO#init()
     */
    @Override
    public MessageModuleDAO init() throws HTTPBindingException {

        Container container = SystemContext.getContext().getSystem().getContainer();
        DataSource ds = (DataSource)container.getSystemDataSource();

        ResultSet rs = null;
        Connection conn = null;
        PreparedStatement stmt = null;
        try {
            conn = ds.getConnection();
            rs = conn.getMetaData().getTables(null, null, "EOA_MESSAGE_MODULE", new String[] { "TABLE" });
            if (!rs.next()) {
                log.warn("Table EOA_MESSAGE_MODULE does not exist. Creating ...");
                stmt = (PreparedStatement)conn.prepareStatement(CREATE_TABLE);
                stmt.executeUpdate();
            }
        } catch (Throwable th) {
            throw new HTTPBindingException(th.getMessage());
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException ignore) {
            }
        }

        try {
            conn = ds.getConnection();
            rs = conn.getMetaData().getTables(null, null, "EOA_MESSAGE_MODULE_ARCHIVE", new String[] { "TABLE" });
            if (!rs.next()) {
                log.warn("Table EOA_MESSAGE_MODULE_ARCHIVE does not exist. Creating ...");
                stmt = (PreparedStatement)conn.prepareStatement(CREATE_ARCHIVE);
                stmt.executeUpdate();
            }
        } catch (Throwable th) {
            throw new HTTPBindingException(th.getMessage());
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (stmt != null) {
                    stmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException ignore) {
            }
        }

        return this;

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.binding.soap.dfault.util.dao.MessageDAO#destroy()
     */
    @Override
    public void destroy() {
        dataSource = null;
    }

}