/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueReceiver;
import javax.jms.QueueSession;
import javax.jms.Session;

import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.HTTPBindingImpl;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.Constants;
import org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.FileUtils;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class JMSReceiver implements MessageListener {

    private static Log log = SystemContext.getContext().getSystem().getLog(JMSReceiver.class);

    private QueueSession session;
    private HTTPBindingImpl binding;
    private File dataDirectory;

    public JMSReceiver(HTTPBindingImpl binding, QueueConnection connection, Queue queue, File dataDirectory)
            throws JMSException {

        this.binding = binding;
        this.dataDirectory = dataDirectory;

        // we create a separate session for each receiver. if multiple receivers
        // are created on a single session, session will serialize calls to
        // onMessage, which is not what we want, ie. we want concurrent requests.

        session = connection.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
        QueueReceiver queueReceiver = session.createReceiver(queue);
        queueReceiver.setMessageListener(this);

    }

    public void onMessage(Message message) {

        File content = null;
        InputStream in = null;

        try {

            // note that we do not use jms message id which is jms provider
            // generated. we use wsa message id

            String messageID = message.getStringProperty(Constants.MESSAGE_ID_PROPERTY);
            
            
            // are we handling a message for my service or for a partner
            // service
            
            boolean myService = message.getBooleanProperty(Constants.MY_SERVICE_PROPERTY);
            
            // using messageID retrieve file from data directory, which we
            // will stream to underlying protocol, i.e. content is parsed
            // as needed by pull parser, which handles closing input stream

            content = new File(dataDirectory, FileUtils.createFileName(messageID));

            // if file does not exist, then we assume user removed the file
            // from the file system, i.e. does not want it processed. just
            // log a debug message

            if (content.exists()) {
                in = new FileInputStream(content);
                binding.readMessage(messageID, myService, in);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("cannot bind message with id '"
                            + messageID
                            + "'. message removed from file system by user.");
                }
                return;
            }

            // message has been processed by engine (or binding if a fault was
            // returned.) so we can now safely delete the file

            if (content != null && content.exists()) {
                try {
                    in.close();
                    if (!content.delete()) {
                        String path = content.getAbsolutePath();
                        log.error("error deleting processed message " + path + " from file system");
                    }
                } catch (IOException ie) {
                    log.error("error closing file stream. " + ie);
                }
            }

        } catch (Throwable th) {

            log.error("caught throwable receiving message. " + th);

            // do not re-throw exception. we want message to be autoack'd. other-
            // wise message will be re-delivered and same error will occur re-
            // peatedly. the only errors which end up here should be file system
            // errors (readMessage method does not propogate exceptions) which
            // should be caught during testing

            if (content != null && content.exists()) {
                try {
                    in.close();
                    if (!content.delete()) {
                        String path = content.getAbsolutePath();
                        log.error("error deleting processed message " + path + " from file system");
                    }
                } catch (IOException ie) {
                    log.error("error closing file stream. " + ie);
                }
            }

        }

        // message will be autack'd by jms provider once we return. if execution
        // fails to make it this far, i.e. if server crashes before readMessage
        // method returns, message should be redelivered by jms provider

    }

    public void destroy() throws JMSException {
        session.close();
    }

}