/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.binding.http.dfault.util.jms;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.jms.JMSException;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.TemporaryQueue;

import org.apache.commons.pool.BasePoolableObjectFactory;
import org.apache.commons.pool.PoolableObjectFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class JMSRequestorFactory extends BasePoolableObjectFactory implements PoolableObjectFactory  {

    private QueueConnection connection;
    private Map<String, Queue> temporaryQueues;

    public JMSRequestorFactory(QueueConnection connection) {
        this.connection = connection;
        this.temporaryQueues = Collections.synchronizedMap(new HashMap<String, Queue>());
    }
    
    /**
     * Returns physical temp queue mapped to name.
     * 
     * @param queueName
     * @return queue or <code>null</code> if queue is undefined, i.e. requestor was garbage
     *         collected and queue was destroyed
     */
    public Queue getTemporaryQueue(String queueName) {
        return temporaryQueues.get(queueName);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.commons.pool.BasePoolableObjectFactory#destroyObject(java.lang.Object)
     */
    @Override
    public void destroyObject(Object object) throws Exception {
        try {
            TemporaryQueue q = ((JMSRequestor)object).getTemporaryQueue();
            temporaryQueues.remove(q.getQueueName());
            ((JMSRequestor)object).destroy();
        } catch (JMSException je) {
            Log log = SystemContext.getContext().getSystem().getLog(JMSRequestorFactory.class);
            log.error(je.toString());
        }
    }

    /*
     * (non-Javadoc)
     * @see org.apache.commons.pool.BasePoolableObjectFactory#makeObject()
     */
    @Override
    public Object makeObject() throws Exception {
        JMSRequestor requestor = new JMSRequestor(this, connection);
        TemporaryQueue tq = requestor.getTemporaryQueue();
        temporaryQueues.put(tq.getQueueName(), tq);
        return requestor;
    }

}