/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util;

import java.security.MessageDigest;

import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.MessageDigestPassword;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPAuthenticationResponse;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureFactory;
import org.mortbay.log.Log;
import org.mortbay.util.StringUtil;
import org.mortbay.util.TypeUtil;

/**
 * Adapted from private member org.mortbay.jetty.security.DigestAuthenticator$Digest. Modifed
 * to use {@link MessageDigestPassword} as required by spec.
 */
public class HTTPDigestResponse implements HTTPAuthenticationResponse {

    private HTTPServerFeatureFactory factory;
    
    private String method = null;
    private String username = null;
    private String realm = null;
    private String nonce = null;
    private String nc = null;
    private String cnonce = null;
    private String qop = null;
    private String uri = null;
    private String response = null;
    
    public HTTPDigestResponse(HTTPServerFeatureFactory factory) {
        this.factory = factory;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPAuthenticationResponse#check(java.lang.Object)
     */
    public boolean check(Object password) {
        
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        
        try {
            
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());

            MessageDigest md = MessageDigest.getInstance("MD5");
            String ha1;

            // calc A1 digest
            if (password instanceof MessageDigestPassword) {
                // digest must be hashed as user:realm:password
                ha1 = TypeUtil.toString(((MessageDigestPassword)password).getDigest(), 16);
            } else {
                // password is assumed to be clear clear text
                md.update(username.getBytes(StringUtil.__ISO_8859_1));
                md.update((byte)':');
                md.update(realm.getBytes(StringUtil.__ISO_8859_1));
                md.update((byte)':');
                md.update(password.toString().getBytes(StringUtil.__ISO_8859_1));
                ha1 = TypeUtil.toString(md.digest(), 16);
            }

            // calc A2 digest
            md.reset();
            md.update(method.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(uri.getBytes(StringUtil.__ISO_8859_1));
            String ha2 = TypeUtil.toString(md.digest(), 16);

            // calc digest
            md.update(ha1.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(nonce.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(nc.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(cnonce.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(qop.getBytes(StringUtil.__ISO_8859_1));
            md.update((byte)':');
            md.update(ha2.getBytes(StringUtil.__ISO_8859_1));
            String digest = TypeUtil.toString(md.digest(), 16);

            // check digest
            return (digest.equalsIgnoreCase(response));

        } catch (Exception e) {
            Log.warn(e.toString());
        } finally {
            thread.setContextClassLoader(cl);
        }

        return false;

    }

    public void setCnonce(String cnonce) {
        this.cnonce = cnonce;
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public void setNc(String nc) {
        this.nc = nc;
    }

    public void setNonce(String nonce) {
        this.nonce = nonce;
    }

    public void setQop(String qop) {
        this.qop = qop;
    }

    public void setRealm(String realm) {
        this.realm = realm;
    }

    public void setResponse(String response) {
        this.response = response;
    }

    public void setUri(String uri) {
        this.uri = uri;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getCnonce() {
        return cnonce;
    }

    public String getMethod() {
        return method;
    }

    public String getNc() {
        return nc;
    }

    public String getNonce() {
        return nonce;
    }

    public String getQop() {
        return qop;
    }

    public String getRealm() {
        return realm;
    }

    public String getResponse() {
        return response;
    }

    public String getUri() {
        return uri;
    }

    public String getUser() {
        return username;
    }

}
