/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util;

import java.security.Principal;

import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginContext;

import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginCallbackHandler;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginContextFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPAuthenticationResponse;
import org.mortbay.jetty.Request;
import org.mortbay.jetty.plus.jaas.JAASUserRealm;
import org.mortbay.log.Log;

/**
 * Overrides behavior within jetty's JAASUserRealm by allowing us to use our callback handler
 * implementation which sets name of realm as a callback. Realm name is used by our login
 * module impl, i.e. configured by jaas-login-context feature, to retrieve credentials for user
 * within context of indicated realm.
 */
public class UserRealmImpl extends JAASUserRealm {

    private String configurationName = "other";

    public UserRealmImpl(String realmName, String configurationName) {
        this.realmName = realmName;
        if (configurationName != null) {
            this.configurationName = configurationName;
        }
        Server server = SystemContext.getContext().getSystem().getServer();
        // verify that login context feature is enabled. otherwise we get cryptic
        // jaas warning 'Unable to locate a login configuration' when attempting
        // to authenticate
        if (server.getFeature(LoginContextFeature.class) == null) {
            throw new IllegalStateException("JAAS authentication requires that optional feature "
                    + LoginContextFeature.TYPE
                    + " be enabled.");
        }
    }

    @Override
    public Principal authenticate(String username, Object credentials, Request request) {
        try {
            CallbackHandler cbh = new LoginCallbackHandler((HTTPAuthenticationResponse)credentials);
            LoginContext loginContext = new LoginContext(configurationName, cbh);
            loginContext.login();
            PrincipalImpl userPrincipal = new PrincipalImpl(this, username);
            userPrincipal.setSubject(loginContext.getSubject());
            userPrincipal.setRoleCheckPolicy(roleCheckPolicy);
            userPrincipal.setLoginContext(loginContext);
            return userPrincipal;
        } catch (Exception loginFailure) {
            Log.warn(loginFailure.toString());
            Log.debug(loginFailure);
            return null;
        }
    }

}
