/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.Principal;
import java.security.acl.Group;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import javax.servlet.Servlet;
import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.BasicAuthenticatorImpl;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.Constants;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.DigestAuthenticatorImpl;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.ErrorHandlerImpl;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.JSPExtClassLoader;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util.UserRealmImpl;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.AuthInfo;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.JettyConfig;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.RequestLog;
import org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.ThreadPool;
import org.bluestemsoftware.specification.eoa.DeploymentClassLoader;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentDependencies;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ScopedDependency;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature.Provider;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature.ServerAuthInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureException.UniquePathException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPAuthenticationScheme;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.mortbay.jetty.AbstractConnector;
import org.mortbay.jetty.Connector;
import org.mortbay.jetty.NCSARequestLog;
import org.mortbay.jetty.Server;
import org.mortbay.jetty.handler.ContextHandlerCollection;
import org.mortbay.jetty.handler.RequestLogHandler;
import org.mortbay.jetty.plus.jaas.RoleCheckPolicy;
import org.mortbay.jetty.security.Constraint;
import org.mortbay.jetty.security.ConstraintMapping;
import org.mortbay.jetty.security.SecurityHandler;
import org.mortbay.jetty.security.SslSocketConnector;
import org.mortbay.jetty.security.UserRealm;
import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.ServletHolder;
import org.mortbay.jetty.webapp.WebAppClassLoader;
import org.mortbay.jetty.webapp.WebAppContext;
import org.mortbay.thread.BoundedThreadPool;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class HTTPServerFeatureImpl implements Provider {

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPServerFeature.class);
    private DeploymentContext deploymentContext;
    private Server server = null;
    private HTTPServerFeature consumer = null;
    private HTTPServerFeatureFactory factory;
    private ArrayList<Connector> connectors = new ArrayList<Connector>();
    private ContextHandlerCollection contexts;
    private Map<String, Servlet> servletMappings = new HashMap<String, Servlet>();
    private Map<String, Context> contextsByServletName = new HashMap<String, Context>();
    private String jaasConfiguration = "other";
    private int[] ports;
    private int[] sslPorts;

    public static final String IMPL = HTTPServerFeatureImpl.class.getName();

    public static final Map<String, String> EMPTY_PARMS = new HashMap<String, String>();
    public static final Map<String, Object> EMPTY_ATTS = new HashMap<String, Object>();

    /*
     * (non-Javadoc)
     * 
     * @seeorg.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.
     * bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (HTTPServerFeature)consumer;
        this.factory = (HTTPServerFeatureFactory)this.consumer.getExtensionFactory();
        this.deploymentContext = factory.getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c
     * .dom.Element)
     */
    public void spi_init(Set<ManagementContext> managementContexts) throws FeatureException {

        log.debug("init begin");

        // create container
        server = new Server();
        
        Element configuration = null;
        File extensionEtcDir = ((Feature)consumer).getExtensionEtcDir();
        File configurationFile = new File(extensionEtcDir, "config.xml");
        if (configurationFile.exists()) {
            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                Document configurationDoc = documentBuilder.parse(configurationFile);
                configuration = configurationDoc.getDocumentElement();
            } catch (Exception ex) {
                throw new FeatureException("Error loading configuration. " + ex);
            }
        }

        // if configuration is null, retrieve default configuration contained within
        // deployment and persist it

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.http.server.jetty.config.1.0.xml";
            Writer writer = null;
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
                writer = new FileWriter(configurationFile);
                DOMSerializer.serializeNode(configuration, writer, "UTF-8", true);
            } catch (Exception ex) {
                throw new FeatureException("Error retrieving default configuration. " + ex);
            } finally {
                if (writer != null) {
                    try {
                        writer.close();
                    } catch (IOException ignore) {
                    }
                }
            }
        }

        // note we do not define dependencies on spring xbean for parsing
        // config. because webapp loaders depend upon our deployment loader
        // for jsp parsing, we do not want to conflict with spring jars
        // which web apps may define as dependencies

        JettyConfig jettyConfig = new JettyConfig();

        validateConfiguration(new DOMSource(configuration));

        QName childName = new QName(Constants.FEATURE_SCHEMA_NS, "authInfo");
        Element authInfoElement = DOMUtils.getChildElement(configuration, childName);
        if (authInfoElement != null) {
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "jaasConfiguration");
            Element jaasConfigurationElement = DOMUtils.getChildElement(authInfoElement, childName);
            AuthInfo authInfo = new AuthInfo();
            authInfo.setJaasConfiguration(DOMUtils.getText(jaasConfigurationElement));
            jettyConfig.setAuthInfo(authInfo);
            configureAuthInfo(jettyConfig);
        }

        childName = new QName(Constants.FEATURE_SCHEMA_NS, "threadPool");
        Element threadPoolElement = DOMUtils.getChildElement(configuration, childName);
        if (threadPoolElement != null) {
            ThreadPool threadPool = new ThreadPool();
            jettyConfig.setThreadPool(threadPool);
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "minThreads");
            Element minThreadsElement = DOMUtils.getChildElement(threadPoolElement, childName);
            if (minThreadsElement != null) {
                threadPool.setMinThreads(Integer.parseInt(DOMUtils.getText(minThreadsElement)));
            }
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "lowThreads");
            Element lowThreadsElement = DOMUtils.getChildElement(threadPoolElement, childName);
            if (lowThreadsElement != null) {
                threadPool.setLowThreads(Integer.parseInt(DOMUtils.getText(lowThreadsElement)));
            }
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "maxThreads");
            Element maxThreadsElement = DOMUtils.getChildElement(threadPoolElement, childName);
            if (maxThreadsElement != null) {
                threadPool.setMaxThreads(Integer.parseInt(DOMUtils.getText(maxThreadsElement)));
            }
            configureThreadPool(jettyConfig);
        }

        childName = new QName(Constants.FEATURE_SCHEMA_NS, "connectors");
        Element connectorsElement = DOMUtils.getChildElement(configuration, childName);
        if (connectorsElement != null) {
            List<org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.Connector> connectors;
            connectors = new ArrayList<org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.Connector>();
            jettyConfig.setConnectors(connectors);
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "connector");
            List<Element> connectorElements = DOMUtils.getChildElements(connectorsElement, childName);
            org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.Connector c;
            for (Element connectorElement : connectorElements) {
                c = new org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.Connector();
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "className");
                Element classNameElement = DOMUtils.getChildElement(connectorElement, childName);
                if (classNameElement != null) {
                    c.setClassName(DOMUtils.getText(classNameElement));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "port");
                Element portElement = DOMUtils.getChildElement(connectorElement, childName);
                if (portElement != null) {
                    c.setPort(Integer.parseInt(DOMUtils.getText(portElement)));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "maxIdleTime");
                Element maxIdleTimeElement = DOMUtils.getChildElement(connectorElement, childName);
                if (maxIdleTimeElement != null) {
                    c.setMaxIdleTime(Integer.parseInt(DOMUtils.getText(maxIdleTimeElement)));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "lowResourcesConnections");
                Element lowResourcesConnectionsElement = DOMUtils.getChildElement(connectorElement, childName);
                if (lowResourcesConnectionsElement != null) {
                    c.setLowResourcesConnections(Integer
                            .parseInt(DOMUtils.getText(lowResourcesConnectionsElement)));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "lowResourceMaxIdleTime");
                Element lowResourceMaxIdleTimeElement = DOMUtils.getChildElement(connectorElement, childName);
                if (lowResourceMaxIdleTimeElement != null) {
                    c.setLowResourceMaxIdleTime(Integer.parseInt(DOMUtils.getText(lowResourceMaxIdleTimeElement)));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "acceptors");
                Element acceptorsElement = DOMUtils.getChildElement(connectorElement, childName);
                if (acceptorsElement != null) {
                    c.setAcceptors(Integer.parseInt(DOMUtils.getText(acceptorsElement)));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "keystore");
                Element keystoreElement = DOMUtils.getChildElement(connectorElement, childName);
                if (keystoreElement != null) {
                    c.setKeystore(DOMUtils.getText(keystoreElement));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "password");
                Element passwordElement = DOMUtils.getChildElement(connectorElement, childName);
                if (passwordElement != null) {
                    c.setPassword(DOMUtils.getText(passwordElement));
                }
                childName = new QName(Constants.FEATURE_SCHEMA_NS, "keyPassword");
                Element keyPasswordElement = DOMUtils.getChildElement(connectorElement, childName);
                if (keyPasswordElement != null) {
                    c.setKeyPassword(DOMUtils.getText(keyPasswordElement));
                }
                connectors.add(c);
            }
            configureConnectors(jettyConfig);
        }

        childName = new QName(Constants.FEATURE_SCHEMA_NS, "requestLog");
        Element requestLogElement = DOMUtils.getChildElement(configuration, childName);
        if (requestLogElement != null) {
            RequestLog requestLog = new RequestLog();
            jettyConfig.setRequestLog(requestLog);
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "retainDays");
            Element retainDaysElement = DOMUtils.getChildElement(requestLogElement, childName);
            if (retainDaysElement != null) {
                requestLog.setRetainDays(Integer.parseInt(DOMUtils.getText(retainDaysElement)));
            }
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "append");
            Element appendElement = DOMUtils.getChildElement(requestLogElement, childName);
            if (appendElement != null) {
                requestLog.setAppend(Boolean.parseBoolean((DOMUtils.getText(appendElement))));
            }
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "extended");
            Element extendedElement = DOMUtils.getChildElement(requestLogElement, childName);
            if (extendedElement != null) {
                requestLog.setExtended(Boolean.parseBoolean((DOMUtils.getText(extendedElement))));
            }
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "logTimeZone");
            Element logTimeZoneElement = DOMUtils.getChildElement(requestLogElement, childName);
            if (logTimeZoneElement != null) {
                requestLog.setLogTimeZone(DOMUtils.getText(logTimeZoneElement));
            }
            configureRequestLog(jettyConfig);
        }

        // create a container for contexts, i.e. we create one ctxt
        // per added servlet or webapp to allow for partitioning of
        // security handlers
        contexts = new ContextHandlerCollection();
        server.addHandler(contexts);

        // for added security, suppress sending jetty advertisement
        server.setSendServerVersion(false);

        // set system property which will be used by jetty to parse
        // context paths. note that we override the default which
        // includes the ':' character as a separator. because eoa spec
        // may use xpath style namespace declarations within address,
        // a tokenizer configured with ':' as a separator would break
        // address path semantics. remove ':' and just use ','. i
        // don't believe context path lists are used anyway

        System.setProperty("org.mortbay.http.PathMap.separators", ",");

        // start server which will begin request processing
        try {
            server.start();
        } catch (Exception ex) {
            throw new HTTPServerFeatureException(ex);
        }

        log.debug("init end");

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public synchronized void spi_destroy() {
        log.debug("destroy begin");
        servletMappings.clear();
        connectors.clear();
        if (server != null) {
            try {
                server.stop();
            } catch (Exception ex) {
                log.error("error stopping jetty server. " + ex);
            }
        }
        log.debug("embedded server shutdown");
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature$Provider
     * #spi_addServlet(javax.servlet.Servlet, java.lang.String)
     */
    public synchronized void spi_addServlet(Servlet servlet, String contextPath, ServerAuthInfo authInfo) throws HTTPServerFeatureException {
        spi_addServlet(servlet, contextPath, authInfo, EMPTY_PARMS, EMPTY_ATTS);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature$Provider
     * #spi_addServlet(javax.servlet.Servlet, java.lang.String,
     * org.bluestemsoftware.specification
     * .eoa.ext.feature.http.server.HTTPServerFeature.ServerAuthInfo, java.util.Map,
     * java.util.Map)
     */
    public synchronized void spi_addServlet(Servlet servlet, String contextPath, ServerAuthInfo authInfo, Map<String, String> parameters, Map<String, Object> attributes) throws HTTPServerFeatureException {

        // the context path will be matched against path of incoming uri. any
        // uri whose path begins with context path is considered a match, i.e.
        // the implied context path is '/mypath/*'

        if (servletMappings.containsKey(contextPath)) {
            String servletClass = servletMappings.get(contextPath).getClass().getName();
            throw new UniquePathException("Error adding servlet. Path '"
                    + contextPath
                    + "' already mapped to servlet "
                    + servletClass);
        } else {
            servletMappings.put(contextPath, servlet);
        }

        // each servlet has its own context. in servlet parlance, a context
        // corresponds to a 'web application'. this allows us to configure
        // a security handler per endpoint, i.e. each endpoint is a web app

        Context context = new Context();
        context.setErrorHandler(new ErrorHandlerImpl());
        context.setContextPath(contextPath);
        ServletHolder servletHolder = new ServletHolder(servlet);
        for (Map.Entry<String, String> parameter : parameters.entrySet()) {
            servletHolder.setInitParameter(parameter.getKey(), parameter.getValue());
        }
        for (Map.Entry<String, Object> attribute : attributes.entrySet()) {
            context.setAttribute(attribute.getKey(), attribute.getValue());
        }

        // because 'web application' has only one servlet, the implied
        // servlet path spec is '/*'

        context.addServlet(servletHolder, "/*");
        contexts.addHandler(context);

        if (authInfo != null) {

            SecurityHandler securityHandler = new SecurityHandler();

            Constraint constraint = null;
            if (authInfo.getScheme() == HTTPAuthenticationScheme.BASIC) {
                securityHandler.setAuthMethod(Constraint.__BASIC_AUTH);
                securityHandler.setAuthenticator(new BasicAuthenticatorImpl());
                constraint = new Constraint(Constraint.__BASIC_AUTH, Constraint.ANY_ROLE);
                constraint.setAuthenticate(true);
            } else {
                securityHandler.setAuthMethod(Constraint.__DIGEST_AUTH);
                securityHandler.setAuthenticator(new DigestAuthenticatorImpl(factory));
                constraint = new Constraint(Constraint.__DIGEST_AUTH, Constraint.ANY_ROLE);
                constraint.setAuthenticate(true);
            }

            ConstraintMapping cm = new ConstraintMapping();
            cm.setConstraint(constraint);
            cm.setPathSpec("/*");

            // configure security handler with our jaas user realm impl.
            // see comments there

            context.addHandler(securityHandler);
            UserRealm userRealm = new UserRealmImpl(authInfo.getRealm(), jaasConfiguration);
            securityHandler.setUserRealm(userRealm);
            securityHandler.setConstraintMappings(new ConstraintMapping[] { cm });

        }

        try {
            context.start();
        } catch (Exception ex) {
            throw new HTTPServerFeatureException("Error adding servlet. " + ex);
        }

        contextsByServletName.put(servlet.getServletConfig().getServletName(), context);

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature$Provider
     * #spi_removeServlet(javax.servlet.Servlet)
     */
    public synchronized void spi_removeServlet(Servlet servlet) {
        String servletName = servlet.getServletConfig().getServletName();
        Context context = contextsByServletName.get(servletName);
        if (context != null) {
            try {
                context.stop();
            } catch (Exception ex) {
                log.error("error removing servlet. " + ex.getMessage());
            }
            contextsByServletName.remove(servletName);
            contexts.removeHandler(context);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature$Provider
     * #spi_deployWebModule(java.io.File,
     * org.bluestemsoftware.specification.eoa.ApplicationClassLoader, java.lang.String)
     */
    public void spi_deployWebModule(File webArchive, ClassLoader classLoader, String rootContext) throws HTTPServerFeatureException {
        
        if (log.isDebugEnabled()) {
            log.debug("deploying web module to rootContext " + rootContext);
        }
        
        // note that as of 6.1, jetty swallows configuration, servlet initialization
        // errors, i.e. rather than rethrowing. although web app is not deployed,
        // we have no way knowing, i.e. we can't through a deployment exception, which
        // means server will start successfully. this shouldn't affect message flow
        // in anyway, so i guess this is o.k. (check cvs revision 3/6/08 for code
        // that tried to extend webappcontext so that we could catch exceptions. it
        // got ugly)

        WebAppContext wac = new WebAppContext();
        wac.setContextPath(rootContext);
        wac.setExtractWAR(true); // jetty docs recommend, i.e. if it has uncompiled jsp's
        wac.setParentLoaderPriority(false); // use servlet spec web app classloading priority
        try {
            wac.setWar(webArchive.toURI().toURL().toString());
        } catch (MalformedURLException me) {
            throw new HTTPServerFeatureException("Error processing war path. " + me.getMessage());
        }
        
        // if webapp load contains duplicate artifacts (actually classes, but it's
        // impossible to check at that level of granularity,)  we will get class
        // loading related errors, e.g. ClassCastException, etc ... so try and
        // catch it now
        
        DeploymentDependencies myDD = factory.getFactoryContext().getDeployment().getDependencies();
        if (classLoader instanceof URLClassLoader) {
            URLClassLoader uc = (URLClassLoader)classLoader;
            for (ScopedDependency scopedDependency : myDD.getScopedDependencies())    {
                String ref = scopedDependency.getOrganizationID();
                ref = ref.replace(".", "/");
                ref = ref + "/" + scopedDependency.getArtifactID();
                try {
                    ref = new URI(ref).getPath(); // re-orient the slashes
                } catch (Exception ex) {
                    throw new HTTPServerFeatureException("Error creating war classloader. " + ex.getMessage());
                } 
                while (uc.getParent() != null && uc.getParent() instanceof URLClassLoader) {
                    for (URL url : uc.getURLs()) {
                        String path = url.getPath();
                        if (path.contains(ref)) {
                            log.warn("Classloader "
                                    + System.getProperty("line.separator")
                                    + uc.toString()
                                    + "contains URL '"
                                    + path
                                    + "' which MAY conflict with an artifact on my classpath.");
                            break;
                        }                    
                    }
                    uc = (URLClassLoader)uc.getParent();
                }
            }
        }
        
        WebAppClassLoader webAppClassLoader;
        try {
            DeploymentClassLoader dcl = deploymentContext.getClassLoader();
            ClassLoader parent = new JSPExtClassLoader(dcl, classLoader);
            webAppClassLoader = new WebAppClassLoader(parent, wac);
        } catch (IOException ie) {
            throw new HTTPServerFeatureException("Error creating war classloader. " + ie.getMessage());
        }
        wac.setClassLoader(webAppClassLoader);
        contexts.addHandler(wac);
        try {
            wac.start();
        } catch (Exception e) {
            try {
                contexts.removeHandler(wac);
            } catch (Exception ignore) {
            }
            throw new HTTPServerFeatureException("Module failed to deploy. " + e.getMessage());
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature$Provider
     * #spi_getPort(boolean)
     */
    public int[] spi_getPorts(boolean isSSL) {
        if (isSSL) {
            return sslPorts;
        } else {
            return ports;
        }
    }

    private void configureAuthInfo(JettyConfig config) {
        AuthInfo authInfo = config.getAuthInfo();
        jaasConfiguration = authInfo.getJaasConfiguration();
    }

    private void configureThreadPool(JettyConfig config) {
        ThreadPool threadPoolBean = config.getThreadPool();
        threadPoolBean = threadPoolBean == null ? new ThreadPool() : threadPoolBean;
        BoundedThreadPool threadPool = new BoundedThreadPool();
        threadPool.setMaxThreads(threadPoolBean.getMaxThreads());
        threadPool.setLowThreads(threadPoolBean.getLowThreads());
        threadPool.setMinThreads(threadPoolBean.getMinThreads());
        server.setThreadPool(threadPool);
    }

    private void configureRequestLog(JettyConfig config) throws FeatureException {
        RequestLog requestLogBean = config.getRequestLog();
        if (requestLogBean == null) {
            return;
        }
        int logRetainDays = requestLogBean.getRetainDays();
        boolean logAppend = requestLogBean.isAppend();
        boolean logExtended = requestLogBean.isExtended();
        File temp = new File(((Feature)consumer).getExtensionVarDir(), "yyyy_mm_dd.jetty.request.log");
        String logLocation = temp.getAbsolutePath();
        String logTimeZone = null;
        if (requestLogBean.getLogTimeZone() == null) {
            logTimeZone = TimeZone.getDefault().getID();
        } else {
            logTimeZone = requestLogBean.getLogTimeZone();
        }
        RequestLogHandler requestLogHandler = new RequestLogHandler();
        NCSARequestLog requestLog = new NCSARequestLog(logLocation);
        requestLogHandler.setRequestLog(requestLog);
        requestLog.setExtended(logExtended);
        requestLog.setAppend(logAppend);
        requestLog.setLogTimeZone(logTimeZone);
        requestLog.setRetainDays(logRetainDays);
        server.addHandler(requestLogHandler);
    }

    private void configureConnectors(JettyConfig config) throws FeatureException {

        int portCount = 0;
        int sslPortCount = 0;

        for (org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.xbean.Connector xbean : config
                .getConnectors()) {
            AbstractConnector connector = null;
            String className = xbean.getClassName();
            try {
                Class<?> clazz = deploymentContext.getClassLoader().loadClass(className);
                connector = (AbstractConnector)clazz.newInstance();
            } catch (Exception ex) {
                throw new HTTPServerFeatureException("Error instantiating connector '" + className + "'. " + ex);
            }
            int connectorPort = xbean.getPort();
            if (connectorPort < 1) {
                throw new HTTPServerFeatureException("Invalid port "
                        + connectorPort
                        + " defined for connector "
                        + className
                        + ".");
            }
            connector.setPort(connectorPort);
            connector.setMaxIdleTime(xbean.getMaxIdleTime());
            connector.setLowResourceMaxIdleTime(xbean.getLowResourceMaxIdleTime());
            connector.setAcceptors(xbean.getAcceptors());
            if (connector instanceof SslSocketConnector) {
                ((SslSocketConnector)connector).setKeystore(xbean.getKeystore());
                ((SslSocketConnector)connector).setPassword(xbean.getPassword());
                ((SslSocketConnector)connector).setKeyPassword(xbean.getKeyPassword());
                sslPortCount++;
            } else {
                portCount++;
            }
            connectors.add(connector);
        }

        ports = new int[portCount];
        sslPorts = new int[sslPortCount];

        int i = 0;
        for (Connector connector : connectors) {
            if (connector instanceof SslSocketConnector) {
                sslPorts[i] = connector.getPort();
            }
        }
        i = 0;
        for (Connector connector : connectors) {
            if (!(connector instanceof SslSocketConnector)) {
                ports[i] = connector.getPort();
            }
        }

        server.setConnectors(connectors.toArray(new Connector[] {}));
    }

    /*
     * Roles are used for declarative security within webapps and ejb's, i.e. for authorizing
     * access to specific methods - not authentication. Authorization is acheived via
     * ws-security. Consequently, this class authorizes everyone.
     */
    static class NoRoleCheckPolicy implements RoleCheckPolicy {

        public boolean checkRole(String roleName, Principal runAsRole, Group roles) {
            return true;
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl
     * ()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

    private void validateConfiguration(DOMSource domSource) throws FeatureException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource(Constants.FEATURE_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new FeatureException("Error validating configuration. " + de);
        }

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = javax.xml.validation.SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new FeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new FeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new FeatureException("Configuration invalid. " + validationError);
        }

    }

}
