/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util;

import java.io.IOException;
import java.security.Principal;

import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.mortbay.jetty.HttpHeaders;
import org.mortbay.jetty.Request;
import org.mortbay.jetty.Response;
import org.mortbay.jetty.security.Constraint;
import org.mortbay.jetty.security.DigestAuthenticator;
import org.mortbay.jetty.security.UserRealm;
import org.mortbay.util.QuotedStringTokenizer;

public class DigestAuthenticatorImpl extends DigestAuthenticator {

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(DigestAuthenticatorImpl.class);

    private HTTPServerFeatureFactory factory;

    public DigestAuthenticatorImpl(HTTPServerFeatureFactory factory) {
        this.factory = factory;
    }

    /*
     * Copied implementation from super and modified to use HTTPDigestResponse implementation
     * instead of private member super$Digest.
     * 
     * (non-Javadoc)
     * @see org.mortbay.jetty.security.DigestAuthenticator#authenticate(org.mortbay.jetty.security.UserRealm,
     *      java.lang.String, org.mortbay.jetty.Request, org.mortbay.jetty.Response)
     */
    @Override
    public Principal authenticate(UserRealm realm, String pathInContext, Request request, Response response) throws IOException {

        boolean stale = false;
        Principal user = null;
        String credentials = request.getHeader(HttpHeaders.AUTHORIZATION);

        if (credentials != null) {

            QuotedStringTokenizer tokenizer = new QuotedStringTokenizer(credentials, "=, ", true, false);

            HTTPDigestResponse digest = new HTTPDigestResponse(factory);
            digest.setMethod(request.getMethod());

            String last = null;
            String name = null;

            while (tokenizer.hasMoreTokens()) {
                String tok = tokenizer.nextToken();
                char c = (tok.length() == 1) ? tok.charAt(0) : '\0';
                switch (c) {
                    case '=':
                        name = last;
                        last = tok;
                        break;
                    case ',':
                        name = null;
                    case ' ':
                        break;

                    default:
                        last = tok;
                        if (name != null) {
                            if ("username".equalsIgnoreCase(name))
                                digest.setUsername(tok);
                            else if ("realm".equalsIgnoreCase(name))
                                digest.setRealm(tok);
                            else if ("nonce".equalsIgnoreCase(name))
                                digest.setNonce(tok);
                            else if ("nc".equalsIgnoreCase(name))
                                digest.setNc(tok);
                            else if ("cnonce".equalsIgnoreCase(name))
                                digest.setCnonce(tok);
                            else if ("qop".equalsIgnoreCase(name))
                                digest.setQop(tok);
                            else if ("uri".equalsIgnoreCase(name))
                                digest.setUri(tok);
                            else if ("response".equalsIgnoreCase(name))
                                digest.setResponse(tok);
                            break;
                        }
                }
            }

            int n = checkNonce(digest.getNonce(), request);
            if (n > 0) {
                user = realm.authenticate(digest.getUser(), digest, request);
            } else if (n == 0) {
                stale = true;
            }

            if (user == null) {
                if (log.isDebugEnabled()) {
                    log.debug("AUTH FAILURE: realm: " + realm.getName() + " user: " + digest.getUser());
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("credentials: " + credentials);
                }
                request.setAuthType(Constraint.__DIGEST_AUTH);
                request.setUserPrincipal(user);
            }
        }

        if (user == null && response != null) {
            sendChallenge(realm, request, response, stale);
        }

        return user;

    }

}
