/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.server.jetty.util;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;

import org.bluestemsoftware.specification.eoa.DeploymentClassLoader;

/**
 * Delegates to HTTPServerFeature deployment classloader. This is necessary because jetty
 * assumes its jars, including the jsp 2.1 jar, are artifacts on the java system classloader
 * which is typically the parent classloader of jetty's WebAppClassLoader.
 * <p>
 * Within our classloading architecture, this is not the case, i.e. parent is the EOA
 * ApplicationClassLoader which cannot see the jetty jars (which is a good thing). In some
 * situations, however, the jetty classes may be required to compile jsp's, e.g. jetty defines
 * an index_jsp class that gets loaded on the fly. The generated servlet extends from and
 * references classes within jsp-2.1-x.x.x.jar which is scoped to the
 * HTTPServerFeatureDeployment ...
 */
public class JSPExtClassLoader extends URLClassLoader {

    private DeploymentClassLoader jettyClassLoader;

    public JSPExtClassLoader(DeploymentClassLoader jettyClassLoader, ClassLoader parent) {
        super(new URL[]{}, parent);
        this.jettyClassLoader = jettyClassLoader;
    }

    @Override
    public synchronized URL getResource(String name) {
        URL url = super.getResource(name);
        if (url == null) {
            url = jettyClassLoader.getResource(name);
        }
        return url;
    }

    @Override
    public synchronized Class<?> loadClass(String name) throws ClassNotFoundException {
        try {
            return super.loadClass(name);
        } catch (ClassNotFoundException cfe) {
            return jettyClassLoader.loadClass(name);
        }
    }

    @Override
    public synchronized InputStream getResourceAsStream(String name) {
        InputStream in = super.getResourceAsStream(name);
        if (in == null) {
            in = jettyClassLoader.getResourceAsStream(name);
        }
        return in;
    }

    @Override
    public synchronized Enumeration<URL> getResources(String name) throws IOException {
        Enumeration<URL> urls = super.getResources(name);
        if (urls == null) {
            urls = jettyClassLoader.getResources(name);
        }
        return urls;
    }
    
    /*
     * (non-Javadoc)
     * @see java.net.URLClassLoader#getURLs()
     */
    @Override
    public URL[] getURLs() {
        URL[] a = jettyClassLoader.getURLs();
        URL[] b = super.getURLs();
        URL[] combined = new URL[a.length + b.length];
        System.arraycopy(a, 0, combined, 0, a.length);
        System.arraycopy(b, 0, combined, a.length, b.length);
        return combined;
    }

}
