/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.NotificationFilterSupport;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.management.remote.JMXAuthenticator;
import javax.management.remote.JMXConnectionNotification;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;

import org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.mgmt.JMXServer;
import org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDeployment;
import org.bluestemsoftware.specification.eoa.ext.ManageableExtension;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginContextFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.rmi.RMIRegistryFeature;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.container.Container;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;

public final class JMXServerFeatureImpl implements JMXServerFeature.Provider {

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(JMXServerFeature.class);

    public static final String IMPL = JMXServerFeatureImpl.class.getName();

    private JMXServerFeature consumer;
    private MBeanServer mbeanServer;
    private JMXConnectorServer connectorServer;
    private JMXServer jmxServer = new JMXServer();
    private List<ObjectName> registeredMBeans = Collections.synchronizedList(new ArrayList<ObjectName>());
    private Map<String, Map<String, String>> bindingNames = new HashMap<String, Map<String, String>>();
    private Map<String, Map<String, String>> engineNames = new HashMap<String, Map<String, String>>();
    private Map<String, Map<String, String>> featureNames = new HashMap<String, Map<String, String>>();

    public JMXServerFeatureImpl() {
    }

    public void spi_setConfiguration(Element configuration) {
        jmxServer.setConfiguration(configuration);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws JMXServerFeatureException {

        log.debug("init begin");

        Server server = SystemContext.getContext().getSystem().getServer();
        
        // instantiate object which implements our management interface(s) and
        // manages configuration on our behalf

        ExtensionFactoryContext context = consumer.getExtensionFactory().getFactoryContext();
        jmxServer.configure(context, this);

        // if authentication is enabled, create an authenticator which will be
        // used by connector server to perform jaas authentication

        Map<String, JMXAuthenticator> env = null;
        JMXAuthenticator authenticator = null;
        if (jmxServer.isAuthenticationEnabled()) {
            if (server.getFeature(LoginContextFeature.class) == null) {
                throw new JMXServerFeatureException("Optional feature "
                        + LoginContextFeature.TYPE
                        + " must be explicitly enabled to perform authentication.");
            }
            env = new HashMap<String, JMXAuthenticator>();
            String jaasConfiguration = jmxServer.getJAASConfiguration();
            String user = jmxServer.getUser();
            authenticator = new JMXAuthenticatorImpl(jaasConfiguration, user);
            env.put(JMXConnectorServer.AUTHENTICATOR, authenticator);
        } else {
            log.warn("JMX Server authentication disabled");
        }

        // create mbean server instance with default domain
        String defaultDomain = SystemContext.getContext().getSystem().getRef();
        mbeanServer = MBeanServerFactory.createMBeanServer(defaultDomain);

        // retrieve port upon which rmi registry is configured to listen
        RMIRegistryFeature feature = server.getFeature(RMIRegistryFeature.class);
        int rmiPort = feature.getPort();

        // create a jmx connector which will accept rmi connections.
        // if authenticator exists, register it to accept connection
        // events so that it can manage login/logout pairs

        try {
            String urlStr = "service:jmx:rmi:///jndi/rmi://localhost:" + rmiPort + "/server";
            JMXServiceURL url = new JMXServiceURL(urlStr);
            connectorServer = JMXConnectorServerFactory.newJMXConnectorServer(url, env, mbeanServer);
            if (authenticator != null) {
                NotificationFilterSupport filter = new NotificationFilterSupport();
                filter.enableType(JMXConnectionNotification.OPENED);
                filter.enableType(JMXConnectionNotification.CLOSED);
                filter.enableType(JMXConnectionNotification.FAILED);
                NotificationListener listener = (NotificationListener)authenticator;
                connectorServer.addNotificationListener(listener, filter, null);
            }
            connectorServer.start();
        } catch (Exception ex) {
            throw new JMXServerFeatureException("Error creating rmi connector. " + ex.getMessage());
        }

        // register our managment interface with mbean server so that feature
        // can be managed

        spi_registerMBean(consumer, jmxServer, null, jmxServer.getClass().getSimpleName());

        // TODO: create a portlet and register with portlet container if the
        // optional feature is enabled

        log.debug("init end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        if (connectorServer != null) {
            try {
                connectorServer.stop();
            } catch (IOException ie) {
                log.error("Error stopping rmi connector. " + ie.getMessage());
            }
        }
        if (mbeanServer != null) {
            for (ObjectName objectName : registeredMBeans) {
                try {
                    mbeanServer.unregisterMBean(objectName);
                } catch (Exception ex) {
                    log.error("error unregistering mbean. " + ex);
                }
            }
        }
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (JMXServerFeature)consumer;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature$Provider#spi_getMBeanServer()
     */
    public MBeanServer spi_getMBeanServer() {
        return mbeanServer;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature$Provider#spi_registerMBean(org.bluestemsoftware.specification.eoa.ext.ManageableExtension,
     *      java.lang.Object, java.lang.String)
     */
    public ObjectName spi_registerMBean(ManageableExtension manageableExtension, Object mbean, String type, String name) throws JMXServerFeatureException {
        if (manageableExtension == null) {
            throw new IllegalArgumentException("manageableExtension null");
        }
        if (mbean == null) {
            throw new IllegalArgumentException("mbean null");
        }
        if (name == null) {
            throw new IllegalArgumentException("name null");
        }
        ObjectName objectName = null;
        try {
            objectName = new ObjectName(generateObjectName(manageableExtension, type, name));
            mbeanServer.registerMBean(mbean, objectName);
            registeredMBeans.add(objectName);
        } catch (Exception ex) {
            throw new JMXServerFeatureException("Error registering mbean", ex);
        }
        return objectName;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature$Provider#spi_registerSystemMBean(java.lang.Object,
     *      java.lang.String)
     */
    public ObjectName spi_registerSystemMBean(Object mbean, String type, String name) throws JMXServerFeatureException {
        if (mbean == null) {
            throw new IllegalArgumentException("mbean null");
        }
        if (name == null) {
            throw new IllegalArgumentException("name null");
        }
        ObjectName objectName = null;
        try {
            objectName = new ObjectName(generateObjectName(null, type, name));
            mbeanServer.registerMBean(mbean, objectName);
            registeredMBeans.add(objectName);
        } catch (Exception ex) {
            throw new JMXServerFeatureException("Error registering system mbean.", ex);
        }
        return objectName;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature$Provider#spi_unRegisterMBean(javax.management.ObjectName)
     */
    public void spi_unRegisterMBean(ObjectName objectName) throws JMXServerFeatureException {
        if(registeredMBeans.contains(objectName)) {
            try {
                mbeanServer.unregisterMBean(objectName);
            } catch (Exception ex) {
                throw new JMXServerFeatureException(ex);
            }
            registeredMBeans.remove(objectName);
        }
    }

    /*
     * (non-Javadoc) management operation
     */
    public void stopConnector() {
        if (connectorServer != null) {
            try {
                connectorServer.stop();
            } catch (IOException ie) {
                log.error("Error stopping jmx connector. " + ie);
            }
        }
    }

    private synchronized String generateObjectName(ManageableExtension manageableExtension, String type, String name) {

        // create a domain name that is unique within context of this server instance, i.e.
        // if more than one server instance is running, a separate registry will be
        // created by rmi-registry feature on a different port. when creating object name,
        // the goal is to organize mbeans as follows:

        // system
        // -- mbean(s)
        // -- Bindings
        // ----- binding.name
        // -------- mbean(s)
        // -- Container
        // ----- mbean(s)
        // -- Engines
        // ----- engine.name
        // -------- mbean(s)
        // -- Server
        // ----- mbean(s)
        // ----- Features
        // -------- feature.artifact.id
        // -------- mbean(s)

        if (manageableExtension == null) {

            StringBuilder sb = new StringBuilder();
            sb.append(mbeanServer.getDefaultDomain());
            sb.append(":");
            sb.append("name=");
            sb.append(name);
            return sb.toString();

        }

        ExtensionFactoryContext dc = manageableExtension.getExtensionFactory().getFactoryContext();
        String organizationID = ((ExtensionFactoryDeployment)dc.getDeployment()).getOrganizationID();
        String artifactID = ((ExtensionFactoryDeployment)dc.getDeployment()).getArtifactID();

        StringBuilder sb = new StringBuilder();
        sb.append(mbeanServer.getDefaultDomain());
        sb.append(":");

        // we must account for the fact that a binding, engine and/or feature
        // short name may not be unique, i.e. if more than one organization
        // uses same short name

        if (manageableExtension instanceof BindingRT) {
            sb.append("category=Bindings,");
            sb.append("binding=");
            BindingRT binding = (BindingRT)manageableExtension;
            String bindingName = binding.getName().getLocalPart();
            Map<String, String> organizations = bindingNames.get(bindingName);
            if (organizations != null) {
                bindingName = organizations.get(organizationID);
                if (bindingName != null) {
                    bindingName = bindingName + organizations.size();
                    organizations.put(organizationID, bindingName);
                }
            } else {
                organizations = new HashMap<String, String>();
                organizations.put(organizationID, bindingName);
                bindingNames.put(bindingName, organizations);
            }
            sb.append(bindingName);
            if (type != null) {
                sb.append(",stype=");
                sb.append(type);
            }
            sb.append(",name=");
            sb.append(name);
        } else if (manageableExtension instanceof Container) {
            sb.append("category=Container");
            if (type != null) {
                sb.append(",stype=");
                sb.append(type);
            }
            sb.append(",name=");
            sb.append(name);
        } else if (manageableExtension instanceof EngineRT) {
            sb.append("category=Engines,");
            sb.append("engine=");
            String engineName = ((EngineRT)manageableExtension).getName().getLocalPart();
            Map<String, String> organizations = engineNames.get(engineName);
            if (organizations != null) {
                engineName = organizations.get(organizationID);
                if (engineName != null) {
                    engineName = engineName + organizations.size();
                    organizations.put(organizationID, engineName);
                }
            } else {
                organizations = new HashMap<String, String>();
                organizations.put(organizationID, engineName);
                engineNames.put(engineName, organizations);
            }
            sb.append(engineName);
            if (type != null) {
                sb.append(",stype=");
                sb.append(type);
            }
            sb.append(",name=");
            sb.append(name);
        } else if (manageableExtension instanceof Server) {
            sb.append("category=Server");
            if (type != null) {
                sb.append(",stype=");
                sb.append(type);
            }
            sb.append(",name=");
            sb.append(name);
        } else if (manageableExtension instanceof Feature) {
            sb.append("category=Server,");
            sb.append("subcategory=Features,");
            sb.append("feature=");
            String featureName = artifactID;
            Map<String, String> organizations = featureNames.get(featureName);
            if (organizations != null) {
                featureName = organizations.get(organizationID);
                if (featureName != null) {
                    featureName = featureName + organizations.size();
                    organizations.put(organizationID, featureName);
                }
            } else {
                organizations = new HashMap<String, String>();
                organizations.put(organizationID, featureName);
                featureNames.put(featureName, organizations);
            }
            sb.append(featureName);
            if (type != null) {
                sb.append(",stype=");
                sb.append(type);
            }
            sb.append(",name=");
            sb.append(name);
        }

        return sb.toString();

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

}
