/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.mgmt;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.JMXServerFeatureImpl;
import org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.mgmt.jmx.JMXServerMXBean;
import org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.util.Constants;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeatureException;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 * Encapsulates feature management which includes configuration management and implementation
 * of administrative operations which MAY be invoked at runtime.
 * <p>
 * Note that xbean objects represent a read-only model of our configuration, i.e all changes
 * are manifested within configuration element which is persisted on {@link Server}. Changes
 * to configuration will not take effect until feature is re-loaded, at which point, xbean
 * object model will reflect the new configuration.
 */
public class JMXServer implements JMXServerMXBean {

    private ExtensionFactoryContext factoryContext;
    private JMXServerFeatureImpl jmxServerFeature;
    private Element configuration;
    private boolean isAuthenticationEnabled;
    private String user;
    private String jaasConfiguration;

    public JMXServer() {
    }

    public String getJAASConfiguration() {
        return jaasConfiguration;
    }

    public String getUser() {
        return user;
    }

    public boolean isAuthenticationEnabled() {
        return isAuthenticationEnabled;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.feature.jmx.server.dfault.mgmt.JMXServerMXBean#stopConnector()
     */
    public void stopConnector() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factoryContext.getClassLoader());
            jmxServerFeature.stopConnector();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    public void setConfiguration(Element configuration) {
        this.configuration = configuration;
    }

    // TODO: expose configuration setters via a JMXServer portlet. when a managed
    // attribute is changed, we will update the configuration element and issue
    // an update on server

    public void setAuthenticationEnabled(boolean isAuthenticationEnabled) {
        throw new UnsupportedOperationException("Configuration is read only");
    }

    public void setJaasConfiguration(String jaasConfiguration) {
        throw new UnsupportedOperationException("Configuration is read only");
    }

    public void setUser(String user) {
        throw new UnsupportedOperationException("Configuration is read only");
    }

    public void configure(ExtensionFactoryContext factoryContext, JMXServerFeatureImpl jmxServerFeature) throws JMXServerFeatureException {

        this.factoryContext = factoryContext;
        this.jmxServerFeature = jmxServerFeature;

        if (configuration == null) {
            try {
                Resource resource = factoryContext.getResource(Constants.DEFAULT_CONFIGURATION_LOC);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(Constants.DEFAULT_CONFIGURATION_LOC);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(factoryContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
            } catch (Exception ex) {
                throw new JMXServerFeatureException("Error retrieving default configuration. " + ex);
            }
        }

        validateConfiguration(new DOMSource(configuration));

        QName childName = new QName(Constants.FEATURE_SCHEMA_NS, "authentication");
        Element authElement = DOMUtils.getChildElement(configuration, childName);

        // TODO: provide ability, via configuration, to configure an ssl
        // connector, i.e. with keystore config. i think jconsole sends
        // credentials as clear text by default

        if (authElement != null) {
            isAuthenticationEnabled = true;
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "user");
            Element userElement = DOMUtils.getChildElement(authElement, childName);
            user = DOMUtils.getText(userElement);
            childName = new QName(Constants.FEATURE_SCHEMA_NS, "jaasConfiguration");
            Element jcElement = DOMUtils.getChildElement(authElement, childName);
            if (jcElement != null) {
                jaasConfiguration = DOMUtils.getText(jcElement);
            } else {
                jaasConfiguration = "other";
            }
        }

    }

    private void validateConfiguration(DOMSource domSource) throws JMXServerFeatureException {

        Resource resource = null;
        try {
            resource = factoryContext.getResource(Constants.FEATURE_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new JMXServerFeatureException("Error validating configuration. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new JMXServerFeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new JMXServerFeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new JMXServerFeatureException("Configuration invalid. " + validationError);
        }

    }

}
