/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.management.jmx.server.dfault;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.remote.JMXAuthenticator;
import javax.management.remote.JMXConnectionNotification;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.bluestemsoftware.open.eoa.ext.management.jmx.server.dfault.util.JMXAuthenticationRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginCallbackHandler;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginCredentials;

public class JMXAuthenticatorImpl implements JMXAuthenticator, NotificationListener {

    private String jaasConfiguration;
    private String user;
    private ThreadLocal<LoginContext> threadLocal;
    private Map<String, LoginContext> contextMap;

    public JMXAuthenticatorImpl(String jaasConfiguration, String user) {
        this.jaasConfiguration = jaasConfiguration;
        this.user = user;
        threadLocal = new ThreadLocal<LoginContext>();
        contextMap = new HashMap<String, LoginContext>();
        Collections.synchronizedMap(contextMap);
    }

    /*
     * (non-Javadoc)
     * @see javax.management.remote.JMXAuthenticator#authenticate(java.lang.Object)
     */
    public Subject authenticate(Object credentials) {

        String[] parameters = null;
        if (!(credentials instanceof String[])) {
            throw new SecurityException("Invalid credentials. Expected String[2]");
        } else {
            parameters = (String[])credentials;
        }

        String user = null;
        String pass = null;
        if (parameters.length != 2) {
            throw new SecurityException("Invalid credentials. Expected String[2]");
        } else {
            user = parameters[0];
            pass = parameters[1];
        }
        
        // note that jmx authentication does not provide
        // a realm, consequently we define a single user
        // authorized to connect via jmx
        
        if (user != null && !user.equals(this.user)) {
            throw new SecurityException("Login Failed");
        }
        
        // user matches. now we use jaas to retrieve and
        // check password retrieved from credential store

        LoginContext loginContext = null;
        try {
            LoginCredentials creds = new JMXAuthenticationRequest(user, pass);
            CallbackHandler cbh = new LoginCallbackHandler(creds);
            loginContext = new LoginContext(jaasConfiguration, cbh);
            loginContext.login();
        } catch (LoginException ex) {
            throw new SecurityException("Login Failed");
        }

        // set a reference to login context on thread so
        // that we can associate it with a connection id
        // when notified of connection open event below
        threadLocal.set(loginContext);

        return loginContext.getSubject();
    }

    public void handleNotification(Notification notification, Object handback) {
        if (notification instanceof JMXConnectionNotification) {
            
            JMXConnectionNotification jmn = (JMXConnectionNotification)notification;
            if (JMXConnectionNotification.OPENED.equals(jmn.getType())) {
                
                // now that we know the connectionid, retrieve
                // thread local variable set in authenticate
                // method and associate it with connection id
                
                LoginContext context = threadLocal.get();
                threadLocal.set(null);
                contextMap.put(jmn.getConnectionId(), context);
                
            } else {
                
                // connection is either closed or failed. in
                // either case logout and remove from map
                
                LoginContext context = contextMap.remove(jmn.getConnectionId());
                if (context != null) {
                    try {
                        context.logout();
                    } catch (LoginException ignore) {
                    }
                }
                
            }
        }
    }

}
