/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
 
package org.bluestemsoftware.open.eoa.aspect.axiom;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.ByteArrayInputStream;
import java.util.Iterator;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMOutputFormat;
import org.apache.axiom.om.impl.MTOMXMLStreamWriter;
import org.apache.axiom.om.impl.OMNodeEx;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.impl.dom.DocumentImpl;
import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.appdata.ApplicationData;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public aspect ContentFactoryImpl {
    
    declare parents : DocumentImpl implements ContentFactory;

    public Content DocumentImpl.importContent(Content source) {
        
        // a one-to-one relationship exists between factory and a document. we cannot pass
        // our factory to builder, i.e because getDocumentElement would return the message
        // element - not the imported content. so ... we create a new factory and detach the
        // content from document created by builder and then adopt the node

        StAXOMBuilder builder = new StAXOMBuilder(new OMDOMFactory(), source.getReader());
        DocumentElement documentElement = (DocumentElement)builder.getDocumentElement();
        documentElement.detachFromDocument();
        
        // TODO: currently as of version 1.2.6, adoptNode is not implemented, i.e. it
        // just calls importNode. we need to verify that once implemented, it actually
        // leverages the reader, i.e. that it doesn't build the tree to memory (which
        // is what currently happens)
        
        return (Content)this.adoptNode((Node)documentElement);
        
    }

    public Content DocumentImpl.createContent(String namespaceURI, String qualifiedName) {
        return (Content)this.createElementNS(namespaceURI, qualifiedName);
    }
    
    public Content DocumentImpl.parseContent(InputStream inputStream) throws SAXException, IOException {
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), inputStream, (ContentSerialization)ContentSerializationXML.getInstance(), "UTF-8");
    }
    
    public Content DocumentImpl.parseContent(String content) throws SAXException, IOException {
        if (content == null) {
            throw new IllegalArgumentException("content null");
        }
        ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes("UTF-8"));
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), in, (ContentSerialization)ContentSerializationXML.getInstance(), "UTF-8");
    }
    
    public Content DocumentImpl.parseContent(InputStream inputStream, ContentSerialization serialization) throws SAXException, IOException {
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), inputStream, serialization, "UTF-8");
    }
    
    public Content DocumentImpl.parseContent(String content, ContentSerialization serialization) throws SAXException, IOException {
        if (content == null) {
            throw new IllegalArgumentException("content null");
        }
        ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes("UTF-8"));
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), in, serialization, "UTF-8");
    }

    public Content DocumentImpl.parseContent(InputStream inputStream, ContentSerialization serialization, String encoding) throws SAXException, IOException {
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), inputStream, serialization, encoding);
    }
    
    public Content DocumentImpl.parseContent(String content, ContentSerialization serialization, String encoding) throws SAXException, IOException {
        if (content == null) {
            throw new IllegalArgumentException("content null");
        }
        ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes(encoding));
        return STAXUtils.parseContent((OMDOMFactory)this.getOMFactory(), in, serialization, encoding);
    }
    
    public ApplicationData DocumentImpl.createApplicationData(String namespaceURI, String qualifiedName) {
        return (ApplicationData)this.createElementNS(namespaceURI, qualifiedName);
    }
    
    /*
     * Overrides method on DocumentImpl which throws unsupported operation exception. note that
     * i couldn't get 'call' join point to work with aspectjrt as of version 1.5.3
     */
    DocumentType around(): execution(DocumentType org.apache.axiom.om.impl.dom.DocumentImpl.getDoctype()) {
        return null;
    }
    
    
    /*
     * TODO: remove (AS OF VERSION 1.2.5 SUPER THROWS UNSUPPORTED OPERATION EXCEPTION)
     */
    void around(DocumentImpl d, OutputStream out, OMOutputFormat format) throws XMLStreamException:
        execution(void DocumentImpl.serialize(OutputStream, OMOutputFormat)) && args(out, format) && this(d) {
            STAXUtils.serializeDocument(d, new MTOMXMLStreamWriter(out, format), true);
    }
    
    /*
     * TODO: remove (AS OF VERSION 1.2.5 SUPER THROWS UNSUPPORTED OPERATION EXCEPTION)
     */
    void around(DocumentImpl d, XMLStreamWriter writer) throws XMLStreamException:
        execution(void DocumentImpl.serialize(XMLStreamWriter)) && args(writer) && this(d) {
            STAXUtils.serializeDocument(d, new MTOMXMLStreamWriter(writer), true);
    }
    
    /*
     * TODO: remove (AS OF VERSION 1.2.5 SUPER THROWS UNSUPPORTED OPERATION EXCEPTION)
     */
    void around(DocumentImpl d, OutputStream out, OMOutputFormat format) throws XMLStreamException:
        execution(void DocumentImpl.serializeAndConsume(OutputStream, OMOutputFormat)) && args(out, format) && this(d) {
            STAXUtils.serializeDocument(d, new MTOMXMLStreamWriter(out, format), false);
    }
    
    /*
     * TODO: remove (AS OF VERSION 1.2.5 SUPER THROWS UNSUPPORTED OPERATION EXCEPTION)
     */
    void around(DocumentImpl d, XMLStreamWriter writer) throws XMLStreamException:
        execution(void DocumentImpl.serializeAndConsume(XMLStreamWriter)) && args(writer) && this(d) {
            STAXUtils.serializeDocument(d, new MTOMXMLStreamWriter(writer), false);
    }
    
    /*
     * TODO: remove (AS OF VERSION 1.2.6 method is not implemented within axiom)
     */
    void around(DocumentImpl d, XMLStreamWriter writer) throws XMLStreamException:
        execution(void DocumentImpl.internalSerialize(XMLStreamWriter)) && args(writer) && this(d) {
            STAXUtils.serializeDocument(d, writer, true);
    }
    
    
    /*
     * TODO: remove (AS OF VERSION 1.2.6 method is not implemented within axiom)
     */
    void around(DocumentImpl d, XMLStreamWriter writer) throws XMLStreamException:
        execution(void DocumentImpl.internalSerializeAndConsume(XMLStreamWriter)) && args(writer) && this(d) {
            STAXUtils.serializeDocument(d, writer, false);
    }

}
