/**
 * Copyright  2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.aspect.axiom.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.sax.SAXSource;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMNode;
import org.apache.axiom.om.impl.MTOMXMLStreamWriter;
import org.apache.axiom.om.impl.OMNodeEx;
import org.apache.axiom.om.impl.builder.StAXBuilder;
import org.apache.axiom.om.impl.dom.DocumentImpl;
import org.apache.axiom.om.impl.dom.factory.OMDOMFactory;
import org.bluestemsoftware.open.eoa.aspect.axiom.DocumentElement;
import org.bluestemsoftware.open.eoa.aspect.axiom.ParsedContent;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerialization;
import org.bluestemsoftware.specification.eoa.component.message.rt.ContentSerializationXML;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationBadgerJSON;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationJSON;
import org.bluestemsoftware.specification.eoa.ext.message.ContentSerializationMappedJSON;
import org.codehaus.jettison.badgerfish.BadgerFishXMLInputFactory;
import org.codehaus.jettison.badgerfish.BadgerFishXMLOutputFactory;
import org.codehaus.jettison.mapped.Configuration;
import org.codehaus.jettison.mapped.MappedXMLInputFactory;
import org.codehaus.jettison.mapped.MappedXMLOutputFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Overcomes issue with axiom StAXUtils class which sets context classloader to its defining
 * classloader prior to invoking XMLInputFactory.newInstance() and/or
 * XMLOutputFactory.newInstance(), the java service provider discovery mechanism breaks.
 * Because an instance of this class is on SHARED classpath, as it should be (its defined
 * within axiom-api artifact), we would not be able to specify woodstox as our stax factory
 * impl, i.e. service provider file is on DEPLOYMENT classpath. Also axiom StAXUtils class
 * pools factories, but woodstox factories are threadsafe, i.e. no pooling required.
 */
public class STAXUtils {

    private static ContentSerialization XML = ContentSerializationXML.getInstance();
    private static ContentSerialization JSON = ContentSerializationMappedJSON.getInstance();
    private static ContentSerialization FISH = ContentSerializationBadgerJSON.getInstance();

    private static final String XML_INPUT_CLASS = "com.ctc.wstx.stax.WstxInputFactory";
    private static final String XML_OUTPUT_CLASS = "com.ctc.wstx.stax.WstxOutputFactory";
    private static final String EOF_EXCEPTION = "com.ctc.wstx.exc.WstxEOFException";
    private static Map<ContentSerialization, XMLInputFactory> inputFactories = new HashMap<ContentSerialization, XMLInputFactory>();
    private static Map<ContentSerialization, XMLOutputFactory> outputFactories = new HashMap<ContentSerialization, XMLOutputFactory>();

    public static XMLStreamReader createXMLStreamReader(InputStream in, String encoding) throws XMLStreamException {
        return createXMLStreamReader(XML, in, encoding);
    }

    public static XMLStreamReader createXMLStreamReader(ContentSerialization ser, InputStream in, String encoding) throws XMLStreamException {
        return getXMLInputFactory(ser).createXMLStreamReader(in, encoding);
    }

    public static XMLStreamWriter createXMLStreamWriter(OutputStream out, String encoding) throws XMLStreamException {
        return createXMLStreamWriter(XML, out, encoding);
    }
    
    public static XMLStreamWriter createXMLStreamWriter(ContentSerialization ser, OutputStream out, String encoding) throws XMLStreamException {
        return getXMLOutputFactory(ser).createXMLStreamWriter(out, encoding);
    }

    /**
     * Detaches document element from document and sets content as root element. Useful in
     * situations, e.g. xpath:
     * </p>
     * The context Node from which fragment expression is evaluated is a synthetic document
     * information item which contains a single element information item representing root
     * element of message part ...
     * 
     * @param documentElement
     * @param content
     * @return ContextInfo which is required to restore context
     */
    public static ContextInfo adjustContext(Content content) {

        // retrieve part accessor and force a build. note this
        // MUST be done to keep the builder in synch with our
        // content model. note that content may not have a
        // parent node if user is serializing random content
        Node pa = content.getParentNode();
        if (pa != null) {
            ((OMNode)pa).build();
        }

        // retrieve a reference to next sibling, which could
        // be null, so that we can re-insert content in the
        // appropriate position
        Node ns = content.getNextSibling();

        // create object which will be used to restore context
        Document ownerDocument = content.getOwnerDocument();
        Element documentElement = ownerDocument.getDocumentElement();
        ContextInfo contextInfo = new ContextInfo(documentElement, pa, ns);

        // detach document element. note that we do not use dom
        // detach method, which would build the entire message
        if (documentElement != null) {
            ((DocumentElement)documentElement).detachFromDocument();
        }

        // now detach the content from part accessor and
        // set it as the document node
        if (content.getParentNode() != null) {
            ((OMElement)content).detach();
        }
        ((DocumentElement)content).attachToDocument();

        return contextInfo;

    }

    /**
     * Restores
     * @param contextInfo
     * @param content
     */
    public static void restoreContext(ContextInfo contextInfo, Content content) {
        ((DocumentElement)content).detachFromDocument();
        Element documentElement = contextInfo.getDocumentElement();
        if (documentElement != null) {
            ((DocumentElement)documentElement).attachToDocument();
        }
        Node partAccessor = contextInfo.getPartAccessor();
        if (partAccessor != null) {
            partAccessor.insertBefore(content, contextInfo.getNextSibling());
        }
    }

    public static Content parseContent(OMDOMFactory factory, InputStream inputStream, ContentSerialization serialization, String encoding) throws SAXException, IOException {

        // a one-to-one relationship exists between factory and a document. we cannot pass
        // factory to builder, i.e because getDocumentElement would return the message
        // element - not the parsed content. so ... we create a new factory and detach the
        // content from document created by builder and set owner document to document
        // retrieved from factory, i.e. the document that's parsing the content

        if (serialization instanceof ContentSerializationXML) {

            try {
                InputSource is = new InputSource(inputStream);
                is.setEncoding(encoding);
                XMLStreamReader parser = getXMLInputFactory(serialization).createXMLStreamReader(new SAXSource(is));
                OMDOMFactory newFactory = factory.getClass().newInstance();
                StAXBuilder builder = new ContentBuilder(newFactory, parser);
                DocumentElement documentElement = (DocumentElement)builder.getDocumentElement();
                documentElement.detachFromDocument();
                return ((ParsedContent)documentElement).setOwnerDocument(factory.getDocument());
            } catch (Throwable th) {
                Throwable cause = th.getCause();
                if (cause != null && cause.getClass().getName().equals(EOF_EXCEPTION)) {
                    // file is empty, i.e. empty payload. return null
                    return null;
                }
                throw new IOException("Error reading message. " + th.getMessage());
            }

        } else if (serialization instanceof ContentSerializationJSON) {

            try {
                XMLStreamReader parser = getXMLInputFactory(serialization).createXMLStreamReader(inputStream,
                        encoding);
                OMDOMFactory newFactory = factory.getClass().newInstance();
                StAXBuilder builder = new ContentBuilder(newFactory, parser);
                DocumentElement documentElement = (DocumentElement)builder.getDocumentElement();
                documentElement.detachFromDocument();
                return ((ParsedContent)documentElement).setOwnerDocument(factory.getDocument());
            } catch (Throwable th) {
                throw new IOException("Error reading message. " + th.getMessage());
            }

        } else {

            throw new IllegalArgumentException("Error serializing content. Unsupported serialization '"
                    + serialization
                    + "'.");

        }

    }

    public static String serializeContent(OMElement omElement, ContentSerialization serialization, String encoding) {

        if (serialization instanceof ContentSerializationXML) {

            // do not consume stream if it exists. also
            // default charset is utf-8 and optimize is
            // false. as of 1.2.5 snapshot, i couldn't
            // find a way to turn indenting off. so ...
            // just invoke toString

            return omElement.toString();

        } else if (serialization instanceof ContentSerializationJSON) {

            // we need to adjust context, i.e. set content as root element of
            // document which as of version 1.0.1 is required by jettison (it
            // writes cached content within the end document event)

            ContextInfo contextInfo = null;
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                XMLStreamWriter writer = getXMLOutputFactory(serialization).createXMLStreamWriter(baos, encoding);
                contextInfo = adjustContext((Content)omElement);
                writer = new MTOMXMLStreamWriter(writer);
                serializeDocument(((DocumentImpl)omElement.getParent()), writer, true);
                return baos.toString(encoding);
            } catch (Throwable th) {
                throw new IllegalStateException("Error writing message. " + th.getMessage());
            } finally {
                if (contextInfo != null) {
                    restoreContext(contextInfo, (Content)omElement);
                }
            }

        } else {

            throw new IllegalArgumentException("Error serializing content. Unsupported serialization '"
                    + serialization
                    + "'.");

        }

    }

    private static synchronized XMLInputFactory getXMLInputFactory(ContentSerialization ser) throws XMLStreamException {

        XMLInputFactory xmlInputFactory = inputFactories.get(ser);
        
        if (xmlInputFactory == null) {
   
            if (ser.equals(XML)) {

                // note that chosen parser implementation is woodstox. wstx parser factory is thread
                // safe and can be used concurrently, the bundled jdk streaming factory (as of 1.6.0),
                // is NOT threadsafe

                Class<?> temp;
                try {
                    temp = Thread.currentThread().getContextClassLoader().loadClass(XML_INPUT_CLASS);
                } catch (ClassNotFoundException cfe) {
                    throw new IllegalStateException("Deployment classloader must scope factory class "
                            + XML_INPUT_CLASS
                            + " to deployment classpath");
                }
                try {
                    xmlInputFactory = (XMLInputFactory)temp.newInstance();
                } catch (Exception ex) {
                    throw new XMLStreamException("Error instantiating parser factory. " + ex);
                }

                xmlInputFactory.setProperty(XMLInputFactory.IS_NAMESPACE_AWARE, Boolean.TRUE);
                xmlInputFactory.setProperty(XMLInputFactory.IS_VALIDATING, Boolean.FALSE);
                xmlInputFactory.setProperty(XMLInputFactory.SUPPORT_DTD, Boolean.FALSE);

                // adds 10% overhead
                // xmlInputFactory.setProperty(XMLInputFactory.IS_COALESCING, Boolean.TRUE);
                
            } else if (ser.equals(JSON)) {
                
                // the namespace to prefix mappings cannot be known when parsing
                // a json string sent over the wire, so we set ignore namespaces
                // to true which will strip namespace prefixes from output rather
                // than throwing an exception
                
                Configuration configuration = new Configuration();
                configuration.setIgnoreNamespaces(true);
                xmlInputFactory = new MappedXMLInputFactory(configuration);
                
            } else if (ser.equals(FISH)) {
                
                xmlInputFactory = new BadgerFishXMLInputFactory();
                
            } else {
                
                throw new IllegalArgumentException("Error parsing content. Unsupported serialization '"
                        + ser
                        + "'.");
                
            }

            inputFactories.put(ser, xmlInputFactory);
        
        }
        
        return xmlInputFactory;

    }

    private static synchronized XMLOutputFactory getXMLOutputFactory(ContentSerialization ser) throws XMLStreamException {

        XMLOutputFactory xmlOutputFactory = outputFactories.get(ser);
        
        if (xmlOutputFactory == null) {

            if (ser.equals(XML)) {

                // note that chosen parser implementation is woodstox. wstx parser factory is thread
                // safe and can be used concurrently, the bundled jdk streaming factory (as of 1.6.0),
                // is NOT threadsafe
               
                Class<?> temp;
                try {
                    temp = Thread.currentThread().getContextClassLoader().loadClass(XML_OUTPUT_CLASS);
                } catch (ClassNotFoundException cfe) {
                    throw new IllegalStateException("Deployment classloader must scope factory class "
                            + XML_OUTPUT_CLASS
                            + " to deployment classpath");
                }
                try {
                    xmlOutputFactory = (XMLOutputFactory)temp.newInstance();
                } catch (Exception ex) {
                    throw new XMLStreamException("Error instantiating parser factory. " + ex);
                }
                // xmlOutputFactory.setProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES,
                // Boolean.TRUE);
                
            } else if (ser.equals(JSON)) {
                
                // the namespace to prefix mappings cannot be known when parsing
                // a json string sent over the wire, so we set ignore namespaces
                // to true which will strip namespace prefixes from output rather
                // than throwing an exception
                
                Configuration configuration = new Configuration();
                configuration.setIgnoreNamespaces(true);
                xmlOutputFactory = new MappedXMLOutputFactory(configuration);

            } else if (ser.equals(FISH)) {
                
                xmlOutputFactory = new BadgerFishXMLOutputFactory();
                
            } else {
                
                throw new IllegalArgumentException("Error serializing content. Unsupported serialization '"
                        + ser
                        + "'.");
                
            }

            outputFactories.put(ser, xmlOutputFactory);

        }
        
        return xmlOutputFactory;

    }
    
    public static void serializeDocument(DocumentImpl document, XMLStreamWriter writer, boolean cache) throws XMLStreamException {
        MTOMXMLStreamWriter temp = (MTOMXMLStreamWriter)writer;
        if (!temp.isIgnoreXMLDeclaration()) {
            String charSetEncoding = document.getCharsetEncoding() == null ? "UTF-8" : document.getCharsetEncoding();
            String xmlVersion = document.getXMLVersion() == null ? "1.0" : document.getXMLVersion();
            String outputCharEncoding = temp.getCharSetEncoding();
            if (outputCharEncoding == null || outputCharEncoding.equals("")) {
                temp.getXmlStreamWriter().writeStartDocument(charSetEncoding, xmlVersion);
            } else {
                temp.getXmlStreamWriter().writeStartDocument(outputCharEncoding, xmlVersion);
            }
        }
        Iterator<?> children = document.getChildren();
        while (children.hasNext()) {
            OMNodeEx omNode = (OMNodeEx)children.next();
            if (cache) {
                omNode.internalSerialize(temp);
            } else {
                omNode.internalSerializeAndConsume(temp);
            }
        }
        temp.getXmlStreamWriter().writeEndDocument();
        temp.flush();
    }

}
