/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.rmi.dfault;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.ext.feature.rmi.dfault.util.DOMSerializer;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.jaas.LoginContextFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.rmi.RMIRegistryFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.rmi.RMIRegistryFeatureException;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class RMIRegistryFeatureImpl implements RMIRegistryFeature.Provider {

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(RMIRegistryFeature.class);

    public static final String IMPL = RMIRegistryFeatureImpl.class.getName();

    private int port = Registry.REGISTRY_PORT;
    private Extension consumer;
    private DeploymentContext deploymentContext;
    private Registry registry;

    public RMIRegistryFeatureImpl() {
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c
     * .dom.Element)
     */
    public void spi_init(Set<ManagementContext> managementContexts) throws RMIRegistryFeatureException {
        log.debug("init begin");

        Element configuration = null;
        File extensionEtcDir = ((Feature)consumer).getExtensionEtcDir();
        File configurationFile = new File(extensionEtcDir, "config.xml");
        if (configurationFile.exists()) {
            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                Document configurationDoc = documentBuilder.parse(configurationFile);
                configuration = configurationDoc.getDocumentElement();
            } catch (Exception ex) {
                throw new RMIRegistryFeatureException("Error loading configuration. " + ex);
            }
        }

        // if configuration is null, retrieve default configuration contained within
        // deployment and persist it

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.rmi.registry.config.1.0.xml";
            Writer writer = null;
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
                writer = new FileWriter(configurationFile);
                DOMSerializer.serializeNode(configuration, writer, "UTF-8", true);
            } catch (Exception ex) {
                throw new RMIRegistryFeatureException("Error retrieving default configuration. " + ex);
            } finally {
                if (writer != null) {
                    try {
                        writer.close();
                    } catch (IOException ignore) {
                    }
                }
            }
        }
        
        validateConfiguration(new DOMSource(configuration));
        
        try {
            QName childName = new QName("http://bluestemsoftware.org/open/eoa/ext/feature/rmi/registry/config/1.0", "port");
            Element portElement = DOMUtils.getChildElement(configuration, childName);
            port = Integer.parseInt(DOMUtils.getText(portElement));
            registry = LocateRegistry.createRegistry(port);
        } catch (RemoteException re) {
            throw new RMIRegistryFeatureException(re);
        }
        
        log.debug("init end");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        if (registry != null) {
            try {
                UnicastRemoteObject.unexportObject(registry, true);
            } catch (NoSuchObjectException ignore) {
            }
        }
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.
     * bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = consumer;
        this.deploymentContext = consumer.getExtensionFactory().getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.bluestemsoftware.specification.eoa.ext.feature.rmi.RMIRegistryFeature$Provider#
     * spi_getPort()
     */
    public int spi_getPort() {
        return port;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl
     * ()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }
    
    private void validateConfiguration(DOMSource domSource) throws RMIRegistryFeatureException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource("classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.rmi.registry.config.1.0.xsd");
        } catch (DeploymentException de) {
            throw new RMIRegistryFeatureException("Error validating configuration. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new RMIRegistryFeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new RMIRegistryFeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new RMIRegistryFeatureException("Configuration invalid. " + validationError);
        }

    }

}
