/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.application.spring10;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationReference;
import org.bluestemsoftware.specification.eoa.component.application.rt.RoleReference;
import org.bluestemsoftware.specification.eoa.component.policy.rt.ElementPolicy;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.application.ApplicationFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * A type specific <code>Application</code> factory that creates instances of
 * <code>SpringApplication</code>.
 */
public class SpringApplicationFactory extends ApplicationFactory {

    public interface Provider extends ApplicationFactory.Provider {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ExtensionFactory#getExtensionType()
     */
    public String getExtensionType() {
        return SpringApplication.TYPE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.ProviderReader#readProviders(org.bluestemsoftware.specification.eoa.component.ComponentContext,
     *      javax.xml.transform.dom.DOMSource)
     */
    public Set<ApplicationRT> readProviders(ComponentContext componentContext, DOMSource source) throws DeploymentException {

        if (componentContext == null) {
            throw new IllegalArgumentException("componentContext null");
        }
        if (source == null) {
            throw new IllegalArgumentException("source null");
        }

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(factoryContext.getClassLoader());

            Set<ApplicationRT> providers = new HashSet<ApplicationRT>();

            // first validate the definition against schema
            validateDefinition(source);

            // now parse definition into an application component
            Element definition = (Element)source.getNode();
            String tns = definition.getNamespaceURI();

            // parse name of application according to component name contract
            ComponentName cname = ComponentName.valueOf(definition, definition.getAttribute("name"));

            // retrieve optional configuration
            QName childName = new QName(tns, "configuration");

            // parse application references
            Map<QName, ApplicationReference> arefs = new HashMap<QName, ApplicationReference>();
            childName = new QName(tns, "partners");
            Element partnersElement = getChildElement(definition, childName);
            if (partnersElement != null) {
                childName = new QName(tns, "applicationReference");
                List<Element> appRefElements = getChildElements(partnersElement, childName);
                for (Element appRefElement : appRefElements) {
                    String prefixedName = appRefElement.getAttribute("applicationName");
                    ComponentName applicationName = ComponentName.valueOf(appRefElement, prefixedName);
                    boolean isMyApplication = applicationName.equals(cname);
                    childName = new QName(tns, "roleReference");
                    Map<String, RoleReference> rr = new HashMap<String, RoleReference>();
                    List<Element> roleRefElements = getChildElements(appRefElement, childName);
                    for (Element roleRefElement : roleRefElements) {
                        String roleName = roleRefElement.getAttribute("roleName");
                        String beanType = roleRefElement.getAttribute("beanType");
                        ElementPolicy ep = getElementPolicy(componentContext, roleRefElement);
                        rr.put(roleName, new SpringRoleReference(roleName, ep, beanType));
                    }
                    ApplicationReference aref = new SpringApplicationReference(applicationName, rr, isMyApplication);
                    arefs.put(applicationName, aref);
                }
            }

            // note that there is no 'provided' behavior for spring
            // app
            
            ApplicationRT.Provider p = new ApplicationRT.Provider() {
                public void spi_setConsumer(Extension consumer) {
                }
            };

            providers.add(new SpringApplication(this, p, cname, arefs));
            return providers;

        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    private List<Element> getChildElements(Element parent, QName childName) {
        List<Element> answer = new ArrayList<Element>();
        NodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            if (childName.getNamespaceURI().equals(XMLConstants.NULL_NS_URI)) {
                if (element.getNamespaceURI() != null) {
                    continue;
                }
            } else {
                if (element.getNamespaceURI() == null) {
                    continue;
                }
                if (!childName.getNamespaceURI().equals("*")
                        && !element.getNamespaceURI().equals(childName.getNamespaceURI())) {
                    continue;
                }
            }
            if (!childName.getLocalPart().equals("*") && !element.getLocalName().equals(childName.getLocalPart())) {
                continue;
            }
            answer.add(element);
        }
        return answer;
    }

    private Element getChildElement(Element parent, QName childName) {
        NodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            if (childName.getNamespaceURI().equals(XMLConstants.NULL_NS_URI)) {
                if (element.getNamespaceURI() != null) {
                    continue;
                }
            } else {
                if (element.getNamespaceURI() == null) {
                    continue;
                }
                if (!childName.getNamespaceURI().equals("*")
                        && !element.getNamespaceURI().equals(childName.getNamespaceURI())) {
                    continue;
                }
            }
            if (!childName.getLocalPart().equals("*") && !element.getLocalName().equals(childName.getLocalPart())) {
                continue;
            }
            return element;
        }
        return null;
    }

    private void validateDefinition(DOMSource source) throws DeploymentException {

        Resource resource = null;
        try {
            resource = factoryContext.getResource(SpringApplication.PROVIDER_SCHEMA_LOCATION);
        } catch (DeploymentException de) {
            throw new DeploymentException("Error validating definition. " + de);
        }

        SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new DeploymentException("Error parsing schema. " + ex);
        }
        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(source);
        } catch (Exception ex) {
            throw new DeploymentException("Error validating definition. " + ex);
        }
        if (validationError != null) {
            throw new DeploymentException("Engine definition invalid. " + validationError);
        }

    }

    private ElementPolicy getElementPolicy(ComponentContext componentContext, Element parent) throws DeploymentException {

        Set<PolicyFactory> factories = null;
        factories = SystemContext.getContext().getSystem().getPolicyFactories();
        ElementPolicy result = null;
        for (PolicyFactory factory : factories) {
            ElementPolicy temp = null;
            try {
                DOMSource source = new DOMSource(parent, parent.getOwnerDocument().getDocumentURI());
                temp = factory.readElementPolicy(componentContext, source);
            } catch (DeploymentException de) {
                throw new DeploymentException("Error reading element policy. " + de.getMessage());
            }
            if (result == null) {
                result = temp;
            } else {
                throw new DeploymentException("Attached policy expressions MUST be identically typed.");
            }
        }

        return result;

    }

    static final class DOMValidator {

        private Schema schema;

        public DOMValidator(Schema schema) {
            this.schema = schema;
        }

        public String validate(DOMSource source) throws SAXException, IOException {
            Validator validator = schema.newValidator();
            validator.setErrorHandler(new ErrorHandlerImpl());
            validator.validate(source);
            return ((ErrorHandlerImpl)validator.getErrorHandler()).getErrors();
        }

        static class ErrorHandlerImpl implements ErrorHandler {

            private ArrayList<String> errorMessages = new ArrayList<String>();

            public void error(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void warning(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void fatalError(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public String getErrors() {
                if (errorMessages.size() > 0) {
                    Iterator<String> messageIterator = errorMessages.iterator();
                    StringBuilder errorMessage = new StringBuilder();
                    while (messageIterator.hasNext()) {
                        errorMessage.append(messageIterator.next() + "  ");
                    }
                    return errorMessage.toString();
                }
                return null;
            }

        }

    }

}
