/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.engine.spring10.impl;

import java.util.Iterator;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.ext.engine.spring10.util.PartnerCallback;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerFault;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.RequestMessageReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.MEP;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ApplicationFault;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.FaultContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.rt.TransportFault;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;

public class PartnerOperationImpl implements PartnerOperation {

    private Log log = SystemContext.getContext().getSystem().getLog(PartnerOperation.class);

    private String id;
    private SpringEngineImpl engine;
    private ServiceReference sr;
    private QName operationName;
    private String endpointName;
    private InterfaceOperation interfaceOperation;
    private ApplicationRT application;

    public PartnerOperationImpl(SpringEngineImpl engine, ServiceReference sr, String endpointName,
            QName operationName) {
        this.engine = engine;
        this.sr = sr;
        this.endpointName = endpointName;
        this.operationName = operationName;
        Interface i = sr.getReferencedComponent().getInterfaceReference().getReferencedComponent();
        if (i.getOperation(operationName) == null) {
            throw new IllegalArgumentException("Operation "
                    + operationName
                    + " is undefined on interface "
                    + i.getName());
        } else {
            interfaceOperation = i.getOperation(operationName);

        }
        application = engine.getConsumer().getApplication();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MyOperation#getName()
     */
    public QName getName() {
        return operationName;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MyOperation#getMetaData()
     */
    public InterfaceOperation getMetaData() {
        return interfaceOperation;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation#createRequestAction()
     */
    public ActionContext createRequestAction() {
        return ((RequestMessageReference)interfaceOperation.getInputMessageReference()).createAction(application);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation#sendRequest(org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext)
     */
    public void sendRequest(ActionContext request) throws SystemFault {

        log.trace("sendRequest begin");

        // if service is defined on my engine, i.e. partner models
        // my application, this method is only allowed if mep is
        // in-only

        if (sr.isMyService()) {
            if (interfaceOperation.getMessageExchangePattern() == MEP.IN_ONLY) {
                if (log.isDebugEnabled()) {
                    log.debug("sending request to 'my' service operation " + operationName + ". mep is in-only");
                }
                sendAction(request);
            } else {
                throw new SystemFault(application, "Invocation of partner which models 'my' application and MEP "
                        + interfaceOperation.getMessageExchangePattern()
                        + " requires a blocking invocation, i.e. using method"
                        + " 'ActionContext sendRequest(ActionContext request, long timeout)' or"
                        + " 'Element sendRequest(Element element, long timeout).'");
            }
        } else {
            if (log.isDebugEnabled()) {
                log.debug("sending partner service request to operation "
                        + operationName
                        + " via non blocking invocation");
            }
            sendAction(request);
        }

        log.trace("sendRequest end");

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation#sendRequest(org.w3c.dom.Element)
     */
    public Element sendRequest(Element element, long timeout) throws PartnerFault, SystemFault {
        log.trace("sendRequest begin");
        ActionContext requestContext = createRequestAction();
        Message message = requestContext.getMessage();
        if (element instanceof Content) {
            message.setContent((Content)element);
        } else {
            Content imported = (Content)message.importNode(element, true);
            message.setContent(imported);
        }
        if (getMetaData().getMessageExchangePattern() == MEP.IN_ONLY) {
            sendAction(requestContext);
            return null;
        }
        PartnerCallback partnerCallback = new PartnerCallback(requestContext.getMessageID());
        engine.registerPartnerCallback(partnerCallback);
        if (log.isDebugEnabled()) {
            log.debug("sending partner service request to operation " + operationName + ".");
        }
        sendAction(requestContext);
        if (log.isDebugEnabled()) {
            log.debug("user employed blocking invocation");
            log.debug("blocking '" + timeout + "' milliseconds for response");
        }
        long waited = timeout;
        long current = System.currentTimeMillis();
        synchronized (partnerCallback) {
            while (partnerCallback.getPartnerResponse() == null && waited >= timeout) {
                try {
                    engine.addWaitingThread(Thread.currentThread());
                    partnerCallback.wait(timeout);
                    waited = System.currentTimeMillis() - current;
                    engine.removeWaitingThread(Thread.currentThread());
                } catch (InterruptedException e) {
                    throw new SystemFault(application, "Interrupted while waiting for response.");
                }
            }
        }
        ActionContext responseContext = partnerCallback.getPartnerResponse();
        if (responseContext == null) {
            if (log.isDebugEnabled()) {
                log.debug("notified. timed out waiting for response on operation " + operationName + ".");
            }
            throw new SystemFault(application, "Timed out while waiting for partner response.");
        }
        if (log.isDebugEnabled()) {
            log.debug("notified. partner response received from operation " + operationName + ".");
        }
        if (responseContext instanceof FaultContext) {
            ApplicationFault applicationFault = ((FaultContext)responseContext).getFault();
            if (applicationFault instanceof SystemFault) {
                throw (SystemFault)applicationFault;
            }
            QName faultName = applicationFault.getFaultName();
            Element payload = applicationFault.toMessage().getContent();
            throw new PartnerFault(faultName, payload);

        }
        log.trace("sendRequest end");
        return partnerCallback.getPartnerResponse().getMessage().getContent();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation#sendRequest(org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext,
     *      long)
     */
    public ActionContext sendRequest(ActionContext request, long timeout) throws SystemFault {
        log.trace("sendRequest begin");
        PartnerCallback partnerCallback = new PartnerCallback(request.getMessageID());
        engine.registerPartnerCallback(partnerCallback);
        if (log.isDebugEnabled()) {
            log.debug("sending partner service request to operation " + operationName + ".");
        }
        sendAction(request);
        if (log.isDebugEnabled()) {
            log.debug("user employed blocking invocation");
            log.debug("blocking '" + timeout + "' milliseconds for response");
        }
        long waited = timeout;
        long current = System.currentTimeMillis();
        synchronized (partnerCallback) {
            while (partnerCallback.getPartnerResponse() == null && waited >= timeout) {
                try {
                    engine.addWaitingThread(Thread.currentThread());
                    partnerCallback.wait(timeout);
                    waited = System.currentTimeMillis() - current;
                    engine.removeWaitingThread(Thread.currentThread());
                } catch (InterruptedException e) {
                    throw new SystemFault(application, "Interrupted while waiting for partner response.");
                }
            }
        }
        ActionContext responseContext = partnerCallback.getPartnerResponse();
        if (responseContext == null) {
            if (log.isDebugEnabled()) {
                log.debug("notified. timed out waiting for response on operation " + operationName + ".");
            }
            throw new SystemFault(application, "Timed out while waiting for response.");
        }
        if (log.isDebugEnabled()) {
            log.debug("notified. partner response received from operation " + operationName + ".");
        }
        log.trace("sendRequest end");
        return responseContext;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getAction()
     */
    public String getAction() {
        return sr.getEndpointReference(endpointName).getOperationReference(operationName).getRequestAction()
                .getAction();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getID()
     */
    public String getID() {
        if (id == null) {
            id = getPartner() + "%" + getRole() + "%" + getAction();
        }
        return id;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getPartner()
     */
    public QName getPartner() {
        return sr.getParent().getReferencedComponentName();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getRole()
     */
    public String getRole() {
        return sr.getRoleName();
    }

    private void sendAction(ActionContext request) throws SystemFault {
        if (endpointName == null) {
            SystemFault sf = null;
            Iterator<EndpointReference> itr = sr.getEndpointReferences();
            while (itr.hasNext()) {
                EndpointReference epr = itr.next();
                try {
                    if (log.isDebugEnabled()) {
                        log.debug("invoking endpoint '" + epr.getEndpointName() + "'");
                    }
                    epr.sendAction(request);
                    sf = null; // previous attempt may have failed
                    break; // success. no need to try other epr's
                } catch (SystemFault tmp) {
                    if (tmp.getFaultCode().equals(TransportFault.TRANSPORT_ERROR)) {
                        sf = tmp;
                        if (log.isDebugEnabled()) {
                            log.debug("encountered transient error when invoking partner endpoint '"
                                    + epr.getEndpointName()
                                    + "'. "
                                    + sf.getMessage()
                                    + ". attempting invocation of alternate endpoint(s), if any.");
                        }
                    } else {
                        throw tmp;
                    }
                }
            }
            if (sf != null) {
                throw sf;
            }
        } else {
            EndpointReference epr = sr.getEndpointReference(endpointName);
            if (epr == null) {
                throw new SystemFault(application, "Endpoint '"
                        + endpointName
                        + "' is undefined on service "
                        + sr.getReferencedComponentName());
            }
            epr.sendAction(request);
        }
    }

}
