/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.engine.spring10.impl;

import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineException;
import org.bluestemsoftware.open.eoa.ext.engine.spring10.proxy.PartnerServiceProxy;
import org.bluestemsoftware.specification.eoa.ApplicationClassLoader;
import org.bluestemsoftware.specification.eoa.application.spring.AbstractRole;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerRole;
import org.bluestemsoftware.specification.eoa.component.application.rt.RoleReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.springframework.context.ApplicationContext;

public class PartnerApplicationImpl implements PartnerApplication {

    private EngineReference engineReference;
    private Map<QName, PartnerServiceProxy> partnerProxies;

    public PartnerApplicationImpl(EngineReference engineReference, ApplicationContext applicationContext)
            throws SpringEngineException {
        
        ApplicationClassLoader acl = (ApplicationClassLoader)Thread.currentThread().getContextClassLoader();

        this.engineReference = engineReference;

        // retrieve beans mapped to partner services and create a proxy
        // for each

        partnerProxies = new HashMap<QName, PartnerServiceProxy>();
        
        // if we modeling 'my' application as a partner, i.e. to allow for
        // invocation of self, we do not create proxies, i.e. we enforce
        // that user uses blocking method invocations which employ a callback
        // object (no user code to handle response)
        
        if (engineReference.isMyEngine()) {
            return;
        }

        for (ServiceReference sr : engineReference.getServiceReferences()) {
            
            RoleReference rr = sr.getCorrespondingRoleReference();

            // bean is optional for partner role refs, i.e.
            // either no callbacks expected or user used sync
            // programming style

            String beanType = ((AbstractRole)rr).getBeanType();
            
            if (beanType != null) {
                Class<?> type = null;
                try {
                    type = acl.loadClass(beanType);
                } catch (ClassNotFoundException cfe) {
                    throw new SpringEngineException("ClassNotFound - beanType '"
                            + beanType
                            + "' defined on 'partner' role '"
                            + rr.getRoleName()
                            + " is undefined.");
                }
                Object bean = null;
                Map<?,?> result = applicationContext.getBeansOfType(type, false, false);
                if (result != null) {
                    if (result.size() > 1) {
                        throw new SpringEngineException("More than one bean of type '"
                                + beanType
                                + "' defined. Expected exactly one instance.");
                    }
                    bean = result.values().iterator().next();
                }
                if (bean == null) {
                    throw new SpringEngineException("Singleton bean of type '"
                            + beanType
                            + "' required by 'partner' role "
                            + rr.getRoleName()
                            + "' is undefined. Bean of indicated type must be defined"
                            + " within engine configuration.");
                }
                PartnerServiceProxy proxy = new PartnerServiceProxy(sr, bean);
                partnerProxies.put(sr.getReferencedComponentName(), proxy);
            }

        }

    }

    public PartnerServiceProxy getPartnerServiceProxy(QName serviceName) {
        return partnerProxies.get(serviceName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication#createPartnerOperation(java.lang.String,
     *      javax.xml.namespace.QName)
     */
    public PartnerOperation createPartnerOperation(String roleName, QName operationName) {
        PartnerRole partnerRole = getPartnerRole(roleName);
        if (partnerRole == null) {
            throw new IllegalArgumentException("Role '"
                    + roleName
                    + "' not defined on partner "
                    + engineReference.getReferencedComponentName());
        }
        return partnerRole.createOperation(operationName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication#getName()
     */
    public QName getName() {
        return engineReference.getReferencedComponentName();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication#getPartnerRole(java.lang.String)
     */
    public PartnerRole getPartnerRole(String roleName) {
        return (PartnerRole)engineReference.getServiceReference(roleName);
    }

}
