/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.engine.spring10.impl;

import java.io.File;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.engine.spring.SpringEngine;
import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineConfiguration;
import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineException;
import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineConfiguration.ModuleInfo;
import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineConfiguration.WebModuleInfo;
import org.bluestemsoftware.open.eoa.ext.engine.spring10.proxy.MyServiceProxy;
import org.bluestemsoftware.open.eoa.ext.engine.spring10.proxy.PartnerServiceProxy;
import org.bluestemsoftware.open.eoa.ext.engine.spring10.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.engine.spring10.util.PartnerCallback;
import org.bluestemsoftware.specification.eoa.ApplicationClassLoader;
import org.bluestemsoftware.specification.eoa.ModuleDependency;
import org.bluestemsoftware.specification.eoa.application.spring.AbstractRole;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.ComponentDependencies;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationReference;
import org.bluestemsoftware.specification.eoa.component.application.rt.RoleReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.ext.ExecutableException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ByteArrayResource;
import org.w3c.dom.Document;

public class SpringEngineImpl implements SpringEngine.Provider {

    private static Log log = SystemContext.getContext().getSystem().getLog(SpringEngine.class);

    private SpringEngine consumer;
    private SpringEngineConfiguration configuration;
    private GenericApplicationContext applicationContext;
    private Map<QName, MyServiceProxy> myProxies;
    private Map<String, PartnerCallback> partnerCallbacks;
    private Map<QName, PartnerApplication> partners;
    private Set<Thread> waitingThreads;

    private Boolean isSuspended = Boolean.FALSE;

    public SpringEngineImpl(SpringEngineConfiguration configuration) {
        this.configuration = configuration;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_init()
     */
    public void spi_init() throws SpringEngineException {

        ApplicationClassLoader acl = (ApplicationClassLoader)Thread.currentThread().getContextClassLoader();

        // extend static class which models our application context and which
        // wraps root spring application context which may be shared by contexts
        // created by modules

        applicationContext = new GenericApplicationContext();
        XmlBeanDefinitionReader xmlReader = new XmlBeanDefinitionReader(applicationContext);

        // unfortunately spring has no dom resource, so we must convert dom
        // to a byte stream so that it can be reparsed by spring as a dom ...

        byte[] bytes = null;
        try {
            Document beanDefinitions = configuration.getBeanDefinitions();
            StringWriter stringWriter = new StringWriter();
            DOMSerializer.serializeNode(beanDefinitions, stringWriter, "UTF-8", true);
            bytes = stringWriter.toString().getBytes("UTF-8");
        } catch (Exception ex) {
            throw new SpringEngineException("Error parsing bean definitions. ", ex);
        }

        xmlReader.loadBeanDefinitions(new ByteArrayResource(bytes));

        // to provide support for variable placeholders, run bean definitions
        // through post processor before refreshing context

        PropertyPlaceholderConfigurer pphc = new PropertyPlaceholderConfigurer();
        pphc.postProcessBeanFactory(applicationContext.getBeanFactory());

        // refresh the context which will instantiate beans and inject any
        // dependencies

        try {
            applicationContext.refresh();
        } catch (Exception ex) {
            throw new SpringEngineException("Error loading bean definitions. " + ex.getMessage());
        }

        // deploy optional modules (only web modules are currently supported). note
        // if web application requires access to parent context created above, it is
        // user's responsibility to 'wire' child web application context to parent
        // using spring config parameters defined within web-inf. for rqmnts, see
        // org.springframework.web.context.ContextLoader.loadParentContext()

        ComponentContext ctx = consumer.getApplication().getComponentContext();
        ComponentDependencies cd = (ComponentDependencies)ctx.getDeployment().getDependencies();
        Map<String, ModuleDependency> moduleDependencies = cd.getModuleDependencies();

        Server server = SystemContext.getContext().getSystem().getServer();
        HTTPServerFeature httpServer = server.getFeature(HTTPServerFeature.class);

        if (moduleDependencies.size() > 0) {
            if (httpServer == null) {
                throw new SpringEngineException("Error deploying web module. Feature "
                        + HTTPServerFeature.class
                        + " not enabled.");
            }
        }

        for (ModuleInfo moduleInfo : configuration.getModuleInfos()) {
            if (moduleInfo instanceof WebModuleInfo == false) {
                throw new SpringEngineException("Unsupported module type " + moduleInfo.getClass());
            }
            if (httpServer == null) {
                throw new SpringEngineException("Error deploying web module. Feature "
                        + HTTPServerFeature.class
                        + " not enabled.");
            }
            WebModuleInfo wmi = (WebModuleInfo)moduleInfo;
            String ref = wmi.getRef();
            ModuleDependency md = moduleDependencies.get(ref);
            if (md == null) {
                throw new SpringEngineException("No ModuleDependency found matching ref " + ref);
            }
            if (md.getExtension().equals("war") == false) {
                throw new SpringEngineException("Expected type 'war' for ModuleDependency " + ref);
            }

            String rootContext = normalizeContext(wmi.getRootContext());
            File webArchive = md.getFile();
            try {
                httpServer.deployWebModule(webArchive, acl, rootContext);
            } catch (HTTPServerFeatureException he) {
                throw new SpringEngineException(he.getMessage());
            }

            if (log.isDebugEnabled()) {
                log.debug("deployed web module to context '" + wmi.getRootContext() + "'.");
            }

        }

        // retrieve bean matching type defined on each my role reference
        // and create a service proxy for each

        myProxies = new HashMap<QName, MyServiceProxy>();

        ApplicationRT application = consumer.getApplication();
        EngineReference myER = consumer.getEngineReference(consumer.getName());
        ApplicationReference myAR = application.getApplicationReference(application.getName());
        for (RoleReference rr : myAR.getRoleReferences()) {
            String beanType = ((AbstractRole)rr).getBeanType();
            Class<?> type = null;
            try {
                type = acl.loadClass(beanType);
            } catch (ClassNotFoundException cfe) {
                throw new SpringEngineException("ClassNotFound - beanType '"
                        + beanType
                        + "' defined on 'my' role '"
                        + rr.getRoleName()
                        + " is undefined.");
            }
            Object bean = null;
            Map<?, ?> result = applicationContext.getBeansOfType(type, false, false);
            if (result != null) {
                if (result.size() > 1) {
                    throw new SpringEngineException("More than one bean of type '"
                            + beanType
                            + "' defined within application context. Expected exactly one instance.");
                }
                bean = result.values().iterator().next();
            }
            if (bean == null) {
                throw new SpringEngineException("Singleton bean of type '"
                        + beanType
                        + "' required by 'my' role "
                        + rr.getRoleName()
                        + "' is undefined. Bean of indicated type must be defined"
                        + " within engine configuration.");
            }
            ServiceReference sr = myER.getServiceReference(rr.getRoleName());
            myProxies.put(sr.getReferencedComponentName(), new MyServiceProxy(sr, bean));
        }

        this.partnerCallbacks = new HashMap<String, PartnerCallback>();
        this.waitingThreads = new HashSet<Thread>();

        // process all engine references as partners, i.e. we include
        // my engine as well which will allow an engine to invoke an
        // operation on itself. this is important because we want to
        // give user option to run message through eoa application
        // layer (so that appliction module will process the message)
        // and through binding layer (an endpoint with a virtual
        // transport should be defined)

        this.partners = new HashMap<QName, PartnerApplication>();
        for (EngineReference er : consumer.getEngineReferences()) {
            PartnerApplication pa = new PartnerApplicationImpl(er, applicationContext);
            partners.put(er.getReferencedComponentName(), pa);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_isSuspended()
     */
    public boolean spi_isSuspended() {
        synchronized (isSuspended) {
            return isSuspended.booleanValue();
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_receiveAction(org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference,
     *      org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext)
     */
    public void spi_receiveAction(EndpointReference epr, ActionContext actionContext) {
        log.trace("spi_receiveAction begin");
        synchronized (isSuspended) {
            while (isSuspended) {
                try {
                    waitingThreads.add(Thread.currentThread());
                    isSuspended.wait();
                    waitingThreads.remove(Thread.currentThread());
                } catch (InterruptedException ie) {
                    throw new SystemFault(consumer.getApplication(), "Suspended thread interrupted.");
                }
            }
        }
        EndpointActionReference ear = epr.getEndpointActionReference(actionContext.getAction());
        Direction direction = null;
        if (ear == null) {
            direction = Direction.OUT; // it's a system fault
        } else {
            direction = ear.getReferencedComponent().getInterfaceAction().getDirection();
        }
        if (direction == Direction.IN) {
            handleMyServiceRequest(epr, actionContext);
        } else {
            handlePartnerServiceResponse(epr, actionContext);
        }
        log.trace("spi_receiveAction end");
    }

    private void handleMyServiceRequest(EndpointReference endpointReference, ActionContext actionContext) {
        log.debug("handling my service request");
        ServiceReference serviceReference = endpointReference.getParent();
        MyServiceProxy myServiceProxy = myProxies.get(serviceReference.getReferencedComponentName());
        myServiceProxy.handleRequest(endpointReference, actionContext);
    }

    private void handlePartnerServiceResponse(EndpointReference epr, ActionContext actionContext) {

        log.debug("handling partner service response");

        PartnerCallback partnerCallback = null;
        synchronized (partnerCallbacks) {
            partnerCallback = partnerCallbacks.remove(actionContext.getRelatesTo());
        }

        // if partner callback found, user used blocking invocation.
        // a thread is waiting within sendRequest method on class
        // PartnerOperationImpl

        if (partnerCallback != null) {
            log.debug("user employed blocking invocation. notifying callback");
            synchronized (partnerCallback) {
                partnerCallback.setPartnerResponse(actionContext);
                partnerCallback.notify();
            }
            return;
        }

        // user employed non blocking invocation api. pass response to bean
        // class mapped to partner role reference

        log.debug("user employed non-blocking invocation");

        EngineReference er = epr.getParent().getParent();
        PartnerApplication pa = partners.get(er.getReferencedComponentName());
        QName sn = epr.getParent().getReferencedComponentName();
        PartnerServiceProxy proxy = ((PartnerApplicationImpl)pa).getPartnerServiceProxy(sn);

        // because choice of method used to invoke partner is made
        // by user at runtime, we cannot know if user failed to
        // map a class to partner service reference until now

        if (proxy == null) {
            ApplicationRT app = consumer.getApplication();
            throw new SystemFault(app, "No spring bean mapped to role reference "
                    + epr.getParent().getCorrespondingRoleReference().getFragmentIdentifier());

        }
        
        log.debug("invoking handleResponse on partner service proxy");
        proxy.handleResponse(epr, actionContext);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (SpringEngine)consumer;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_resume()
     */
    public void spi_resume() throws ExecutableException {
        synchronized (isSuspended) {
            isSuspended = Boolean.FALSE;
            isSuspended.notifyAll();
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_destroy()
     */
    public void spi_destroy() {
        if (applicationContext != null) {
            applicationContext.close();
        }
        for (Thread waitingThread : waitingThreads) {
            synchronized (waitingThread) {
                waitingThread.interrupt();
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT$Provider#spi_suspend()
     */
    public void spi_suspend() throws ExecutableException {
        synchronized (isSuspended) {
            isSuspended = Boolean.TRUE;
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.engine.spring10.SpringEngine$Provider#spi_createOperation(org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference,
     *      javax.xml.namespace.QName)
     */
    public PartnerOperation spi_createOperation(ServiceReference serviceReference, QName operationName) {
        return new PartnerOperationImpl(this, serviceReference, null, operationName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.engine.spring10.SpringEngine$Provider#spi_createOperation(org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference,
     *      java.lang.String, javax.xml.namespace.QName)
     */
    public PartnerOperation spi_createOperation(ServiceReference serviceReference, String endpointName, QName operationName) {
        return new PartnerOperationImpl(this, serviceReference, endpointName, operationName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.engine.spring10.SpringEngine$Provider#spi_getPartner(javax.xml.namespace.QName)
     */
    public PartnerApplication spi_getPartner(QName engineName) {
        return partners.get(engineName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.engine.spring.SpringEngine$Provider#spi_getApplicationContext()
     */
    public ApplicationContext spi_getApplicationContext() {
        return applicationContext;
    }

    /*
     * Registers a callback. Note that callbacks are not persistent, i.e. because blocking
     * style invocation relies upon threads and wait/notify mechanism.
     */
    synchronized void registerPartnerCallback(PartnerCallback partnerCallback) {
        partnerCallbacks.put(partnerCallback.getRequestMessageID(), partnerCallback);
    }

    /*
     * adjusts context string. note that this logic was taken from jetty's web app deployer.
     */
    private String normalizeContext(String rootContext) {
        if (rootContext.equalsIgnoreCase("root") || rootContext.equalsIgnoreCase("root/")) {
            rootContext = "/";
        } else {
            if (!rootContext.startsWith("/")) {
                rootContext = "/" + rootContext;
            }
            int length = rootContext.length();
            if (rootContext.endsWith("/") && length > 0) {
                rootContext = rootContext.substring(0, length - 1);
            }
        }
        return rootContext;
    }

    SpringEngine getConsumer() {
        return consumer;
    }

    void addWaitingThread(Thread waitingThread) {
        waitingThreads.add(waitingThread);
    }

    void removeWaitingThread(Thread waitingThread) {
        waitingThreads.remove(waitingThread);
    }

}
