/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.engine.spring10.proxy;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineException;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerRole;
import org.bluestemsoftware.specification.eoa.component.application.rt.RoleReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class PartnerServiceProxy {

    private static Log log = SystemContext.getContext().getSystem().getLog(PartnerRole.class);

    private Map<QName, OperationInfo> operations = new HashMap<QName, OperationInfo>();
    private Object partnerService;
    private Interface intrface;

    public PartnerServiceProxy(ServiceReference serviceReference, Object partnerService)
            throws SpringEngineException {
        
        if (serviceReference.getEndpointReferences().hasNext() == false) {
            throw new SpringEngineException("Service "
                    + serviceReference.getReferencedComponentName()
                    + " defined on engine "
                    + serviceReference.getParent().getReferencedComponentName()
                    + " defines no remotely invocable endpoints.");
        }

        this.partnerService = partnerService;
        RoleReference roleReference = serviceReference.getCorrespondingRoleReference();
        this.intrface = roleReference.getRole().getReferencedComponent();

        for (InterfaceOperation io : intrface.getOperations()) {

            Class<?> clazz = partnerService.getClass();

            Class<?>[] signature = new Class[] { ActionContext.class };
            Method method = null;
            String methodName = io.getName().getLocalPart() + "Response";
            try {
                method = clazz.getDeclaredMethod(methodName, signature);
                if (!method.getReturnType().equals(Void.TYPE)) {
                    throw new SpringEngineException("Method "
                            + method.toGenericString()
                            + " defined on service class "
                            + clazz.getName()
                            + " defines a return type. Expected "
                            + Void.TYPE);
                }
            } catch (NoSuchMethodException ne) {
                throw new SpringEngineException("Service class "
                        + clazz.getName()
                        + " missing required method '"
                        + methodName
                        + "' with signature (ActionContext) and"
                        + " return type void.");
            } catch (Exception ex) {
                throw new SpringEngineException(ex.getMessage());
            }

            if (method.getExceptionTypes().length > 0) {
                throw new SpringEngineException("Method '"
                        + methodName
                        + " defined on Service class "
                        + clazz.getName()
                        + " declares Exceptions. No exceptions are allowed.");
            }

            operations.put(io.getName(), new OperationInfo(method, io));

        }

    }

    public void handleResponse(EndpointReference er, ActionContext actionContext) {
        log.trace("handleResponse begin");
        log.debug("handling partner service response");
        EndpointActionReference ear = er.getEndpointActionReference(actionContext.getAction());
        QName operationName = ear.getParent().getEndpointOperationName();
        OperationInfo operationInfo = operations.get(operationName);
        Method method = operationInfo.getMethod();
        try {
            if (log.isDebugEnabled()) {
                log.debug("invoking method '"
                        + method.toGenericString()
                        + "' on class '"
                        + partnerService.getClass().getName()
                        + "'");
            }
            method.invoke(partnerService, new Object[] { actionContext });
        } catch (Throwable th) {
            if (th instanceof InvocationTargetException) {
                th = ((InvocationTargetException)th).getCause();
            }
            log.error("Service class "
                    + partnerService.getClass().getName()
                    + " threw an exception while handling partner response delivered to method "
                    + method.toGenericString()
                    + ". "
                    + th);
        }
        log.trace("handleResponse end");
    }

    private static class OperationInfo {

        private Method method;
        private InterfaceOperation metadata;

        public OperationInfo(Method method, InterfaceOperation metadata) {
            this.method = method;
            this.metadata = metadata;
        }

        public InterfaceOperation getMetadata() {
            return metadata;
        }

        public Method getMethod() {
            return method;
        }

    }

}
