/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.engine.spring10.proxy;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineException;
import org.bluestemsoftware.specification.eoa.application.spring.MyFault;
import org.bluestemsoftware.specification.eoa.application.spring.MyOperation;
import org.bluestemsoftware.specification.eoa.application.spring.MyRole;
import org.bluestemsoftware.specification.eoa.component.application.rt.ApplicationRT;
import org.bluestemsoftware.specification.eoa.component.application.rt.RoleReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference.ResponseActionReference;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceFault;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceFaultReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.MEP;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.message.MessagePart;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class MyServiceProxy {

    private static final Log log = SystemContext.getContext().getSystem().getLog(MyRole.class);

    private Map<QName, OperationInfo> operations = new HashMap<QName, OperationInfo>();
    private Object myService;
    private String roleName;
    private Interface intrface;

    public MyServiceProxy(ServiceReference serviceReference, Object myService) throws SpringEngineException {

        this.myService = myService;

        RoleReference roleReference = serviceReference.getCorrespondingRoleReference();
        this.roleName = roleReference.getRoleName();
        this.intrface = roleReference.getRole().getReferencedComponent();

        for (InterfaceOperation io : intrface.getOperations()) {

            Class<?> clazz = myService.getClass();

            // first we look for signature that refs objects within
            // the eoa api, i.e. which enables a non-blocking style

            Class<?>[] signature = new Class[] { MyOperation.class, ActionContext.class };
            boolean isBlockingInvocation = false;
            Method method = null;
            try {

                method = clazz.getDeclaredMethod(io.getName().getLocalPart(), signature);
                isBlockingInvocation = false;

                if (!method.getReturnType().equals(Void.TYPE)) {
                    throw new SpringEngineException("Method "
                            + method.toGenericString()
                            + " defined on service class "
                            + clazz.getName()
                            + " defines a return type. Expected "
                            + Void.TYPE);
                }

                // because response is returned asynchronously via
                // MyOperation object, we do not allow checked
                // exceptions with this signature. fault response
                // should be returned asynchronously, i.e. via
                // MyOperation object

                for (Class<?> exceptionType : method.getExceptionTypes()) {
                    if (Exception.class.isAssignableFrom(exceptionType)) {
                        throw new SpringEngineException("Method "
                                + method.toGenericString()
                                + " defined on service class "
                                + clazz.getName()
                                + " declares checked exception "
                                + exceptionType.getName()
                                + ". Checked exceptions not allowed. Fault responses"
                                + " must be returned asynchronously via 'MyOperation'"
                                + " method 'sendResponse(ActionContext)'");
                    }
                }
            } catch (NoSuchMethodException fallthrough) {
            } catch (Exception ex) {
                throw new SpringEngineException(ex.getMessage());
            }

            // alternatively, user may define signature that does not
            // ref the eoa api, i.e. which uses only dom api. the
            // drawback here is that it must be a blocking request,
            // user has no access to application data, message props,
            // etc ...

            if (method == null) {
                signature = new Class[] { Element.class };
                try {
                    method = clazz.getDeclaredMethod(io.getName().getLocalPart(), signature);
                    isBlockingInvocation = true;
                } catch (NoSuchMethodException ne) {
                    throw new SpringEngineException("Service class "
                            + clazz.getName()
                            + " missing required method '"
                            + io.getName()
                            + "' with signature (PartnerOperation, ActionContext) and"
                            + " return type void. Alternatively signature (Element) and"
                            + " return type Element may be used.");
                } catch (Exception ex) {
                    throw new SpringEngineException(ex.getMessage());
                }
                if (!method.getReturnType().equals(Element.class)) {
                    throw new SpringEngineException("Method "
                            + method.toGenericString()
                            + " defined on service class "
                            + clazz.getName()
                            + " defines invalid return type "
                            + method.getReturnType().getName()
                            + ". Expected "
                            + Element.class.getName()
                            + ".");
                }
                InterfaceMessage input = io.getInputMessageReference().getReferencedComponent();
                if (input.getParts().size() > 1) {
                    throw new SpringEngineException("Method "
                            + method.toGenericString()
                            + " defined on service class "
                            + clazz.getName()
                            + " uses (Element) signature with Element as return type."
                            + " But, input message "
                            + input.getName()
                            + " is a multipart message which requires signature"
                            + " (PartnerOperation, ActionContext) and return type void.");
                }
                InterfaceMessage output = io.getOutputMessageReference().getReferencedComponent();
                if (output != null && output.getParts().size() > 1) {
                    throw new SpringEngineException("Method "
                            + method.toGenericString()
                            + " defined on service class "
                            + clazz.getName()
                            + " uses (Element) signature with Element as return type."
                            + " But, output message "
                            + output.getName()
                            + " is a multipart message which requires signature"
                            + " (PartnerOperation, ActionContext) and return type void.");
                }
                
                // the blocking style api allows checked exceptions, as long as they
                // are either instances of MyFault or SystemFault (which is unchecked).
                // note that we can't tell if the fault is declared until the exception
                // is actually thrown, i.e. we need an object to examine name of fault
                
                for (Class<?> exceptionType : method.getExceptionTypes()) {
                    if (!MyFault.class.isAssignableFrom(exceptionType)) {
                        if (!SystemFault.class.isAssignableFrom(exceptionType)) {
                            throw new SpringEngineException("Method "
                                    + method.toGenericString()
                                    + " defined on service class "
                                    + clazz.getName()
                                    + " declares checked exception "
                                    + exceptionType.getName()
                                    + ". Only exceptions assignable from "
                                    + MyFault.class.getName()
                                    + " or "
                                    + SystemFault.class.getName()
                                    + " are allowed.");
                        }
                    }
                }
                
            }

            operations.put(io.getName(), new OperationInfo(method, io, isBlockingInvocation));

        }

    }

    public void handleRequest(EndpointReference er, ActionContext actionContext) {
        log.trace("handleRequest begin");
        EndpointActionReference ear = er.getEndpointActionReference(actionContext.getAction());
        QName operationName = ear.getParent().getEndpointOperationName();
        OperationInfo operationInfo = operations.get(operationName);
        if (operationInfo.isBlockingInvocation()) {
            handleBlockingInvocation(er, actionContext);
        } else {
            handleNonBlockingInvocation(er, actionContext);
        }
        log.trace("handleRequest end");
    }

    private void handleNonBlockingInvocation(EndpointReference er, ActionContext actionContext) {

        log.debug("handling non blocking invocation");

        EndpointActionReference ear = er.getEndpointActionReference(actionContext.getAction());
        QName operationName = ear.getParent().getEndpointOperationName();
        OperationInfo operationInfo = operations.get(operationName);
        String relatesTo = actionContext.getMessageID();

        // method has no return type. user shouldn't throw any exceptions. if
        // they do, translate to a system fault (if it's not a system fault)

        Method method = null;
        try {
            method = operationInfo.getMethod();
            MyOperation myOperation = new MyOperation(ear.getParent(), relatesTo);
            if (log.isDebugEnabled()) {
                log.debug("invoking method '"
                        + method.toGenericString()
                        + "' on class '"
                        + myService.getClass().getName()
                        + "'");
            }
            method.invoke(myService, new Object[] { myOperation, actionContext });
        } catch (Throwable th) {
            if (th instanceof InvocationTargetException) {
                th = ((InvocationTargetException)th).getCause();
            }
            InterfaceOperation io = operationInfo.getMetadata();
            if (io.getMessageExchangePattern() == MEP.IN_ONLY) {
                log.error("Method '"
                        + method.toGenericString()
                        + "' defined on user class "
                        + myService.getClass().getName()
                        + " which implements 'my' role '"
                        + roleName
                        + "' threw an exception while handling 'in-only' MEP. "
                        + th);
                return;
            }
            ActionContext rc = convertFaultToResponseContext(io, relatesTo, method, operationName, ear, th);
            er.sendAction(rc);
        }

    }

    private void handleBlockingInvocation(EndpointReference er, ActionContext actionContext) {
        
        log.debug("handling blocking invocation");

        EndpointActionReference ear = er.getEndpointActionReference(actionContext.getAction());
        QName operationName = ear.getParent().getEndpointOperationName();
        OperationInfo operationInfo = operations.get(operationName);
        InterfaceOperation interfaceOperation = operationInfo.getMetadata();
        Message request = actionContext.getMessage();
        String relatesTo = actionContext.getMessageID();

        // user defined method should only throw instances of system
        // fault, or my fault (which must map to a fault declared on
        // abstract operation. return value must represent normal
        // response message

        Element element = request.getContent();
        Method method = operationInfo.getMethod();
        ActionContext responseContext = null;
        try {
            if (log.isDebugEnabled()) {
                log.debug("invoking method '"
                        + method.toGenericString()
                        + "' on class '"
                        + myService.getClass().getName()
                        + "'");
            }
            Element response = (Element)method.invoke(myService, new Object[] { element });
            if (interfaceOperation.getMessageExchangePattern() == MEP.IN_OUT) {
                if (response == null) {
                    log.error("Method '"
                            + method.toGenericString()
                            + "' defined on user class "
                            + myService.getClass().getName()
                            + " which implements 'my' role '"
                            + roleName
                            + "' returned a null response, but MEP 'in-out'. "
                            + " Returning system fault response to caller.");
                    ApplicationRT app = er.getRootComponent().getApplication();
                    SystemFault sf = new SystemFault(app, "A null response was returned by" + " application.");
                    responseContext = er.createSystemFaultAction(relatesTo, sf);
                } else {
                    QName responseType = new QName(response.getNamespaceURI(), response.getLocalName());
                    InterfaceMessage am = interfaceOperation.getOutputMessageReference().getReferencedComponent();
                    MessagePart payload = am.getPart(MessagePart.PAYLOAD);
                    if (responseType.equals(payload.getSchemaComponentQName())) {
                        ResponseActionReference rar = ear.getParent().getResponseAction();
                        responseContext = rar.createAction(relatesTo);
                        if (response instanceof Content) {
                            responseContext.getMessage().setContent((Content)response);
                        } else {
                            Node imported = responseContext.getMessage().importNode(response, true);
                            responseContext.getMessage().setContent((Content)imported);
                        }
                    } else {
                        log.error("Method '"
                                + method.toGenericString()
                                + "' defined on user class "
                                + myService.getClass().getName()
                                + " which implements 'my' role '"
                                + roleName
                                + "' returned response of type "
                                + responseType
                                + ". Expected "
                                + payload.getSchemaComponentName()
                                + ".");
                        ApplicationRT app = er.getRootComponent().getApplication();
                        SystemFault sf = new SystemFault(app, "An incorrectly typed response was"
                                + " returned by application.");
                        responseContext = er.createSystemFaultAction(relatesTo, sf);
                    }
                }
            } else {
                if (response != null) {
                    log.error("Method '"
                            + method.toGenericString()
                            + "' defined on user class "
                            + myService.getClass().getName()
                            + " which implements 'my' role '"
                            + roleName
                            + "' returned a normal response, but MEP not 'in-out'. "
                            + " Discarding response.");
                    return;
                }
            }
        } catch (Throwable th) {
            if (th instanceof InvocationTargetException) {
                th = ((InvocationTargetException)th).getCause();
            }
            if (interfaceOperation.getMessageExchangePattern() == MEP.IN_ONLY) {
                log.error("Method '"
                        + method.toGenericString()
                        + "' defined on user class "
                        + myService.getClass().getName()
                        + " which implements 'my' role '"
                        + roleName
                        + "' threw an exception while handling 'in-only' MEP. "
                        + th);
                return;
            }
            responseContext = convertFaultToResponseContext(interfaceOperation, relatesTo, method, operationName,
                    ear, th);
        }

        log.debug("returning response via send action on epr");
        er.sendAction(responseContext);

    }

    private ActionContext convertFaultToResponseContext(InterfaceOperation interfaceOperation, String relatesTo, Method method, QName operationName, EndpointActionReference ear, Throwable th) {
        ActionContext responseContext = null;
        EndpointReference er = ear.getParent().getParent();
        if (th instanceof SystemFault) {
            responseContext = er.createSystemFaultAction(relatesTo, (SystemFault)th);
        } else if (th instanceof MyFault) {
            QName faultName = ((MyFault)th).getFaultName();
            InterfaceFaultReference fr = interfaceOperation.getFaultReference(faultName);
            if (fr == null) {
                log.error("Method '"
                        + method.toGenericString()
                        + "' defined on user class "
                        + myService.getClass().getName()
                        + " which implements 'my' role '"
                        + roleName
                        + "' threw an undeclared instance of 'MyFault'. Fault "
                        + faultName
                        + " not declared on operation "
                        + operationName
                        + ".");
                ApplicationRT app = er.getRootComponent().getApplication();
                SystemFault sf = new SystemFault(app, "Application returned an undeclared fault response.");
                responseContext = er.createSystemFaultAction(relatesTo, sf);
            } else {
                Element response = ((MyFault)th).getPayload();
                QName responseType = new QName(response.getNamespaceURI(), response.getLocalName());
                InterfaceFault interfaceFault = fr.getFault();
                InterfaceMessage am = interfaceFault.getReferencedComponent();
                MessagePart payload = am.getPart(MessagePart.PAYLOAD);
                if (responseType.equals(payload.getSchemaComponentQName())) {
                    ResponseActionReference rar = ear.getParent().getFaultAction(faultName);
                    responseContext = rar.createAction(relatesTo);
                    if (response instanceof Content) {
                        responseContext.getMessage().setContent((Content)response);
                    } else {
                        Node imported = responseContext.getMessage().importNode(response, true);
                        responseContext.getMessage().setContent((Content)imported);
                    }
                } else {
                    log.error("Method '"
                            + method.toGenericString()
                            + "' defined on user class "
                            + myService.getClass().getName()
                            + " which implements 'my' role '"
                            + roleName
                            + "' threw MyFault "
                            + faultName
                            + " with payload of type "
                            + responseType
                            + ". Expected "
                            + payload.getSchemaComponentName()
                            + ".");
                    ApplicationRT app = er.getRootComponent().getApplication();
                    SystemFault sf = new SystemFault(app, "An incorrectly typed fault response was"
                            + " returned by application.");
                    responseContext = er.createSystemFaultAction(relatesTo, sf);
                }
            }
        } else {
            ApplicationRT app = er.getRootComponent().getApplication();
            log.error("Method '"
                    + method.toGenericString()
                    + "' defined on user class "
                    + myService.getClass().getName()
                    + " which implements 'my' role '"
                    + roleName
                    + "' threw an unchecked exception while handling operation "
                    + operationName
                    + ": "
                    + th);
            SystemFault sf = new SystemFault(app, "Application threw unchecked exception. " + th.getMessage());
            responseContext = er.createSystemFaultAction(relatesTo, sf);
        }
        return responseContext;
    }

    private static class OperationInfo {

        private Method method;
        private InterfaceOperation metadata;
        private boolean isBlockingInvocation;

        public OperationInfo(Method method, InterfaceOperation metadata, boolean isBlockingInvocation) {
            this.method = method;
            this.metadata = metadata;
            this.isBlockingInvocation = isBlockingInvocation;
        }

        public InterfaceOperation getMetadata() {
            return metadata;
        }

        public Method getMethod() {
            return method;
        }

        public boolean isBlockingInvocation() {
            return isBlockingInvocation;
        }

    }

}
