/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.container.standalone10;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.resource.spi.ConnectionManager;
import javax.transaction.TransactionManager;
import javax.transaction.UserTransaction;
import javax.transaction.xa.XAException;

import org.apache.geronimo.connector.outbound.GenericConnectionManager;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.LocalTransactions;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.NoPool;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.NoTransactions;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.PartitionedPool;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.SinglePool;
import org.apache.geronimo.connector.outbound.connectionmanagerconfig.XATransactions;
import org.apache.geronimo.connector.outbound.connectiontracking.ConnectionTrackingCoordinator;
import org.apache.geronimo.transaction.GeronimoUserTransaction;
import org.apache.geronimo.transaction.manager.TransactionLog;
import org.apache.geronimo.transaction.manager.TransactionManagerImpl;
import org.bluestemsoftware.specification.eoa.DeploymentClassLoader;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionException;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.connector.ConnectorFactory;
import org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainer;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerException;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.ConnectorInfo;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.PoolingSupportType;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.TransactionSupportInfo;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.TransactionSupportType;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.ConnectorInfo.ConnectorTransactionSupportInfo;
import org.bluestemsoftware.specification.eoa.ext.container.standalone10.StandaloneContainerConfiguration.ConnectorInfo.PoolingSupportInfo;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.container.ConnectionSupport;
import org.bluestemsoftware.specification.eoa.system.container.Connector;
import org.bluestemsoftware.specification.eoa.system.container.Container;

public final class StandaloneContainerImpl extends StandaloneContainer {

    private static Log log = SystemContext.getContext().getSystem().getLog(Container.class);
    
    private static final String XA_LOG = "org.apache.geronimo.transaction.log.HOWLLog";
    private static final String NON_XA_LOG = "org.apache.geronimo.transaction.log.UnrecoverableLog";
    
    private StandaloneContainerConfiguration configuration;
    private TransactionManager transactionManager;
    private Map<String, Connector> connectors;
    private DataSourceConnector systemDataSource;
    private UserTransaction userTransaction;

    public StandaloneContainerImpl(ExtensionFactory factory, StandaloneContainerConfiguration configuration) {
        super(factory, new StandaloneContainer.Provider() {
            public void spi_setConsumer(Extension arg0) {
            }
        });
        this.configuration = configuration;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ManageableExtension#init()
     */
    public void init() throws ExtensionException {

        log.debug("init begin");

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();

        try {

            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());

            TransactionSupportInfo transactionSupportInfo = configuration.getTransactionSupportInfo();

            if (transactionSupportInfo != null) {
                createTransactionSupport();
            } else {
                log.info("transaction support disabled");
            }

            if (transactionManager == null) {
                if (configuration.getConnectorInfos().size() > 0) {
                    throw new StandaloneContainerException("Failed to create connectors"
                            + ". Transaction support disabled.");
                }
            } else {
                createConnectors();
                systemDataSource = (DataSourceConnector)connectors.get(DataSourceConnector.SYSTEM_DATA_SOURCE);
            }

            if (systemDataSource == null) {
                log.warn(DataSourceConnector.SYSTEM_DATA_SOURCE + " is undefined");
            }

        } finally {
            thread.setContextClassLoader(cl);
        }

        log.debug("init end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ManageableExtension#destroy()
     */
    public void destroy() {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.container.Container#getConnector(java.lang.String)
     */
    public final Connector getConnector(String name) {
        if (connectors == null) {
            return null;
        }
        return connectors.get(name);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.container.Container#getConnectors()
     */
    @SuppressWarnings("unchecked")
    public Set<Connector> getConnectors() {
        if (connectors == null) {
            return Collections.EMPTY_SET;
        }
        return Collections.unmodifiableSet(new HashSet<Connector>(connectors.values()));
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.container.Container#getSystemDataSource()
     */
    @Override
    public Connector getSystemDataSource() {
        return systemDataSource;
    }
    
    @Override
    public UserTransaction getUserTransaction() {
        return userTransaction;
    }

    private void createTransactionSupport() throws StandaloneContainerException {

        log.debug("creating transaction support");

        TransactionSupportInfo transactionSupportInfo = configuration.getTransactionSupportInfo();
        
        // iterate over connectors to see if xa is used by at least one, in
        // which case, we must use the xa capable transaction log
        
        String className = NON_XA_LOG;
        for (ConnectorInfo connectorInfo : configuration.getConnectorInfos()) {
            ConnectorTransactionSupportInfo cti = connectorInfo.getTransactionSupportInfo();
            TransactionSupportType transactionSupportType = cti.getTransactionSupportType();
            if (transactionSupportType == TransactionSupportType.XA) {
                className = XA_LOG;
                break;
            }
        }    

        TransactionLog transactionLog = null;
        try {
            Class<?> clazz = getExtensionFactory().getFactoryContext().getClassLoader().loadClass(className);
            transactionLog = (TransactionLog)clazz.newInstance();
        } catch (Exception ex) {
            throw new StandaloneContainerException("Error instantiating transactionLog '" + className + "'. " + ex);
        }

        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        try {
            int defaultTransactionTimeoutSeconds = transactionSupportInfo.getDefaultTransactionTimeoutSeconds();
            transactionManager = new TransactionManagerImpl(defaultTransactionTimeoutSeconds, transactionLog);
        } catch (XAException xa) {
            throw new StandaloneContainerException("Error instantiating transaction manager. " + xa);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

        userTransaction = new GeronimoUserTransaction(transactionManager);

        log.info("transaction support enabled");
    }

    private void createConnectors() throws ExtensionException {

        log.debug("creating connectors");

        connectors = new HashMap<String, Connector>();

        for (ConnectorInfo connectorInfo : configuration.getConnectorInfos()) {

            String type = connectorInfo.getType();
            String impl = connectorInfo.getImpl();
            String connectorName = connectorInfo.getConnectorName();

            log.debug("creating connector " + connectorName);

            ConnectorFactory connectorFactory = SystemContext.getContext().getSystem().getConnectorFactory(type,
                    impl);
            if (connectorFactory == null) {
                throw new StandaloneContainerException("Failed to create connector '"
                        + connectorName
                        + "'. No ConnectorFactory extension found matching type ref '"
                        + type
                        + "' and impl '"
                        + impl
                        + "'.");
            }

            Properties properties = connectorInfo.getProperties();

            ConnectorTransactionSupportInfo transactionSupportInfo = connectorInfo.getTransactionSupportInfo();
            TransactionSupportType transactionSupportType = transactionSupportInfo.getTransactionSupportType();

            boolean isXA = false;
            org.apache.geronimo.connector.outbound.connectionmanagerconfig.TransactionSupport transactionSupport;
            if (transactionSupportType == TransactionSupportType.NONE) {
                transactionSupport = NoTransactions.INSTANCE;
            } else if (transactionSupportType == TransactionSupportType.LOCAL) {
                transactionSupport = LocalTransactions.INSTANCE;
            } else {
                isXA = true;
                boolean useTransactionCaching = transactionSupportInfo.isUseTransactionCaching();
                boolean useThreadCaching = transactionSupportInfo.isUseThreadCaching();
                transactionSupport = new XATransactions(useTransactionCaching, useThreadCaching);
            }

            PoolingSupportInfo poolingSupportInfo = connectorInfo.getPoolingSupportInfo();

            PoolingSupportType poolingSupportType = poolingSupportInfo.getPoolingSupportType();

            org.apache.geronimo.connector.outbound.connectionmanagerconfig.PoolingSupport poolingSupport;
            if (poolingSupportType == PoolingSupportType.NONE) {
                poolingSupport = new NoPool();
            } else {
                int minPoolSize = poolingSupportInfo.getMinPoolSize();
                int maxPoolSize = poolingSupportInfo.getMaxPoolSize();
                int blockingTimeoutMilliseconds = poolingSupportInfo.getBlockingTimeoutMilliseconds();
                int idleTimeoutMilliseconds = poolingSupportInfo.getIdleTimeoutMilliseconds();
                int idleTimeoutMinutes = idleTimeoutMilliseconds / (1000 * 60);
                boolean matchOne = poolingSupportInfo.isMatchOne();
                boolean matchAll = poolingSupportInfo.isMatchAll();
                boolean selectOneAssumeMatch = poolingSupportInfo.isSelectOneAssumeMatch();
                if (poolingSupportType == PoolingSupportType.SINGLE) {
                    poolingSupport = new SinglePool(maxPoolSize, minPoolSize, blockingTimeoutMilliseconds,
                            idleTimeoutMinutes, matchOne, matchAll, selectOneAssumeMatch);
                } else {
                    boolean partitionByConnectionRequestInfo = poolingSupportInfo.isPartitionByConnectionRequest();
                    boolean partitionBySubject = poolingSupportInfo.isPartitionBySubject();
                    // TODO partitionBySubject requires a security domain. is this part of wss
                    // feature ??
                    poolingSupport = new PartitionedPool(maxPoolSize, minPoolSize, blockingTimeoutMilliseconds,
                            idleTimeoutMinutes, matchOne, matchAll, selectOneAssumeMatch,
                            partitionByConnectionRequestInfo, partitionBySubject);
                }

            }

            DeploymentClassLoader fl = factory.getFactoryContext().getClassLoader();
            ConnectionManager cm = new GenericConnectionManager(transactionSupport, poolingSupport, false,
                    new ConnectionTrackingCoordinator(), transactionManager, StandaloneContainer.class.getName(),
                    fl);
            ConnectionSupport cs = new ConnectionSupportImpl(fl, cm);
            Connector connector = connectorFactory.createConnector(connectorName, cs, properties, isXA);
            connectors.put(connectorName, connector);

            if (connectorName.equals(DataSourceConnector.SYSTEM_DATA_SOURCE)) {
                if (!(connector instanceof DataSourceConnector)) {
                    throw new StandaloneContainerException("Connector "
                            + connectorName
                            + " is not an instance of "
                            + DataSourceConnector.class.getName()
                            + ".");
                }
            }

            log.info("created connector " + connectorName);
        }

        log.debug("connectors created");
    }

}
