/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.server.standalone10;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.bluestemsoftware.specification.eoa.ComponentDependency;
import org.bluestemsoftware.specification.eoa.DeploymentDependencies;
import org.bluestemsoftware.specification.eoa.FactoryDependency;
import org.bluestemsoftware.specification.eoa.FeatureDependency;
import org.bluestemsoftware.specification.eoa.SystemDependencies;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionException;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryDeployment;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.rt.TransportProtocolRT;
import org.bluestemsoftware.specification.eoa.ext.server.standalone10.StandaloneServer;
import org.bluestemsoftware.specification.eoa.ext.server.standalone10.StandaloneServerException;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.BindingFeature;
import org.bluestemsoftware.specification.eoa.system.server.EnabledFeature;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.bluestemsoftware.specification.eoa.system.server.ModulableFeature;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.bluestemsoftware.specification.eoa.system.server.TransportFeature;
import org.w3c.dom.Element;

/**
 * A 'standalone' <code>Server</code> implementation.
 */
public final class StandaloneServerImpl extends StandaloneServer {

    private Log log = SystemContext.getContext().getSystem().getLog(Server.class);

    public static final String TYPE = "http://bluestemsoftware.org/open/eoa/ext/server/standalone/1.0/";

    private Boolean isStarted = Boolean.FALSE;
    private Map<String, EnabledFeature> enabledFeatures;
    private Map<String, Feature> features = new LinkedHashMap<String, Feature>();
    private Map<String, String> featureTypes = new HashMap<String, String>();
    private Map<String, Feature> incompleteFeatures = new LinkedHashMap<String, Feature>();
    private Map<String, Set<ModulableFeature>> vocabularies = new HashMap<String, Set<ModulableFeature>>();
    private Map<Class<?>, TransportFeature> transportFeatures = new HashMap<Class<?>, TransportFeature>();
    private org.bluestemsoftware.specification.eoa.system.System system;

    static final String LINE_BREAK = System.getProperty("line.separator");

    /**
     * Constructs a <code>StandaloneServer</code>.
     * 
     * @param enabledFeatures
     * @param transports
     *        A map of <code>Transport</code> components, each of which is mapped to a type
     *        ref or an empty map if none defined.
     * 
     */
    public StandaloneServerImpl(ExtensionFactory factory, Map<String, EnabledFeature> enabledFeatures, File extensionEtcDir, File extensionVarDir) {
        super(factory, new StandaloneServer.Provider() {
            public void spi_setConsumer(Extension consumer) {
            }
        }, extensionEtcDir, extensionVarDir);
        if (enabledFeatures == null) {
            throw new IllegalArgumentException("enabledFeatures null");
        } else {
            this.enabledFeatures = enabledFeatures;
        }
        system = SystemContext.getContext().getSystem();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Server#getFeature(java.lang.Class)
     */
    public synchronized <T extends Feature> T getFeature(Class<T> featureType) {
        return featureType.cast(features.get(featureTypes.get(featureType.getName())));
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ManageableExtension#init()
     */
    public void init(Set<ManagementContext> managementContexts) throws ExtensionException {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            initImpl(managementContexts);
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    private void initImpl(Set<ManagementContext> managementContexts) throws ExtensionException {

        synchronized (isStarted) {

            if (isStarted) {
                return;
            }

            log.debug("enabling features");

            for (EnabledFeature enabledFeature : enabledFeatures.values()) {
                String type = enabledFeature.getType();
                String impl = enabledFeature.getFeatureImpl();
                FeatureFactory ff = system.getFeatureFactory(type, impl);
                if (ff == null) {
                    if (impl == null) {
                        throw new StandaloneServerException("Error enabling feature. Feature factory of type '"
                                + type
                                + "' is undefined");
                    } else {
                        throw new StandaloneServerException("Error enabling feature. Feature factory of type '"
                                + type
                                + "' and impl '"
                                + impl
                                + "' is undefined");
                    }
                } else {
                    enableFeature(null, ff, managementContexts);
                }
            }

            // iterate over extension factories defined within system dependencies
            // and verify that required features, if any, were enabled. note that
            // feature factories defined within system dependencies serve only to
            // force loading of a specific version of a specific factory impl, i.e.
            // no need to check these

            SystemDependencies deps = system.getSystemDependencies();
            for (FactoryDependency efd : deps.getFactoryDependencies()) {
                if (efd instanceof FeatureDependency == false) {
                    for (FeatureDependency fd : efd.getFeatureDependencies()) {
                        if (fd.isRequired()) {
                            ExtensionFactory ff = system.getExtensionFactory(fd.getVersionlessRef());
                            if (getFeature(ff.getExtensionType()) == null) {
                                throw new StandaloneServerException("Feature '"
                                        + ff.getExtensionType()
                                        + "' required by factory deployment "
                                        + efd.getVersionlessRef()
                                        + " is undefined. Features must be explicitly enabled within"
                                        + " server definition.");
                            }
                        }
                    }
                }
            }

            for (ComponentDependency cd : deps.getComponentDependencies()) {
                for (FeatureDependency fd : cd.getFeatureDependencies()) {
                    if (fd.isRequired()) {
                        ExtensionFactory ff = system.getExtensionFactory(fd.getVersionlessRef());
                        if (getFeature(ff.getExtensionType()) == null) {
                            throw new StandaloneServerException("Feature '"
                                    + ff.getExtensionType()
                                    + "' required by component deployment "
                                    + cd.getRef()
                                    + " is undefined. Features must be explicitly enabled within"
                                    + " server definition.");
                        }
                    }
                }
            }

            log.debug("features enabled");

            isStarted = Boolean.TRUE;
            return;

        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Server#getFeatures()
     */
    public List<Feature> getFeatures() {
        return new ArrayList<Feature>(features.values());
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Server#getFeature(java.lang.String)
     */
    public Feature getFeature(String featureType) {
        return features.get(featureType);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Server#getTransportFeature(java.lang.Class)
     */
    public TransportFeature getTransportFeature(Class<? extends TransportProtocolRT> transportType) {
        return transportFeatures.get(transportType);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ManageableExtension#destroy()
     */
    public void destroy() {
        synchronized (isStarted) {
            if (!isStarted) {
                return;
            }
            List<Feature> reversedInitOrder = new ArrayList<Feature>(features.values());
            Collections.reverse(reversedInitOrder);
            for (Feature feature : reversedInitOrder) {
                try {
                    log.info("disabling feature " + feature.getExtensionFactory().getName());
                    feature.destroy();
                } catch (Throwable th) {
                    log.error("Error destroying feature. " + th);
                }
            }
            features.clear();
            featureTypes.clear();
            vocabularies.clear();
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Server#updateFeatureConfiguration(org.bluestemsoftware.specification.eoa.system.server.Feature,
     *      org.w3c.dom.Element)
     */
    public void updateFeatureConfiguration(Feature feature, Element configuration) {
        // TODO: implement
        throw new UnsupportedOperationException("Not implemented.");
    }

    private void enableFeature(String dependentRef, FeatureFactory ff, Set<ManagementContext> managementContexts) throws StandaloneServerException {

        log.debug("enableFeature begin");

        EnabledFeature enabledFeature = enabledFeatures.get(ff.getExtensionType());

        if (enabledFeature == null) {
            throw new StandaloneServerException("Feature '"
                    + ff.getExtensionType()
                    + "' required by deployment '"
                    + dependentRef
                    + "' is undefined. Features must be explicitly enabled within server definition.");
        }

        if (log.isDebugEnabled()) {
            log.debug("enabling feature of type '"
                    + ff.getExtensionType()
                    + "' and impl '"
                    + ff.getFeatureImpl()
                    + "'.");
        }

        if (features.get(ff.getExtensionType()) != null) {
            if (log.isDebugEnabled()) {
                Feature existing = features.get(ff.getExtensionType());
                String type = existing.getExtensionType();
                String impl = existing.getFeatureImpl();
                if (ff.getFeatureImpl().equals(existing.getFeatureImpl())) {
                    log.debug("type '" + type + "' and impl '" + impl + "' already enabled.");
                } else {
                    log.debug("ignoring type '"
                            + type
                            + "' and impl '"
                            + impl
                            + "'. impl '"
                            + existing.getFeatureImpl()
                            + "' already enabled.");
                }
            }
            return;
        }

        // check to see if a specific provider was indicated by user

        if (enabledFeature.getFeatureImpl() != null) {
            log.debug("feature provider specified for indicated type");
            if (ff.getFeatureImpl().equals(enabledFeature.getFeatureImpl())) {
                log.debug("which matches current impl");
            } else {
                log.debug("overriding current with impl '" + enabledFeature.getFeatureImpl() + "'");
                String currentType = ff.getExtensionType();
                ff = system.getFeatureFactory(currentType, enabledFeature.getFeatureImpl());
                if (ff == null) {
                    throw new StandaloneServerException("Error enabling feature. Type '"
                            + currentType
                            + "' and impl '"
                            + enabledFeature.getFeatureImpl()
                            + "' is undefined");
                }
            }
        }

        if (!log.isDebugEnabled()) {
            log.info("enabling feature " + ff.getName());
        }

        // create feature and perform some validation against factory
        // attributes

        Feature feature;
        try {
            feature = ff.createFeature();
        } catch (FeatureException fe) {
            throw new StandaloneServerException("Error creating Feature of type '"
                    + ff.getExtensionType()
                    + " and impl '"
                    + ff.getFeatureImpl()
                    + "'. "
                    + fe);
        }

        if (feature == null) {
            throw new StandaloneServerException("Error creating Feature of type '"
                    + ff.getExtensionType()
                    + " and impl '"
                    + ff.getFeatureImpl()
                    + "'. Factory returned null when asked to create feature.");
        }

        if (!feature.getExtensionType().equals(ff.getExtensionType())) {
            throw new StandaloneServerException("Feature indicates type '"
                    + feature.getExtensionType()
                    + "' whereas FeatureFactory from whence it was sourced indicates type '"
                    + ff.getExtensionType()
                    + "'.");
        }

        if (ff.getFeatureImpl() == null) {
            throw new StandaloneServerException("FeatureFactory of type '"
                    + feature.getExtensionType()
                    + "' returns 'null' upon invocation of getFeatureImpl().");
        }

        if (!ff.getFeatureImpl().equals(feature.getFeatureImpl())) {
            throw new StandaloneServerException("Feature indicates impl '"
                    + feature.getFeatureImpl()
                    + "' whereas FeatureFactory from whence it was sourced indicates impl '"
                    + ff.getFeatureImpl()
                    + "'.");
        }

        // before invoking method recursively, set-up preventative measures

        if (incompleteFeatures.containsKey(feature.getExtensionType())) {
            StringBuilder dependencyChain = new StringBuilder();
            for (Feature dependency : incompleteFeatures.values()) {
                dependencyChain.append(dependency.toString());
                dependencyChain.append(System.getProperty("line.separator"));
            }
            throw new StandaloneServerException("Circular dependency attempting to enable feature "
                    + System.getProperty("line.separator")
                    + feature
                    + System.getProperty("line.separator")
                    + "Dependency chain:"
                    + System.getProperty("line.separator")
                    + dependencyChain.toString()
                    + System.getProperty("line.separator"));
        } else {
            incompleteFeatures.put(feature.getExtensionType(), feature);
        }

        // process transitive features before enabling current feature such
        // that they are available for use when current provider is inited

        ExtensionFactoryDeployment deployment = null;
        deployment = (ExtensionFactoryDeployment)ff.getFactoryContext().getDeployment();
        DeploymentDependencies dependencies = deployment.getDependencies();

        for (FeatureDependency ffd : dependencies.getFeatureDependencies()) {

            FeatureFactory temp = (FeatureFactory)system.getExtensionFactory(ffd.getVersionlessRef());

            if (ffd.isRequired()) {

                log.debug("enabling transitive feature " + ffd.getRef() + ".");
                if (temp == null) {
                    throw new StandaloneServerException("Error enabling feature '"
                            + feature.getExtensionType()
                            + "'. transitive feature '"
                            + ffd.getVersionlessRef()
                            + "' is undefined");
                }
                enableFeature(deployment.getRef(), temp, managementContexts);

            } else {

                // if dependency is optional and has been enabled by user, we
                // enable it now so it's available to current feature

                boolean isEnabled = false;
                
                if (temp != null) {
                    for (EnabledFeature ef : enabledFeatures.values()) {
                        if (temp.getExtensionType().equals(ef.getType())) {
                            isEnabled = true;
                            break;
                        }
                    }
                } else {
                    log.debug("not enabling transitive optional feature '"
                            + ffd.getVersionlessRef()
                            + "'. factory is undefined");
                }

                // TODO: we need to add a test case for optional enabled transitive
                // feature scenario

                if (isEnabled) {
                    log.debug("enabling optional, transitive feature " + ffd.getRef() + ".");
                    if (temp == null) {
                        throw new StandaloneServerException("Error enabling feature '"
                                + feature.getExtensionType()
                                + "'. Transitive feature dependency '"
                                + ffd.getVersionlessRef()
                                + "' is undefined");
                    }
                    enableFeature(deployment.getRef(), temp, managementContexts);
                } else {
                    log.debug("optional transitive feature " + ffd.getRef() + " not enabled");
                }

            }
        }

        if (feature instanceof ModulableFeature) {

            Set<String> pvnamespaces = null;
            try {
                pvnamespaces = ((ModulableFeature)feature).getPolicyVocabularyNamespaces();
            } catch (Exception ex) {
                throw new StandaloneServerException("Error enabling modulable feature '"
                        + feature.getExtensionType()
                        + "'. Error retrieving policy vocabularies. "
                        + ex);
            }

            if (pvnamespaces == null) {
                throw new StandaloneServerException("Error enabling modulable feature '"
                        + feature.getExtensionType()
                        + "'. Error retrieving policy vocabularies. "
                        + " Feature returned null.");
            }

            for (String pvn : pvnamespaces) {
                Set<ModulableFeature> mfs = vocabularies.get(pvn);
                if (mfs != null) {
                    if (feature instanceof BindingFeature) {
                        mfs.add((ModulableFeature)feature);
                    } else {
                        throw new StandaloneServerException("Error enabling modulable feature '"
                                + feature.getExtensionType()
                                + "'. Policy vocabulary namespace '"
                                + pvn
                                + "' already mapped to ModulableFeature "
                                + ((Feature)mfs.iterator().next()).getFeatureImpl()
                                + ".");
                    }
                } else {
                    mfs = new HashSet<ModulableFeature>();
                    mfs.add((ModulableFeature)feature);
                }
            }

        }

        try {
            feature.init(managementContexts);
        } catch (Throwable th) {
            StringBuffer sb = new StringBuffer();
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                th.printStackTrace(new PrintStream(baos, true, "UTF-8"));
                sb.append(LINE_BREAK);
                sb.append(new String(baos.toByteArray(), "UTF-8"));
            } catch (Exception exception) {
                sb.append(th.toString());
            }
            throw new StandaloneServerException("Error enabling feature of type '"
                    + ff.getExtensionType()
                    + "' and impl '"
                    + ff.getFeatureImpl()
                    + "'. "
                    + sb.toString());
        }

        features.put(feature.getExtensionType(), feature);
        featureTypes.put(feature.getClass().getName(), feature.getExtensionType());

        if (feature instanceof TransportFeature) {
            Class<? extends TransportProtocolRT> tt = ((TransportFeature)feature).getTransportProtocol();
            if (transportFeatures.get(tt) != null) {
                throw new StandaloneServerException("TransportProtocolRT "
                        + tt.getClass().getName()
                        + " already implemented by TransportFeature "
                        + transportFeatures.get(tt).getClass().getName()
                        + ".");
            } else {
                transportFeatures.put(tt, (TransportFeature)feature);
            }
        }

        incompleteFeatures.remove(feature.getExtensionType());

        if (log.isDebugEnabled()) {
            log.debug("feature " + ff.getName() + " enabled");
        } else {
            log.info("feature " + ff.getName() + " enabled");
        }

        log.debug("enableFeature end");

    }

}
