/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.addressing.soap;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.bluestemsoftware.open.eoa.ext.feature.ws.addressing.soap.util.AddressUtil;
import org.bluestemsoftware.open.eoa.ext.feature.ws.addressing.soap.util.WSAMPolicyUtil;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule.WSSpecification;
import org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.EffectivePolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA.WSA10;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.soap.WSABindingModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.soap.WSAFeature;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;

public final class WSAFeatureImpl implements WSAFeature.Provider {

    public static final String IMPL = WSAFeatureImpl.class.getName();

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(WSAFeature.class);

    public WSAFeatureImpl() {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_setConfiguration(org.w3c.dom.Element)
     */
    public void spi_setConfiguration(Element configuration) {
        // no common configuration defined
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws FeatureException {
        log.debug("init begin");
        log.debug("init end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_destroyFeatureModule(org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule)
     */
    public void spi_destroyFeatureModule(FeatureModule featureModule) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getConnectorRefs()
     */
    @SuppressWarnings("unchecked")
    public Set<String> spi_getConnectorRefs() {
        return Collections.EMPTY_SET;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.BindingFeature$Provider#spi_createBindingModule(org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference,
     *      org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy,
     *      org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy)
     */
    public WSABindingModule.Provider spi_createBindingModule(EndpointActionReference endpointActionReference, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws FeatureException, UnsupportedPolicyException {

        // we only are concerned about policy with policy subject endpoint. any other
        // subject is not supported. public endpoint policy is absolutely required

        if (publicPolicy.getEndpointPolicy() == null) {
            throw new UnsupportedPolicyException(publicPolicy.getMessagePolicy(),
                    "Public policy requires expression with ENDPOINT subject.");
        }

        if (privatePolicy.getMessagePolicy() != null) {
            throw new UnsupportedPolicyException(privatePolicy.getMessagePolicy(),
                    "Policy subject MESSAGE not supported");
        }
        if (privatePolicy.getOperationPolicy() != null) {
            throw new UnsupportedPolicyException(privatePolicy.getOperationPolicy(),
                    "Policy subject OPERATION not supported");
        }
        if (privatePolicy.getServicePolicy() != null) {
            throw new UnsupportedPolicyException(privatePolicy.getServicePolicy(),
                    "Policy subject SERVICE not supported");
        }

        if (publicPolicy.getMessagePolicy() != null) {
            throw new UnsupportedPolicyException(publicPolicy.getMessagePolicy(),
                    "Policy subject MESSAGE not supported");
        }
        if (publicPolicy.getOperationPolicy() != null) {
            throw new UnsupportedPolicyException(publicPolicy.getOperationPolicy(),
                    "Policy subject OPERATION not supported");
        }
        if (publicPolicy.getServicePolicy() != null) {
            throw new UnsupportedPolicyException(publicPolicy.getServicePolicy(),
                    "Policy subject SERVICE not supported");
        }

        EndpointOperationReference eor = (EndpointOperationReference)endpointActionReference.getParent();
        org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference epr = null;
        epr = (org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference)eor.getParent();
        ServiceReference serviceReference = (ServiceReference)epr.getParent();

        // if we make it this far, we know that policy with subject endpoint is
        // defined

        WSABindingModule.Provider wsAddressingModule = null;
        if (serviceReference.isMyService()) {
            wsAddressingModule = processMyEndpoint(epr, endpointActionReference, privatePolicy, publicPolicy);
        } else {
            wsAddressingModule = processPartnerEndpoint(epr, endpointActionReference, privatePolicy, publicPolicy);
        }
        return wsAddressingModule;

    }

    private WSABindingModule.Provider processMyEndpoint(EndpointReference epr, EndpointActionReference ear, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws WSAFeatureException, UnsupportedPolicyException {

        if (privatePolicy.getEndpointPolicy() != null) {
            throw new UnsupportedPolicyException(privatePolicy.getEndpointPolicy(), "Private policy not allowed."
                    + " Private policy serves to clarify public policy defined on a 'partner' endpoint."
                    + " Referenced endpoint is 'my' endpoint.");
        }

        // the goal is to create one instance of ws-addressing spec for each supported
        // version of specification implied by policy. currently, we support only one
        // version, i.e. policy was filtered by system to contain only the version we
        // understand

        WSAMPolicyUtil policyUtil = WSAMPolicyUtil.getInstance(WSA.WSA10.Policy.PVN);
        Boolean anonymousResponses = policyUtil.parseExpression(publicPolicy.getEndpointPolicy());

        // if policy is non-specific, clients determine whether exchange is
        // blocking or non-blocking, i.e. by manipulating replyTo

        if (log.isDebugEnabled()) {
            if (anonymousResponses == null) {
                log.debug("public policy not specific regarding anonymous responses");
            } else {
                log.debug("public policy defines anonymous responses == " + anonymousResponses);
            }
        }

        // configure callback address for self invocation, i.e. if partner which
        // models 'my' application (engine reference where my engine is true) is
        // used to invoke a service defined on self, then address is used to
        // return async response back to self (engine)

        String callbackAddress = null;

        if (anonymousResponses == null || anonymousResponses == Boolean.FALSE) {
            callbackAddress = AddressUtil.generateCallbackAddress(epr);
            log.debug("generated self callback address " + callbackAddress);
        }

        // policy util enforces that policy contains exactly one, non-emtpy alternative,
        // i.e. the returned ws policy set will always contain exactly one entry

        Set<WSSpecification.Policy> alternatives = new HashSet<WSSpecification.Policy>();
        alternatives.add(new WSA10.Policy(anonymousResponses, callbackAddress));
        Map<String, WSSpecification> supportedVersions = new HashMap<String, WSSpecification>();
        supportedVersions.put(WSA10.NAMESPACE_URI, new WSA10(alternatives));

        return new WSABindingModuleImpl((short)0, supportedVersions);

    }

    private WSABindingModule.Provider processPartnerEndpoint(EndpointReference epr, EndpointActionReference ear, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws WSAFeatureException, UnsupportedPolicyException {

        // the goal is to create one instance of ws-addressing spec for each supported
        // version of specification implied by policy. currently, we support only one
        // version, i.e. policy was filtered by system to contain only the version we
        // understand

        WSAMPolicyUtil policyUtil = WSAMPolicyUtil.getInstance(WSA.WSA10.Policy.PVN);
        Boolean anonymousResponses = policyUtil.parseExpression(publicPolicy.getEndpointPolicy());

        String callbackAddress = null;

        if (anonymousResponses == null) {

            // the partner specificies no preference regarding anonymity, binding
            // will determine user preference by checking for existence of a call
            // back address on runtime policy object

            if (privatePolicy.getEndpointPolicy() == null) {

                // if no private policy is defined, default behavior is to force
                // non-anonymous responses. generate default callback address

                callbackAddress = AddressUtil.generateCallbackAddress(epr);

            } else {

                // user is either overriding the default callback address or
                // is forcing anonymous responses. if it's the latter, the
                // policy util returns a null callback address

                callbackAddress = policyUtil.parseReplyToAddress(epr, privatePolicy.getEndpointPolicy());

            }

            log.debug("public policy defined on partner endpoint not specific regarding anon responses");
            if (callbackAddress != null) {
                log.debug("using non-anonymous responses. callback address " + callbackAddress);
            } else {
                log.debug("private endpoint policy defined which forces anonymous responses");
            }

        } else {

            if (anonymousResponses) {

                if (privatePolicy.getEndpointPolicy() != null) {
                    throw new WSAFeatureException("Public policy requires anonymous responses."
                            + " No private policy allowed.");
                }

                log.debug("public policy defined on partner endpoint requires anonymous responses");

            } else {

                log.debug("public policy defined on partner endpoint requires non-anonymous responses");

                EffectivePolicy epp = privatePolicy.getEndpointPolicy();

                // public policy requires non-anonymous responses. if private policy
                // is defined, user is overriding default replyto. if null is
                // returned, user specified an anonymous assertion which is an error

                if (epp == null) {

                    log.debug("generating default callback address.");

                    callbackAddress = AddressUtil.generateCallbackAddress(epr);

                } else {

                    callbackAddress = policyUtil.parseReplyToAddress(epr, epp);

                    if (callbackAddress == null) {
                        throw new WSAFeatureException("Public policy requires non-anonymous responses."
                                + " Private policy cannot assert anonymous responses.");
                    }

                }

                log.debug("using callback address " + callbackAddress);
            }

        }

        // policy util enforces that policy contains exactly one, non-emtpy alternative,
        // i.e. the returned ws policy set will always contain exactly one entry

        Set<WSSpecification.Policy> alternatives = new HashSet<WSSpecification.Policy>();
        alternatives.add(new WSA10.Policy(anonymousResponses, callbackAddress));
        Map<String, WSSpecification> supportedVersions = new HashMap<String, WSSpecification>();
        supportedVersions.put(WSA10.NAMESPACE_URI, new WSA10(alternatives));

        return new WSABindingModuleImpl((short)0, supportedVersions);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAFeature$Provider#spi_createDefaultModules(java.lang.String,
     *      boolean)
     */
    @SuppressWarnings("unchecked")
    public WSABindingModule.Provider spi_createDefaultBindingModule() throws FeatureException {

        // the specification versions supported are moot, i.e. when returing an
        // undeclared fault for my service, version used is version implied by
        // request. when processing an undeclared fault returned by partner
        // service, version used is implied by header infoset on response

        return new WSABindingModuleImpl((short)0, Collections.EMPTY_MAP);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.BindingFeature$Provider#spi_getPolicyVocabularyNamespaces()
     */
    public Set<String> spi_getPolicyVocabularyNamespaces() {
        Set<String> pvns = new HashSet<String>();
        pvns.add(WSA10.Policy.PVN);
        pvns.add(WSA10.Policy.PRIVATE_PVN);
        return pvns;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.BindingFeature$Provider#spi_isNative()
     */
    public boolean spi_isNative() {
        return true;
    }

}
