/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.addressing.soap.util;

import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.policy.rt.EffectivePolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAlternative;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSAFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA.WSA10;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyException;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.Element;

public abstract class WSAMPolicyUtil {

    public static WSAMPolicyUtil getInstance(String policyVocabularyNamespace) {
        if (policyVocabularyNamespace.equals(WSA10.Policy.PVN)) {
            return new WSAM10PolicyUtil();
        }
        return null;
    }

    public abstract String parseReplyToAddress(EndpointReference epr, EffectivePolicy effectivePolicy) throws WSAFeatureException, UnsupportedPolicyException;

    public abstract Boolean parseExpression(EffectivePolicy effectivePolicy) throws WSAFeatureException, UnsupportedPolicyException;

    public static class WSAM10PolicyUtil extends WSAMPolicyUtil {

        public String parseReplyToAddress(EndpointReference epr, EffectivePolicy effectivePolicy) throws WSAFeatureException, UnsupportedPolicyException {

            if (effectivePolicy.getExpression().getAlternatives().size() != 1) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Private policy expression MUST contain exactly one non-empty alternative");
            }

            PolicyAlternative pa = effectivePolicy.getExpression().getAlternatives().iterator().next();

            if (pa.size() != 1) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Alternative defined on private policy expression MUST contain exactly one assertion");
            }

            Element addressingAssertion = (Element)pa.iterator().next();

            PolicyExpression nestedPE = getNestedPolicyExpression(effectivePolicy, addressingAssertion);

            PolicyAlternative nestedPA = null;
            if (nestedPE.getAlternatives().size() != 1) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Private policy invalid. Nested expression contains more than one alternative.");
            } else {
                nestedPA = nestedPE.getAlternatives().iterator().next();
            }

            Element nestedAssertion = null;
            if (nestedPA.size() != 1) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Private policy invalid. Alternative within nested expression MUST contain"
                                + " exactly one assertion.");
            }

            boolean isAnonymousResponses;
            nestedAssertion = nestedPA.iterator().next();
            QName name = new QName(nestedAssertion.getNamespaceURI(), nestedAssertion.getLocalName());
            if (name.equals(WSA10.Policy.Infoset.ANONYMOUS_RESPONSES_ELEM)) {
                isAnonymousResponses = true;
            } else if (name.equals(WSA10.Policy.Infoset.NONANONYMOUS_RESPONSES_ELEM)) {
                isAnonymousResponses = false;
            } else {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Unrecognized assertion within nested policy expression.");
            }
            
            if (isAnonymousResponses) {
                
                // return a null callback address if assertion forces non-specific public
                // policy to require anonymous responses

                return null;
                
            } else {
                
                String attNS = WSA10.Policy.Infoset.REPLY_TO_ATT.getNamespaceURI();
                String attLP = WSA10.Policy.Infoset.REPLY_TO_ATT.getLocalPart();
                String replyTo = nestedAssertion.getAttributeNS(attNS, attLP);

                // if reply to is unspecified, we default value using static method defined
                // on epr class

                if (replyTo == null || replyTo.equals("")) {
                    replyTo = AddressUtil.generateCallbackAddress(epr);
                }

                return replyTo;
                
            }
            
        }

        public Boolean parseExpression(EffectivePolicy effectivePolicy) throws WSAFeatureException, UnsupportedPolicyException {

            // an expression which indicates addressing is supported is acceptable, i.e. one
            // empty alternative and one non-empty alternative. anything else is bad

            if (effectivePolicy.getExpression().getAlternatives().size() > 2) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Expression MUST NOT contain more than one non-empty alternative");
            }

            // so we know expression contains either exactly one non-empty alternative or
            // an empty alternative and one non-empty alternative. process alternatives
            // looking for non-emtpy version which indicates anonymity

            Boolean isAnonymousResponses = null;

            for (PolicyAlternative pa : effectivePolicy.getExpression().getAlternatives()) {
                Element assertion = null;
                if (pa.size() == 0) {
                    continue; // indicates addressing is supported. keep looking
                } else if (pa.size() > 1) {
                    throw new UnsupportedPolicyException(effectivePolicy,
                            "At least one alternative contains more than one assertion.");
                } else {
                    assertion = pa.iterator().next();
                }
                PolicyExpression nestedPE = getNestedPolicyExpression(effectivePolicy, assertion);
                PolicyAlternative nestedPA = null;
                if (nestedPE.getAlternatives().size() != 1) {
                    throw new UnsupportedPolicyException(effectivePolicy,
                            "Nested policy expression contains more than one alternative.");
                } else {
                    nestedPA = nestedPE.getAlternatives().iterator().next();
                }
                Element nestedAssertion = null;
                if (nestedPA.size() > 1) {
                    throw new UnsupportedPolicyException(effectivePolicy,
                            "Alternative within nested expression contains more than one assertion.");
                }
                if (nestedPA.size() == 1) {
                    nestedAssertion = nestedPA.iterator().next();
                    QName name = new QName(nestedAssertion.getNamespaceURI(), nestedAssertion.getLocalName());
                    if (name.equals(WSA10.Policy.Infoset.ANONYMOUS_RESPONSES_ELEM)) {
                        isAnonymousResponses = Boolean.TRUE;
                    } else if (name.equals(WSA10.Policy.Infoset.NONANONYMOUS_RESPONSES_ELEM)) {
                        isAnonymousResponses = Boolean.FALSE;
                    } else {
                        throw new UnsupportedPolicyException(effectivePolicy,
                                "Unrecognized assertion within nested policy expression.");
                    }
                } else {
                    // indicates no assertion regarding anonymity, i.e. we leave value
                    // set to null
                }
            }

            return isAnonymousResponses;

        }

    }

    private static PolicyExpression getNestedPolicyExpression(EffectivePolicy effectivePolicy, Element addressingAssertion) throws WSAFeatureException, UnsupportedPolicyException {

        QName assertionName = new QName(addressingAssertion.getNamespaceURI(), addressingAssertion.getLocalName());
        if (!assertionName.equals(WSA10.Policy.Infoset.ADDRESSING_ELEM)) {
            throw new UnsupportedPolicyException(effectivePolicy, "Unrecognized assertion. Expected "
                    + WSA10.Policy.Infoset.ADDRESSING_ELEM);
        }
        Element nestedExpressionElement = DOMUtils.getChildElement(addressingAssertion, new QName("*", "Policy"));
        if (nestedExpressionElement == null) {
            throw new UnsupportedPolicyException(effectivePolicy, "Expected assertion "
                    + WSA10.Policy.Infoset.ADDRESSING_ELEM
                    + " to contain a nested policy expression.");
        }
        String policyType = effectivePolicy.getExpression().getType();
        ExtensionFactory pf = SystemContext.getContext().getSystem().getPolicyFactory(policyType);
        DOMSource source = new DOMSource(nestedExpressionElement);
        PolicyExpression nestedPE;
        try {
            nestedPE = ((PolicyFactory)pf).readPolicy(null, source).getPolicyExpression();
        } catch (PolicyException ex) {
            throw new WSAFeatureException("Error parsing nested expression. " + ex);
        }

        return nestedPE;

    }

}
