/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.ad.soap.util;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.namespace.QName;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.policy.rt.EffectivePolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAlternative;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.appdata.WSAD.WSAD10;
import org.w3c.dom.Element;

public abstract class WSADPolicyUtil {

    public static WSADPolicyUtil getInstance(String policyVocabularyNamespace) {
        if (policyVocabularyNamespace.equals(WSAD10.Policy.PVN)) {
            return new WSAD10PolicyUtil();
        }
        return null;
    }

    public abstract Set<QName> parseExpression(EffectivePolicy effectivePolicy) throws UnsupportedPolicyException;

    public static class WSAD10PolicyUtil extends WSADPolicyUtil {

        public Set<QName> parseExpression(EffectivePolicy ep) throws UnsupportedPolicyException {

            Set<QName> understoodHeaders = new HashSet<QName>();
            
            if (ep.getExpression().getAlternatives().size() != 1) {
                throw new UnsupportedPolicyException(ep,
                        "Private policy expression MUST contain exactly one non-empty alternative");
            }
            PolicyAlternative pa = ep.getExpression().getAlternatives().iterator().next();
            if (pa.size() != 1) {
                throw new UnsupportedPolicyException(ep,
                        "Alternative defined on private policy expression MUST contain exactly one assertion");
            }
            Element parent = (Element)pa.iterator().next();
            QName qname = new QName(parent.getNamespaceURI(), parent.getLocalName());
            if (!qname.equals(WSAD10.Policy.Infoset.HEADERS_ELEM)) {
                throw new UnsupportedPolicyException(ep, "Unrecognized assertion " + qname);
            }
            List<Element> headers = DOMUtils.getChildElements(parent, new QName("*", "*"));
            for (Element header : headers) {
                qname = new QName(header.getNamespaceURI(), header.getLocalName());
                if (!qname.equals(WSAD10.Policy.Infoset.HEADER_ELEM)) {
                    throw new UnsupportedPolicyException(ep, "Unrecognized child " + qname);
                }
                String prefixedName = DOMUtils.getText(header);
                QName headerName = ComponentName.valueOf(header, prefixedName);
                understoodHeaders.add(headerName);
            }
            
            return understoodHeaders;
            
        }

    }

}
