/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.mtom.soap.util;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.component.policy.rt.EffectivePolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAlternative;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mtom.WSMTOMFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mtom.WSMTOM.WSMTOM10;
import org.w3c.dom.Element;

public abstract class WSOMAPolicyUtil {
    
    public static WSOMAPolicyUtil getInstance(String policyVocabularyNamespace) {
        if (policyVocabularyNamespace.equals(WSMTOM10.Policy.PVN)) {
            return new WSOMA10PolicyUtil();
        }
        return null;
    }

    /**
     * Parses an expression which must be filtered to contain only one policy vocabulary, i.e. vocabulary
     * matching namespace used to retrieve policy util instance
     * 
     * @param effectivePolicy
     * @return
     * @throws WSMTOMFeatureException
     * @throws UnsupportedPolicyException
     */
    public abstract boolean parseExpression(EffectivePolicy effectivePolicy) throws WSMTOMFeatureException, UnsupportedPolicyException;

    public static class WSOMA10PolicyUtil extends WSOMAPolicyUtil {
        
        public boolean parseExpression(EffectivePolicy effectivePolicy) throws WSMTOMFeatureException, UnsupportedPolicyException {

            // an expression which indicates mtom is supported is acceptable, i.e. one
            // empty alternative and one non-empty alternative. anything else is bad
            
            if (effectivePolicy.getExpression().getAlternatives().size() > 2) {
                throw new UnsupportedPolicyException(effectivePolicy,
                        "Expression MUST NOT contain more than one non-empty alternative");
            }
            
            boolean isRequired = true;

            for (PolicyAlternative pa : effectivePolicy.getExpression().getAlternatives()) {
                Element assertion = null;
                if (pa.size() == 0) {
                    isRequired = false;
                } else if (pa.size() > 1) {
                    throw new UnsupportedPolicyException(effectivePolicy,
                            "At least one alternative contains more than one assertion.");
                } else {
                    assertion = pa.iterator().next();
                    QName assertionName = new QName(assertion.getNamespaceURI(), assertion.getLocalName());
                    if (!assertionName.equals(WSMTOM10.Policy.Infoset.OPTIMIZED_ELEM)) {
                        throw new UnsupportedPolicyException(effectivePolicy,
                                "Unrecognized assertion. Expected "
                                + WSMTOM10.Policy.Infoset.OPTIMIZED_ELEM);
                    }
                }                
            }

            return isRequired;

        }

    }
}
