/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.neethi.Policy;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.Constants;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.PolicyFilter;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAlternative;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyAlternative;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyAssertion;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyExpression;
import org.bluestemsoftware.specification.eoa.system.SystemContext;

public class WSPolicyExpressionImpl extends AbstractWSPolicyElement implements WSPolicyExpression {

    private static final long serialVersionUID = 1L;

    private ExtensionFactory factory;
    private Policy underlying;
    private Collection<PolicyAlternative> alternatives;

    public WSPolicyExpressionImpl(WSPolicyDocumentImpl owner) {
        super(owner, NAME.getNamespaceURI(), NAME.getPrefix() + ":" + NAME.getLocalPart());
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#getType()
     */
    public String getType() {
        return WSPolicyDocument.TYPE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyExpression#getExpressionName()
     */
    public String getExpressionName() {
        return getAttributeNS(Constants.NAME_QNAME.getNamespaceURI(), Constants.NAME_QNAME.getLocalPart());
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#getExpressionID()
     */
    public String getExpressionID() {
        return getAttributeNS(Constants.ID_QNAME.getNamespaceURI(), Constants.ID_QNAME.getLocalPart());
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#isSupported()
     */
    public boolean isSupported() {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }
        for (PolicyAlternative policyAlternative : getAlternatives()) {
            if (policyAlternative.isSupported()) {
                return true;
            }
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#getAlternatives()
     */
    public Collection<PolicyAlternative> getAlternatives() {
        
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(getFactory().getFactoryContext().getClassLoader());
            if (alternatives == null) {

                alternatives = new HashSet<PolicyAlternative>();

                Iterator<?> altItr = underlying.getAlternatives();
                while (altItr.hasNext()) {

                    // do not detach the assertion node from the document. the
                    // document was flagged as read only, so we don't have to
                    // worry about user modifying it

                    List<WSPolicyAssertion> temp = new ArrayList<WSPolicyAssertion>();
                    Iterator<?> assItr = ((List<?>)altItr.next()).iterator();
                    while (assItr.hasNext()) {
                        WSPolicyAssertionWrapper w = (WSPolicyAssertionWrapper)assItr.next();
                        temp.add(w.getUnderlyingElement());
                    }

                    alternatives.add(new WSPolicyAlternative(temp));

                }
            }

            return alternatives;

        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#filterExpression(java.util.Set)
     */
    public WSPolicyExpression filterExpression(Set<String> pvns) {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }        
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(getFactory().getFactoryContext().getClassLoader());
            Policy temp = PolicyFilter.filter(underlying, pvns);
            if (temp != null) {
                return (WSPolicyExpression)WSPolicyDocumentWriter.writePolicy(temp).getDocumentElement();
            } else {
                return null;
            }
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#intersectExpression(org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression)
     */
    public WSPolicyExpression intersectExpression(PolicyExpression policyExpression) {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(getFactory().getFactoryContext().getClassLoader());
            Policy temp = underlying.intersect(((WSPolicyExpressionImpl)policyExpression).underlying);
            return (WSPolicyExpression)WSPolicyDocumentWriter.writePolicy(temp).getDocumentElement();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#mergeExpression(org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression)
     */
    public WSPolicyExpression mergeExpression(PolicyExpression policyExpression) {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(getFactory().getFactoryContext().getClassLoader());
            Policy temp = underlying.merge(((WSPolicyExpressionImpl)policyExpression).underlying);
            return (WSPolicyExpression)WSPolicyDocumentWriter.writePolicy(temp).getDocumentElement();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#normalizeExpression()
     */
    public WSPolicyExpression normalizeExpression() {
        
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }

        // set deep to true when invoking normalize on underlying policy, which will force
        // neethi to invoke normalize on our assertion element impl

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();

        try {

            thread.setContextClassLoader(getFactory().getFactoryContext().getClassLoader());

            // this is the only case where targetnamespace of wsdl document within
            // which expression was deployed is still relevant, i.e. merge, intersect
            // and filter involve an operand and result has no associated wsdl doc

            Policy temp = (Policy)underlying.normalize(PolicyRegistryImpl.getInstance(), true);
            WSPolicyDocumentImpl ownerDocument = (WSPolicyDocumentImpl)getOwnerDocument();
            WSPolicyDocument policyDocument = WSPolicyDocumentWriter.writePolicy(temp);
            ((WSPolicyDocumentImpl)policyDocument).setTargetNamespace(ownerDocument.getTargetNamespace());
            return (WSPolicyExpression)policyDocument.getPolicyExpression();

        } finally {
            thread.setContextClassLoader(cl);
        }

    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression#copy()
     */
    public PolicyExpression copy() {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }
        return WSPolicyDocumentWriter.writePolicy(underlying).getPolicyExpression();
    }

    /*
     * (non-Javadoc) Set by expression writer.
     */
    void setUnderlying(Policy underlying) {
        this.underlying = underlying;
    }

    /*
     * (non-Javadoc) Used by policy registry implementation.
     */
    Policy getUnderlying() {
        return underlying;
    }

    private ExtensionFactory getFactory() {
        if (underlying == null) {
            throw new UnsupportedOperationException("Operation not supported on nested expression.");
        }
        if (factory == null) {
            factory = SystemContext.getContext().getSystem().getPolicyFactory(WSPolicyDocument.TYPE);
        }
        return factory;
    }

}
