/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.validation.Validator;

import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public class PolicyDocumentValidator {
    
    private javax.xml.validation.Schema grammar = null;
    
    private static PolicyDocumentValidator singleton = null;
    
    public static PolicyDocumentValidator getInstance() throws WSPolicyException {
        if (singleton == null) {
            singleton = new PolicyDocumentValidator();
        }
        return singleton;
    }
    
    private PolicyDocumentValidator() throws WSPolicyException {

        ExtensionFactory ef = null;
        ef = SystemContext.getContext().getSystem().getPolicyFactory(WSPolicyDocument.TYPE);
        ExtensionFactoryContext factoryContext = ef.getFactoryContext();
        
        // use a dom source as input to schema factory. when a sax source is
        // used, xerces fails to set our entity resolver on sax parser as of
        // version 2.9.0

        Document document = null;
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            builder.setEntityResolver(factoryContext);
            Resource resource = factoryContext.getResource(Constants.WS_POLICY_SCHEMA_LOCATION);
            InputSource inputSource = new InputSource();
            inputSource.setByteStream(resource.getInputStream());
            inputSource.setSystemId(Constants.WS_POLICY_SCHEMA_LOCATION);
            document = builder.parse(inputSource);
        } catch (Exception ex) {
            throw new WSPolicyException("Error creating validator. " + ex);
        }
        
        try {
            javax.xml.validation.SchemaFactory schemaFactory = null;
            schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
            DOMImplementation di = document.getImplementation();
            DOMImplementationLS ls = (DOMImplementationLS)di.getFeature("LS", "3.0");
            schemaFactory.setResourceResolver(new LSResourceResolverImpl(ls, factoryContext));
            DOMSource source = new DOMSource(document);
            source.setSystemId(Constants.WS_POLICY_SCHEMA_LOCATION);
            grammar = schemaFactory.newSchema(source);
        } catch (Exception ex) {
            throw new WSPolicyException("Error creating validator. " + ex);
        }

    }

    public void validate(DOMSource domSource) throws SAXException, IOException {
        
        final ArrayList<String> errorMessages = new ArrayList<String>();

        Validator validator = grammar.newValidator();
        validator.setErrorHandler(new ErrorHandler() {

            public void error(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void warning(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void fatalError(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

        });

        try {
            validator.validate(domSource);
        } catch (Exception ex) {
            throw new SAXException("Unable to validate descriptor. " + ex);
        }

        if (errorMessages.size() > 0) {
            Iterator<String> messageIterator = errorMessages.iterator();
            StringBuilder errorMessage = new StringBuilder();
            while (messageIterator.hasNext()) {
                errorMessage.append(messageIterator.next() + "  ");
            }
            throw new SAXException(errorMessage.toString());
        }
    }

}
