/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces;

import java.util.Iterator;

import javax.xml.namespace.QName;

import org.apache.neethi.All;
import org.apache.neethi.Assertion;
import org.apache.neethi.ExactlyOne;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyComponent;
import org.apache.neethi.PolicyReference;
import org.apache.xerces.dom.DocumentImpl;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.Constants;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class WSPolicyDocumentWriter {

    public static WSPolicyDocumentImpl writePolicy(Policy policy) {

        if (policy == null) {
            throw new IllegalArgumentException("policy null");
        }

        WSPolicyDocumentImpl owner = new WSPolicyDocumentImpl();
        WSPolicyExpressionImpl root = new WSPolicyExpressionImpl(owner);
        owner.appendChild(root);
        root.setUnderlying(policy);

        fixUpNamespace(root);

        QName qname = null;
        if (policy.getName() != null) {
            qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.NAME, Constants.PREFIX);
            setAttribute(root, qname, policy.getName());
        }
        if (policy.getId() != null) {
            qname = new QName(Constants.WSU_NAMESPACE_URI, Constants.ID, Constants.WSU_NAMESPACE_PREFIX);
            setAttribute(root, qname, policy.getId());
        }

        Iterator<?> componentItr = policy.getPolicyComponents().iterator();
        while (componentItr.hasNext()) {
            writePolicyComponent((PolicyComponent)componentItr.next(), root);
        }

        ((DocumentImpl)owner).setReadOnly(true, true);

        return owner;

    }

    private static void writePolicyComponent(PolicyComponent policyComponent, Element parent) {
        if (policyComponent instanceof Assertion) {
            writeAssertion((Assertion)policyComponent, parent);
        } else if (policyComponent instanceof ExactlyOne) {
            writeExactlyOne((ExactlyOne)policyComponent, parent);
        } else if (policyComponent instanceof PolicyReference) {
            writePolicyReference((PolicyReference)policyComponent, parent);
        } else if (policyComponent instanceof Policy) {
            writePolicy((Policy)policyComponent, parent);
        } else if (policyComponent instanceof All) {
            writeAll((All)policyComponent, parent);
        }
    }

    private static void writePolicy(Policy policy, Element parent) {

        QName qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.POLICY, Constants.PREFIX);
        Element child = createElement(parent.getOwnerDocument(), qname);
        if (!parent.getNamespaceURI().equals(Constants.WS_POLICY_NAMESPACE_URI)) {
            fixUpNamespace(child);
        }
        parent.appendChild(child);

        if (policy.getName() != null) {
            qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.NAME, Constants.PREFIX);
            setAttribute(child, qname, policy.getName());
        }
        if (policy.getId() != null) {
            qname = new QName(Constants.WSU_NAMESPACE_URI, Constants.ID, Constants.WSU_NAMESPACE_PREFIX);
            setAttribute(child, qname, policy.getId());
        }

        Iterator<?> componentItr = policy.getPolicyComponents().iterator();
        while (componentItr.hasNext()) {
            writePolicyComponent((PolicyComponent)componentItr.next(), child);
        }

    }

    private static void writeAll(All all, Element parent) {

        QName qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.ALL, Constants.PREFIX);
        Element child = createElement(parent.getOwnerDocument(), qname);
        parent.appendChild(child);

        Iterator<?> componentItr = all.getPolicyComponents().iterator();
        while (componentItr.hasNext()) {
            writePolicyComponent((PolicyComponent)componentItr.next(), child);
        }

    }

    private static void writeExactlyOne(ExactlyOne exactlyOne, Element parent) {

        QName qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.EXACTLY_ONE, Constants.PREFIX);
        Element child = createElement(parent.getOwnerDocument(), qname);
        parent.appendChild(child);

        Iterator<?> componentItr = exactlyOne.getPolicyComponents().iterator();
        while (componentItr.hasNext()) {
            writePolicyComponent((PolicyComponent)componentItr.next(), child);
        }

    }

    private static void writeAssertion(Assertion assertion, Element parent) {

        // clone the assertion node, which is still attached to policy doc
        // associated with the neethi policy object we are using as source
        // and have the document we are building adopt it

        WSPolicyAssertionImpl temp = ((WSPolicyAssertionWrapper)assertion).getUnderlyingElement();
        Element clone = (WSPolicyAssertionImpl)temp.cloneNode(true);
        fixUpNamespace(clone);
        Node node = parent.getOwnerDocument().adoptNode(clone);
        parent.appendChild(node);

    }

    private static void writePolicyReference(PolicyReference policyReference, Element parent) {
        QName qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, Constants.POLICY_REFERENCE, Constants.PREFIX);
        Element child = createElement(parent.getOwnerDocument(), qname);
        parent.appendChild(child);
        qname = new QName(Constants.WS_POLICY_NAMESPACE_URI, "URI", Constants.PREFIX);
        setAttribute(child, qname, policyReference.getURI());
    }

    private static Element createElement(Document owner, QName name) {
        return owner.createElementNS(name.getNamespaceURI(), name.getPrefix() + ":" + name.getLocalPart());
    }

    private static void setAttribute(Element owner, QName name, String value) {
        owner.setAttributeNS(name.getNamespaceURI(), name.getPrefix() + ":" + name.getLocalPart(), value);
    }

    static class PolicyWriterException extends RuntimeException {

        private static final long serialVersionUID = 1L;

        public PolicyWriterException() {
            super();
        }

        public PolicyWriterException(String message, Throwable cause) {
            super(message, cause);
        }

        public PolicyWriterException(String message) {
            super(message);
        }

        public PolicyWriterException(Throwable cause) {
            super(cause);
        }

    }

    /*
     * fix-up the namespace now so that it doesn't get fixed up when document is serialized. as
     * of xerces 2.9.0, the lsserializer modifies the serialized node, i.e. which violates dom
     * 3 spec
     */
    private static void fixUpNamespace(Element element) {

        // TODO: file a bug report w/ xerces after testing against 2.9.1

        if (element.getPrefix() == null || element.getPrefix().equals("")) {
            element.setAttributeNS(Constants.XMLNS_NS, "xmlns", element.getNamespaceURI());
        } else {
            element.setAttributeNS(Constants.XMLNS_NS, "xmlns:" + element.getPrefix(), element.getNamespaceURI());
        }

    }

}
